package ext.grid.column {
import ext.data.Model;
import ext.mixin.IStyleCacher;

[Native("Ext.grid.column.Widget", require)]
/**
 * A widget column is configured with a →<code>widget</code> config object which specifies an
 * →<code>ext.Component.xtype</code> to indicate which type of Widget or Component belongs
 * in the cells of this column.
 * <p>When a widget cell is rendered, a →<code>ext.Widget</code> or →<code>ext.Component</code>
 * of the specified type is rendered into that cell.</p>
 * <p>There are two ways of setting values in a cell widget.</p>
 * <p>The simplest way is to use data binding. To use column widget data binding, the widget must
 * either contain a top-level bind statement, which will cause a
 * →<code>ext.app.ViewModel</code> to be automatically injected into the widget. This ViewModel
 * will inherit data from any ViewModel that the grid is using, and it will also contain
 * two extra properties:</p>
 * <ul>
 * <li><code>record</code>: →<code>ext.data.Model</code><br>The record which backs the grid row.</li>
 * <li><code>recordIndex</code>: →<code>Number</code><br>The index in the dataset of the record which backs
 * the grid row.</li>
 * </ul>
 * <p>For complex widgets, where the widget may be a container that does not directly use any
 * data binding, but has items which do, the specification of a
 * →<code>ext.panel.TablePanel.rowViewModel</code> type or configuration is required on the grid.
 * This can simply be an empty object if grid widgets only require binding to the row record.</p>
 * <p>The deprecated way is to configure the column with a →<code>dataIndex</code>. The widget's
 * →<code>ext.Component.defaultBindProperty</code> will be set using the
 * specified field from the associated record.</p>
 * <p>In the example below we are monitoring the throughput of electricity substations. The capacity
 * being used as a proportion of the maximum rated capacity is displayed as a progress bar.
 * As new data arrives and the instantaneous usage value is updated, the <code>capacityUsed</code>
 * field updates itself, and the record's change is broadcast to all bindings.
 * The <i>Progress Bar Widget</i> (→<code>ext.Progress</code>)'s
 * →<code>ext.Progress.defaultBindProperty</code> (which is
 * "value") is set to the calculated <code>capacityUsed</code>.</p>
 * <p><b>Example:</b></p>
 * <pre>
 * var grid = new Ext.grid.Panel({
 *     title: 'Substation power monitor',
 *     width: 600,
 *     viewConfig: {
 *         enableTextSelection: false,
 *         markDirty: false
 *     },
 *     columns: [{
 *         text: 'Id',
 *         dataIndex: 'id',
 *         width: 120
 *     }, {
 *         text: 'Rating',
 *         dataIndex: 'maxCapacity',
 *         width: 80
 *     }, {
 *         text: 'Avg.',
 *         dataIndex: 'avg',
 *         width: 85,
 *         formatter: 'number("0.00")'
 *     }, {
 *         text: 'Max',
 *         dataIndex: 'max',
 *         width: 80
 *     }, {
 *         text: 'Instant',
 *         dataIndex: 'instant',
 *         width: 80
 *     }, {
 *         text: '%Capacity',
 *         width: 150,
 *
 *         // This is our Widget column
 *         xtype: 'widgetcolumn',
 *
 *         // This is the widget definition for each cell.
 *         // The Progress widget class's defaultBindProperty is 'value'
 *         // so its "value" setting is taken from the ViewModel's record "capacityUsed" field
 *         // Note that a row ViewModel will automatically be injected due to the existence of
 *         // the bind property in the widget configuration.
 *         widget: {
 *             xtype: 'progressbarwidget',
 *             bind: '{record.capacityUsed}',
 *             textTpl: [
 *                 '{percent:number("0")}% capacity'
 *             ]
 *         }
 *     }],
 *     renderTo: document.body,
 *     disableSelection: true,
 *     store: {
 *        fields: [{
 *            name: 'id',
 *            type: 'string'
 *        }, {
 *            name: 'maxCapacity',
 *            type: 'int'
 *        }, {
 *            name: 'avg',
 *            type: 'int',
 *            calculate: function(data) {
 *                // Make this depend upon the instant field being set which sets
 *                // the sampleCount and total.
 *                // Use subscript format to access the other pseudo fields
 *                //  which are set by the instant field's converter
 *                return data.instant &amp;&amp; data['total'] / data['sampleCount'];
 *            }
 *        }, {
 *            name: 'max',
 *            type: 'int',
 *            calculate: function(data) {
 *                // This will be seen to depend on the "instant" field.
 *                // Use subscript format to access this field's current value
 *                //  to avoid circular dependency error.
 *                return (data['max'] || 0) &lt; data.instant ? data.instant : data['max'];
 *            }
 *        }, {
 *            name: 'instant',
 *            type: 'int',
 *
 *            // Upon every update of instantaneous power throughput,
 *            // update the sample count and total so that the max field can calculate itself
 *            convert: function(value, rec) {
 *                rec.data.sampleCount = (rec.data.sampleCount || 0) + 1;
 *                rec.data.total = (rec.data.total || 0) + value;
 *                return value;
 *            },
 *           depends: []
 *        }, {
 *            name: 'capacityUsed',
 *            calculate: function(data) {
 *                return data.instant / data.maxCapacity;
 *            }
 *        }],
 *        data: [{
 *            id: 'Substation A',
 *            maxCapacity: 1000,
 *            avg: 770,
 *            max: 950,
 *            instant: 685
 *        }, {
 *            id: 'Substation B',
 *            maxCapacity: 1000,
 *            avg: 819,
 *            max: 992,
 *            instant: 749
 *        }, {
 *            id: 'Substation C',
 *            maxCapacity: 1000,
 *            avg: 588,
 *              max: 936,
 *            instant: 833
 *        }, {
 *            id: 'Substation D',
 *            maxCapacity: 1000,
 *            avg: 639,
 *            max: 917,
 *            instant: 825
 *        }]
 *    }
 * });
 *
 * // Fake data updating...
 * // Change one record per second to a random power value
 * Ext.interval(function() {
 *     var recIdx = Ext.Number.randomInt(0, 3),
 *         newPowerReading = Ext.Number.randomInt(500, 1000);
 *
 *     grid.store.getAt(recIdx).set('instant', newPowerReading);
 * }, 1000);
 * </pre>
 * @see #widget
 * @see ext.Component#xtype
 * @see ext.Widget
 * @see ext.Component
 * @see ext.app.ViewModel
 * @see ext.data.Model
 * @see Number
 * @see ext.panel.TablePanel#rowViewModel
 * @see #dataIndex
 * @see ext.Component#defaultBindProperty
 * @see ext.Progress
 * @see ext.Progress#defaultBindProperty
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Widget.html Original Ext JS documentation of 'Ext.grid.column.Widget'
 */
public class WidgetColumn extends Column implements IStyleCacher {
  /**
   * @param config @inheritDoc
   */
  public function WidgetColumn(config:WidgetColumn = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * A map of xtype to →<code>ext.Component.ui</code> names to use when using Components
   * in this column.
   * <p>Currently →<code>ext.button.Button</code> and all subclasses of
   * →<code>ext.form.field.TextField</code> default to using <code>ui: "default"</code>
   * when in a WidgetColumn except for in the "classic" theme, when they use ui "grid-cell".</p>
   * @default {}
   * @see ext.Component#ui
   * @see ext.button.Button
   * @see ext.form.field.TextField
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Widget.html#cfg-defaultWidgetUI Original Ext JS documentation of 'defaultWidgetUI'
   * @see #getDefaultWidgetUI()
   * @see #setDefaultWidgetUI()
   */
  public native function get defaultWidgetUI():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set defaultWidgetUI(value:Object):void;

  [ExtConfig]
  /**
   * This flag indicates that this column will be ignored when grid data is exported.
   * <p>When grid data is exported you may want to export only some columns that are important
   * and not everything. Widget, check and action columns are not relevant when data is
   * exported. You can set this flag on any column that you want to be ignored during export.</p>
   * <p>This is used by <i>clipboard plugin</i> (→<code>ext.grid.plugin.GridClipboardPlugin</code>) and
   * <i>exporter plugin</i> (→<code>null</code>).</p>
   * @default true
   * @see ext.grid.plugin.GridClipboardPlugin
   * @see null
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Widget.html#cfg-ignoreExport Original Ext JS documentation of 'ignoreExport'
   */
  override public native function get ignoreExport():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  override public native function set ignoreExport(value:Boolean):void;

  [ExtConfig]
  /**
   * A function that will be called when a widget is attached to a record. This may be useful for
   * doing any post-processing.
   * <pre>
   * Ext.create({
   *     xtype: 'grid',
   *     title: 'Student progress report',
   *     width: 250,
   *     renderTo: Ext.getBody(),
   *     disableSelection: true,
   *     store: {
   *         fields: ['name', 'isHonorStudent'],
   *         data: [{
   *             name: 'Finn',
   *             isHonorStudent: true
   *         }, {
   *             name: 'Jake',
   *             isHonorStudent: false
   *         }]
   *     },
   *     columns: [{
   *         text: 'Name',
   *         dataIndex: 'name',
   *         flex: 1
   *     }, {
   *         xtype: 'widgetcolumn',
   *         text: 'Honor Roll',
   *         dataIndex: 'isHonorStudent',
   *         width: 150,
   *         widget: {
   *             xtype: 'button',
   *             handler: function() {
   *                 // print certificate handler
   *             }
   *         },
   *         // called when the widget is initially instantiated
   *         // on the widget column
   *         onWidgetAttach: function(col, widget, rec) {
   *             widget.setText('Print Certificate');
   *             widget.setDisabled(!rec.get('isHonorStudent'));
   *         }
   *     }]
   * });
   * </pre>
   * @default null
   * <ul>
   * <li><code>column:ext.grid.column.Column</code> —
   * The column.
   * </li>
   * <li><code>widget</code> —
   * The →<code>widget</code> rendered to each cell.
   * </li>
   * <li><code>record:ext.data.Model</code> —
   * The record used with the current widget (cell).
   * </li>
   * </ul>
   * @see #widget
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Widget.html#cfg-onWidgetAttach Original Ext JS documentation of 'onWidgetAttach'
   */
  public native function get onWidgetAttach():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set onWidgetAttach(value:*):void;

  [ExtConfig]
  /**
   * Prevent grid selection upon click on the widget.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Widget.html#cfg-stopSelection Original Ext JS documentation of 'stopSelection'
   */
  public native function get stopSelection():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set stopSelection(value:Boolean):void;

  [ExtConfig]
  /**
   * A config object containing an →<code>ext.Component.xtype</code>.
   * <p>This is used to create the widgets or components which are rendered into the cells
   * of this column.</p>
   * <p>The rendered component has a →<code>ext.app.ViewModel</code> injected which inherits
   * from any ViewModel that the grid is using, and contains two extra properties:</p>
   * <ul>
   * <li><code>record</code>: →<code>ext.data.Model</code><br>The record which backs the grid row.</li>
   * <li><code>recordIndex</code>: →<code>Number</code><br>The index in the dataset of the record which backs
   * the grid row.</li>
   * </ul>
   * <p>The widget configuration may contain a →<code>bind</code> config which uses
   * the ViewModel's data.</p>
   * <p>The derecated way of obtaining data from the record is still supported if the widget
   * does <i>not</i> use a →<code>bind</code> config.</p>
   * <p>This column's →<code>dataIndex</code> is used to update the widget/component's
   * →<code>ext.Component.defaultBindProperty</code>.</p>
   * <p>The widget will be decorated with 2 methods:
   * →<code>getWidgetRecord()</code> - Returns the <i>record</i> (→<code>ext.data.Model</code>) the widget
   * is associated with.
   * →<code>getWidgetColumn()</code> - Returns the <i>column</i> (→<code>ext.grid.column.WidgetColumn</code>)
   * the widget was associated with.</p>
   * @see ext.Component#xtype
   * @see ext.app.ViewModel
   * @see ext.data.Model
   * @see Number
   * @see #bind
   * @see #dataIndex
   * @see ext.Component#defaultBindProperty
   * @see #getWidgetRecord()
   * @see #getWidgetColumn()
   * @see ext.grid.column.WidgetColumn
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Widget.html#cfg-widget Original Ext JS documentation of 'widget'
   */
  public native function get widget():Object;

  [ExtConfig]
  /**
   * @private
   */
  public native function set widget(value:Object):void;

  /**
   * When defined this will take precedence over the
   * →<code>ext.grid.column.Column.renderer</code> config.
   * This is meant to be defined in subclasses that wish to supply their own renderer.
   * <b>Important:</b> Cell must be left blank
   * @see ext.grid.column.Column#renderer
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Widget.html#method-defaultRenderer Original Ext JS documentation of 'defaultRenderer'
   */
  override protected native function defaultRenderer():void;

  /**
   * Returns the value of <code>defaultWidgetUI</code>.
   * @see #defaultWidgetUI
   */
  public native function getDefaultWidgetUI():Object;

  /**
   * getWidgetColumn is a method that decorates every widget.
   * Returns the <i>column</i> (→<code>ext.grid.column.WidgetColumn</code>) the widget was associated with.
   * @see ext.grid.column.WidgetColumn
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Widget.html#method-getWidgetColumn Original Ext JS documentation of 'getWidgetColumn'
   */
  public native function getWidgetColumn():WidgetColumn;

  /**
   * getWidgetRecord is a method that decorates every widget.
   * Returns the <i>record</i> (→<code>ext.data.Model</code>) the widget is associated with.
   * @see ext.data.Model
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.column.Widget.html#method-getWidgetRecord Original Ext JS documentation of 'getWidgetRecord'
   */
  public native function getWidgetRecord():Model;

  /**
   * Sets the value of <code>defaultWidgetUI</code>.
   * @param defaultWidgetUI The new value.
   * @see #defaultWidgetUI
   */
  public native function setDefaultWidgetUI(defaultWidgetUI:Object):void;
}
}