package ext.grid.feature {
import ext.grid.GridPanel;
import ext.view.TableView;

[Mixin("ext.grid.feature.Feature")]
/**
 * A feature is a type of plugin that is specific to the →<code>ext.grid.GridPanel</code>. It provides several
 * hooks that allows the developer to inject additional functionality at certain points throughout
 * the grid creation cycle. This class provides the base template methods that are available
 * to the developer, it should be extended.
 * <p>There are several built in features that extend this class, for example:</p>
 * <ul>
 * <li>→<code>ext.grid.feature.Grouping</code> - Shows grid rows in groups as specified by the
 * →<code>ext.data.Store</code></li>
 * <li>→<code>ext.grid.feature.RowBody</code> - Adds a body section for each grid row that can contain
 * markup.</li>
 * <li>→<code>ext.grid.feature.Summary</code> - Adds a summary row at the bottom of the grid
 * with aggregate totals for a column.</li>
 * </ul>
 * <p><b><i>Using Features</i></b></p>
 * <p>A feature is added to the grid by specifying it an array of features in the configuration:</p>
 * <pre>
 * var groupingFeature = Ext.create('Ext.grid.feature.Grouping');
 * Ext.create('Ext.grid.Panel', {
 *     // other options
 *     features: [groupingFeature]
 * });
 * </pre>
 * <p><b><i>Writing Features</i></b></p>
 * <p>A Feature may add new DOM structure within the structure of a grid.</p>
 * <p>A grid is essentially a <code>&lt;table&gt;</code> element. A →<code>ext.view.TableView</code> instance uses
 * four <i>XTemplates</i> (→<code>ext.XTemplate</code>) to render the grid, <code>tpl</code>, <code>itemTpl</code>, <code>rowTpl</code>, <code>cellTpl</code>.</p>
 * <p>A →<code>ext.view.TableView</code> uses its <code>tpl</code> to emit the item encapsulating HTML tags
 * into its output stream. To render the rows, it invokes →<code>ext.view.TableView.renderRows</code>
 * passing the <code>rows</code> member of its data object and the <code>columns</code> member of its data object.</p>
 * <p>The <code>tpl</code>'s data object Looks like this:
 * {
 * view: owningTableView,
 * rows: recordsToRender,
 * viewStartIndex: indexOfFirstRecordInStore,
 * tableStyle: styleString
 * }</p>
 * <ul>
 * <li>A →<code>ext.view.TableView</code> uses its <code>rowTpl</code> to emit a <code>&lt;tr&gt;</code> HTML tag to its output
 * stream. To render cells, it invokes →<code>ext.view.TableView.renderCell()</code> passing the <code>rows</code> member
 * of its data object.</li>
 * </ul>
 * <p>The <code>itemTpl</code> and <code>rowTpl</code>'s data object looks like this:</p>
 * <pre>
 * {
 *     view:        owningTableView,
 *     record:      recordToRender,
 *     recordIndex: indexOfRecordInStore,
 *     rowIndex:    indexOfRowInView,
 *     columns:     arrayOfColumnDefinitions,
 *     itemClasses: arrayOfClassNames, // For outermost row in case of wrapping
 *     rowClasses:  arrayOfClassNames,  // For internal data bearing row in case of wrapping
 *     rowStyle:    styleString
 * }
 * </pre>
 * <ul>
 * <li>A →<code>ext.view.TableView</code> uses its <code>cellTpl</code> to emit a <code>&lt;td&gt;</code> HTML tag to its output
 * stream.</li>
 * </ul>
 * <p>The <code>cellTpl's</code> data object looks like this:</p>
 * <pre>
 * {
 *     record: recordToRender
 *     column: columnToRender;
 *     recordIndex: indexOfRecordInStore,
 *     rowIndex:    indexOfRowInView,
 *     columnIndex: columnIndex,
 *     align: columnAlign,
 *     tdCls: classForCell
 * }
 * </pre>
 * <p>A Feature may inject its own tpl or rowTpl or cellTpl into the →<code>ext.view.TableView</code>'s
 * rendering by calling →<code>ext.view.TableView.addTpl</code> or →<code>ext.view.TableView.addRowTpl</code> or
 * →<code>ext.view.TableView.addCellTpl</code>.</p>
 * <p>The passed XTemplate is added <i>upstream</i> of the default template for the table element in a link
 * list of XTemplates which contribute to the element's HTML. It may emit appropriate HTML strings
 * into the output stream <i>around</i> a call to</p>
 * <pre>
 * this.nextTpl.apply(values, out, parent);
 * </pre>
 * <p>This passes the current value context, output stream and the parent value context to the next
 * XTemplate in the list.</p>
 * @see ext.grid.GridPanel
 * @see ext.grid.feature.Grouping
 * @see ext.data.Store
 * @see ext.grid.feature.RowBody
 * @see ext.grid.feature.Summary
 * @see ext.view.TableView
 * @see ext.XTemplate
 * @see ext.view.TableView#renderRows
 * @see ext.view.TableView#renderCell()
 * @see ext.view.TableView#addTpl
 * @see ext.view.TableView#addRowTpl
 * @see ext.view.TableView#addCellTpl
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.feature.Feature.html Original Ext JS documentation of 'Ext.grid.feature.Feature'
 * @see ext.grid.feature.Feature
 */
public interface IFeature {
  /**
   * Contains a reference to the element which define the width of a column.
   * <p>This is used by the →<code>null</code> when resizing columns.</p>
   * @see null
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.feature.Feature.html#property-columnSizer Original Ext JS documentation of 'columnSizer'
   */
  function get columnSizer():Object;

  function set columnSizer(value:Object):void;

  /**
   * True when feature is disabled.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.feature.Feature.html#property-disabled Original Ext JS documentation of 'disabled'
   */
  function get disabled():*;

  function set disabled(value:*):void;

  /**
   * Prefix to use when firing events on the view.
   * For example a prefix of group would expose "groupclick", "groupcontextmenu", "groupdblclick".
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.feature.Feature.html#property-eventPrefix Original Ext JS documentation of 'eventPrefix'
   */
  function get eventPrefix():String;

  function set eventPrefix(value:String):void;

  /**
   * Selector used to determine when to fire the event with the eventPrefix.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.feature.Feature.html#property-eventSelector Original Ext JS documentation of 'eventSelector'
   */
  function get eventSelector():String;

  function set eventSelector(value:String):void;

  /**
   * Reference to the grid panel
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.feature.Feature.html#property-grid Original Ext JS documentation of 'grid'
   */
  function get grid():GridPanel;

  function set grid(value:GridPanel):void;

  /**
   * Most features will expose additional events, some may not and will
   * need to change this to false.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.feature.Feature.html#property-hasFeatureEvent Original Ext JS documentation of 'hasFeatureEvent'
   */
  function get hasFeatureEvent():Boolean;

  function set hasFeatureEvent(value:Boolean):void;

  /**
   * <code>true</code> in this class to identify an object as an instantiated Feature, or subclass thereof.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.feature.Feature.html#property-isFeature Original Ext JS documentation of 'isFeature'
   */
  function get isFeature():Boolean;

  function set isFeature(value:Boolean):void;

  /**
   * Reference to the TableView.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.feature.Feature.html#property-view Original Ext JS documentation of 'view'
   */
  function get view():TableView;

  function set view(value:TableView):void;

  /**
   * Disables the feature.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.feature.Feature.html#method-disable Original Ext JS documentation of 'disable'
   */
  function disable():void;

  /**
   * Enables the feature.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.feature.Feature.html#method-enable Original Ext JS documentation of 'enable'
   */
  function enable():void;

  /**
   * Abstract method to be overriden when a feature should add additional
   * arguments to its event signature. By default the event will fire:
   * <ul>
   * <li>view - The underlying Ext.view.Table</li>
   * <li>featureTarget - The matched element by the defined →<code>eventSelector</code></li>
   * </ul>
   * <p>The method must also return the eventName as the first index of the array
   * to be passed to fireEvent.</p>
   * <p class="template-method"><i>This is a template method, a hook into the functionality of this class. Feel free to override it in child classes.</i></p>
   * @see ext.grid.feature.Feature#eventSelector
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.feature.Feature.html#method-getFireEventArgs Original Ext JS documentation of 'getFireEventArgs'
   */
  function getFireEventArgs(eventName:*, view:*, featureTarget:*, e:*):Array;
}
}