package ext.grid.selection {
import ext.grid.column.Column;
import ext.selection.SelectionModel;

[Native("Ext.grid.selection.SpreadsheetModel", require)]
/**
 * @see ext.grid.selection.events.SpreadsheetModelSelection_grid_selectionEvent.SELECTION_CHANGE
 * @eventType ext.grid.selection.events.SpreadsheetModelSelection_grid_selectionEvent.SELECTION_CHANGE
 */
[Event(name='onSelectionChange', type='ext.grid.selection.events.SpreadsheetModelSelection_grid_selectionEvent')]
/**
 * A selection model for <i>grids</i> (→<code>ext.grid.GridPanel</code>) which allows you to select data in
 * a spreadsheet-like manner.
 * <p>Supported features:</p>
 * <ul>
 * <li>Single / Range / Multiple individual row selection.</li>
 * <li>Single / Range cell selection.</li>
 * <li>Column selection by click selecting column headers.</li>
 * <li>Select / deselect all by clicking in the top-left, header.</li>
 * <li>Adds row number column to enable row selection.</li>
 * <li>Optionally you can enable row selection using checkboxes</li>
 * </ul>
 * <p><b>Example usage</b></p>
 * <p><b>Example:</b></p>
 * <pre>
 * var store = Ext.create('Ext.data.Store', {
 *     fields: ['name', 'email', 'phone'],
 *     data: [
 *         { name: 'Lisa', email: 'lisa&#64;simpsons.com', phone: '555-111-1224' },
 *         { name: 'Bart', email: 'bart&#64;simpsons.com', phone: '555-222-1234' },
 *         { name: 'Homer', email: 'homer&#64;simpsons.com', phone: '555-222-1244' },
 *         { name: 'Marge', email: 'marge&#64;simpsons.com', phone: '555-222-1254' }
 *     ]
 * });
 *
 * Ext.create('Ext.grid.Panel', {
 *     title: 'Simpsons',
 *     store: store,
 *     width: 400,
 *     renderTo: Ext.getBody(),
 *     columns: [
 *         { text: 'Name', dataIndex: 'name' },
 *         { text: 'Email', dataIndex: 'email', flex: 1 },
 *         { text: 'Phone', dataIndex: 'phone' }
 *     ],
 *     selModel: {
 *        type: 'spreadsheet'
 *     }
 * });
 * </pre>
 * <p><b>Using →<code>ext.data.BufferedStore</code>s</b></p>
 * <p>It is very important to remember that a →<code>ext.data.BufferedStore</code> does <i>not</i> contain the
 * full dataset. The purpose of a BufferedStore is to only hold in the client, a range of
 * pages from the dataset that corresponds with what is currently visible in the grid
 * (plus a few pages above and below the visible range to allow fast scrolling).</p>
 * <p>When using "select all" rows and a BufferedStore, an <code>allSelected</code> flag is set, and so all
 * records which are read into the client side cache will thenceforth be selected, and will
 * be rendered as selected in the grid.</p>
 * <p><i>But records which have not been read into the cache will obviously not be available
 * when interrogating selected records. As you scroll through the dataset, and more
 * pages are read from the server, they will become available to add to the selection.</i></p>
 * @see ext.grid.GridPanel
 * @see ext.data.BufferedStore
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html Original Ext JS documentation of 'Ext.grid.selection.SpreadsheetModel'
 */
public class SpreadsheetModelSelection extends SelectionModel {
  /**
   * @param config @inheritDoc
   */
  public function SpreadsheetModelSelection(config:SpreadsheetModelSelection = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * Set to <code>true</code> to enable selection of individual cells or a single rectangular
   * range of cells. This will provide cell range selection using click, and
   * potentially drag to select a rectangular range. You can also use "SHIFT + arrow"
   * key navigation to select a range of cells.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-cellSelect Original Ext JS documentation of 'cellSelect'
   * @see #getCellSelect()
   * @see #setCellSelect()
   */
  public native function get cellSelect():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set cellSelect(value:Boolean):void;

  [ExtConfig]
  /**
   * Displays the configured text in the check column's header.
   * <p>if →<code>showHeaderCheckbox</code> is <code>true</code>, the text is shown <i>above</i> the checkbox.</p>
   * @since 6.0.1
   * @see #showHeaderCheckbox
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-checkColumnHeaderText Original Ext JS documentation of 'checkColumnHeaderText'
   */
  public native function get checkColumnHeaderText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set checkColumnHeaderText(value:String):void;

  [ExtConfig]
  /**
   * The index at which to insert the checkbox column.
   * Supported values are a numeric index, and the strings 'first' and 'last'. Only valid when set
   * <i>before</i> render.
   * @default 0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-checkboxColumnIndex Original Ext JS documentation of 'checkboxColumnIndex'
   */
  public native function get checkboxColumnIndex():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set checkboxColumnIndex(value:*):void;

  [ExtConfig]
  /**
   * Width of checkbox column.
   * @default 24
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-checkboxHeaderWidth Original Ext JS documentation of 'checkboxHeaderWidth'
   */
  public native function get checkboxHeaderWidth():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set checkboxHeaderWidth(value:*):void;

  [ExtConfig]
  /**
   * [checkboxSelect=false]
   * Enables selection of the row via clicking on checkbox. Note: this feature will add
   * new column at position specified by →<code>checkboxColumnIndex</code>.
   * @default false
   * @see #checkboxColumnIndex
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-checkboxSelect Original Ext JS documentation of 'checkboxSelect'
   */
  public native function get checkboxSelect():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set checkboxSelect(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * Set to <code>true</code> to enable selection of columns.
   * <p><b>NOTE</b>: This will remove sorting on header click and instead provide column
   * selection and deselection. Sorting is still available via column header menu.</p>
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-columnSelect Original Ext JS documentation of 'columnSelect'
   * @see #getColumnSelect()
   * @see #setColumnSelect()
   */
  public native function get columnSelect():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set columnSelect(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * Set to <code>true</code> to enables cell range selection by cell dragging.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-dragSelect Original Ext JS documentation of 'dragSelect'
   * @see #getDragSelect()
   * @see #setDragSelect()
   */
  public native function get dragSelect():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set dragSelect(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * This configures whether this selection model is to implement a mouse based dragging
   * gesture to extend a <i>contiguous</i> selection.
   * <p>Note that if there are multiple, discontiguous selected rows or columns, selection
   * extension is not available.</p>
   * <p>If set, then the bottom right corner of the contiguous selection will display
   * a drag handle. By dragging this, an extension area may be defined into which
   * the selection is extended.</p>
   * <p>Upon the end of the drag, the
   * →<code>ext.panel.TablePanel.event:onBeforeSelectionExtend</code> event will be fired
   * though the encapsulating grid. Event handlers may manipulate the store data in any way.</p>
   * <p>Possible values for this configuration are</p>
   * <ul>
   * <li><code>"x"</code> Only allow extending the block to the left or right.</li>
   * <li><code>"y"</code> Only allow extending the block above or below.</li>
   * <li><code>"xy"</code> Allow extending the block in both dimensions.</li>
   * <li><code>"both"</code> Allow extending the block in both dimensions.</li>
   * <li><code>true</code> Allow extending the block in both dimensions.</li>
   * <li><code>false</code> Disable the extensible feature</li>
   * <li><code>null</code> Disable the extensible feature</li>
   * </ul>
   * <p>It's important to notice that setting this to <code>"both"</code>, <code>"xy"</code> or <code>true</code> will allow you
   * to extend the selection in both directions, but only one direction at a time.
   * It will NOT be possible to drag it diagonally.</p>
   * @default true
   * @see ext.panel.TablePanel#event:onBeforeSelectionExtend
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-extensible Original Ext JS documentation of 'extensible'
   * @see #getExtensible()
   * @see #setExtensible()
   */
  public native function get extensible():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set extensible(value:String):void;

  [ExtConfig]
  [Bindable]
  /**
   * This configures if the extensible config is also allowed to reduce its selection.
   * <p>Note: This is only relevant if <code>→extensible</code> is not <code>false</code> or <code>null</code></p>
   * @default true
   * @since 6.6.0
   * @see #extensible
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-reducible Original Ext JS documentation of 'reducible'
   * @see #getReducible()
   * @see #setReducible()
   */
  public native function get reducible():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set reducible(value:Boolean):void;

  [ExtConfig]
  /**
   * Width of row numbering column.
   * @default 46
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-rowNumbererHeaderWidth Original Ext JS documentation of 'rowNumbererHeaderWidth'
   */
  public native function get rowNumbererHeaderWidth():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set rowNumbererHeaderWidth(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * Set to <code>true</code> to enable selection of rows by clicking on a row number column.
   * <p><i>Note</i>: This feature will add the row number as the first column.</p>
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-rowSelect Original Ext JS documentation of 'rowSelect'
   * @see #getRowSelect()
   * @see #setRowSelect()
   */
  public native function get rowSelect():Boolean;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set rowSelect(value:Boolean):void;

  [ExtConfig]
  [Bindable]
  /**
   * Pass an instance of one of the subclasses of →<code>ext.grid.selection.Selection</code>.
   * @see ext.grid.selection.Selection
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-selected Original Ext JS documentation of 'selected'
   * @see #getSelected()
   * @see #setSelected()
   */
  public native function get selected():Selection;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set selected(value:Selection):void;

  [ExtConfig]
  /**
   * Configure as <code>false</code> to not display the header checkbox at the top of the checkbox column
   * when →<code>checkboxSelect</code> is set.
   * @default true
   * @see #checkboxSelect
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#cfg-showHeaderCheckbox Original Ext JS documentation of 'showHeaderCheckbox'
   */
  public native function get showHeaderCheckbox():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set showHeaderCheckbox(value:Boolean):void;

  /**
   * Clears the selection.
   * @param suppressEvent Pass <code>true</code> to prevent firing the
   * <code>→event:onSelectionChange</code> event.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-deselectAll Original Ext JS documentation of 'deselectAll'
   * @see #event:onSelectionChange
   */
  override public native function deselectAll(suppressEvent:Boolean = false):void;

  /**
   * Deselects a column.
   * @param column Column to deselect.
   * @param suppressEvent Pass <code>true</code> to prevent firing the
   * <code>→event:onSelectionChange</code> event.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-deselectColumn Original Ext JS documentation of 'deselectColumn'
   * @see #event:onSelectionChange
   */
  public native function deselectColumn(column:Column, suppressEvent:Boolean = false):void;

  /**
   * Returns the value of <code>cellSelect</code>.
   * @see #cellSelect
   */
  public native function getCellSelect():Boolean;

  /**
   * Retrieve a configuration to be used in a HeaderContainer.
   * This should be used when checkboxSelect is set to false.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-getCheckboxHeaderConfig Original Ext JS documentation of 'getCheckboxHeaderConfig'
   */
  protected native function getCheckboxHeaderConfig():Column;

  /**
   * Returns the value of <code>columnSelect</code>.
   * @see #columnSelect
   */
  public native function getColumnSelect():Boolean;

  /**
   * This method is not supported by SpreadsheetModel.
   * <p>To interrogate the selection use →<code>selected</code>'s getter, which will return
   * an instance of one of the three selection types, or <code>null</code> if no selection.</p>
   * <p>The three selection types are:</p>
   * <ul>
   * <li>→<code>ext.grid.selection.RowsSelection</code></li>
   * <li>→<code>ext.grid.selection.ColumnsSelection</code></li>
   * <li>→<code>ext.grid.selection.CellsSelection</code></li>
   * </ul>
   * @see #selected
   * @see ext.grid.selection.RowsSelection
   * @see ext.grid.selection.ColumnsSelection
   * @see ext.grid.selection.CellsSelection
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-getCount Original Ext JS documentation of 'getCount'
   */
  override public native function getCount():Number;

  /**
   * Returns the value of <code>dragSelect</code>.
   * @see #dragSelect
   */
  public native function getDragSelect():Boolean;

  /**
   * Returns the value of <code>extensible</code>.
   * @see #extensible
   */
  public native function getExtensible():String;

  /**
   * Returns the value of <code>reducible</code>.
   * @see #reducible
   */
  public native function getReducible():Boolean;

  /**
   * Returns the value of <code>rowSelect</code>.
   * @see #rowSelect
   */
  public native function getRowSelect():Boolean;

  /**
   * Returns the value of <code>selected</code>.
   * @see #selected
   */
  public native function getSelected():Selection;

  /**
   * This method is not supported by SpreadsheetModel.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-getSelectionMode Original Ext JS documentation of 'getSelectionMode'
   */
  override public native function getSelectionMode():String;

  /**
   * This method is not currently supported by SpreadsheetModel.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-isLocked Original Ext JS documentation of 'isLocked'
   */
  override public native function isLocked():Boolean;

  /**
   * This method is not supported by SpreadsheetModel.
   * <p>To interrogate the selection use →<code>selected</code>'s getter, which will return
   * an instance of one of the three selection types, or <code>null</code> if no selection.</p>
   * <p>The three selection types are:</p>
   * <ul>
   * <li>→<code>ext.grid.selection.RowsSelection</code></li>
   * <li>→<code>ext.grid.selection.ColumnsSelection</code></li>
   * <li>→<code>ext.grid.selection.CellsSelection</code></li>
   * </ul>
   * @param startRow The start of the range to check.
   * @param endRow The end of the range to check.
   * @see #selected
   * @see ext.grid.selection.RowsSelection
   * @see ext.grid.selection.ColumnsSelection
   * @see ext.grid.selection.CellsSelection
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-isRangeSelected Original Ext JS documentation of 'isRangeSelected'
   */
  override public native function isRangeSelected(startRow:*, endRow:*):Boolean;

  /**
   * Select all the data if possible.
   * <p>If →<code>rowSelect</code> is <code>true</code>, then all <i>records</i> will be selected.</p>
   * <p>If →<code>cellSelect</code> is <code>true</code>, then all <i>rendered cells</i> will be selected.</p>
   * <p>If →<code>columnSelect</code> is <code>true</code>, then all <i>columns</i> will be selected.</p>
   * @param suppressEvent True to suppress any select events
   * @see #rowSelect
   * @see #cellSelect
   * @see #columnSelect
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-selectAll Original Ext JS documentation of 'selectAll'
   */
  override public native function selectAll(suppressEvent:Boolean):void;

  /**
   * This method allows programmatic selection of the cell range.
   * <p><b>Example:</b></p>
   * <pre>
   * var store = Ext.create('Ext.data.Store', {
   *     fields  : ['name', 'email', 'phone'],
   *     data    : {
   *         items : [
   *             { name : 'Lisa',  email : 'lisa&#64;simpsons.com',  phone : '555-111-1224' },
   *             { name : 'Bart',  email : 'bart&#64;simpsons.com',  phone : '555-222-1234' },
   *             { name : 'Homer', email : 'homer&#64;simpsons.com', phone : '555-222-1244' },
   *             { name : 'Marge', email : 'marge&#64;simpsons.com', phone : '555-222-1254' }
   *         ]
   *     },
   *     proxy   : {
   *         type   : 'memory',
   *         reader : {
   *             type : 'json',
   *             root : 'items'
   *         }
   *     }
   * });
   *
   * var grid = Ext.create('Ext.grid.Panel', {
   *     title    : 'Simpsons',
   *     store    : store,
   *     width    : 400,
   *     renderTo : Ext.getBody(),
   *     columns  : [
   *        columns: [
   *           { text: 'Name',  dataIndex: 'name' },
   *           { text: 'Email', dataIndex: 'email', flex: 1 },
   *           { text: 'Phone', dataIndex: 'phone', width:120 },
   *           {
   *               text:'Combined', dataIndex: 'name', width : 300,
   *               renderer: function(value, metaData, record, rowIndex, colIndex, store, view) {
   *                   console.log(arguments);
   *                   return value + ' has email: ' + record.get('email');
   *               }
   *           }
   *       ],
   *     ],
   *     selType: 'spreadsheet'
   * });
   *
   * var model = grid.getSelectionModel();  // get selection model
   *
   * // We will create range of 4 cells.
   *
   * // Now set the range  and prevent rangeselect event from being fired.
   * // We can use a simple array when we have no locked columns.
   * model.selectCells([0, 0], [1, 1], true);
   * </pre>
   * @param rangeStart Range starting position. Can be either Cell
   * context or a <code>[rowIndex, columnIndex]</code> numeric array.
   * <p>Note that when a numeric array is used in a locking grid, the column indices are relative
   * to the outermost grid, encompassing locked <i>and</i> normal sides.</p>
   * @param rangeEnd Range end position. Can be either
   * Cell context or a <code>[rowIndex, columnIndex]</code> numeric array.
   * <p>Note that when a numeric array is used in a locking grid, the column indices are relative
   * to the outermost grid, encompassing locked <i>and</i> normal sides.</p>
   * @param suppressEvent Pass <code>true</code> to prevent firing the
   * <code>→event:onSelectionChange</code> event.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-selectCells Original Ext JS documentation of 'selectCells'
   * @see #event:onSelectionChange
   */
  public native function selectCells(rangeStart:*, rangeEnd:*, suppressEvent:Boolean = false):void;

  /**
   * Selects a column.
   * @param column Column to select.
   * @param keepSelection Pass <code>true</code> to keep previous selection.
   * @default false
   * @param suppressEvent Pass <code>true</code> to prevent firing the
   * <code>→event:onSelectionChange</code> event.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-selectColumn Original Ext JS documentation of 'selectColumn'
   * @see #event:onSelectionChange
   */
  public native function selectColumn(column:Column, keepSelection:Boolean = false, suppressEvent:Boolean = false):void;

  /**
   * Select one or more rows.
   * @param rows Records to select.
   * @param keepSelection Pass <code>true</code> to keep previous selection.
   * @default false
   * @param suppressEvent Pass <code>true</code> to prevent firing the
   * <code>→event:onSelectionChange</code> event.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-selectRows Original Ext JS documentation of 'selectRows'
   * @see #event:onSelectionChange
   */
  public native function selectRows(rows:Array, keepSelection:Boolean = false, suppressEvent:Boolean = false):void;

  /**
   * Sets the value of <code>cellSelect</code>.
   * @param cellSelect The new value.
   * @see #cellSelect
   */
  public native function setCellSelect(cellSelect:Boolean):void;

  /**
   * Sets the value of <code>columnSelect</code>.
   * @param columnSelect The new value.
   * @see #columnSelect
   */
  public native function setColumnSelect(columnSelect:Boolean):void;

  /**
   * Sets the value of <code>dragSelect</code>.
   * @param dragSelect The new value.
   * @see #dragSelect
   */
  public native function setDragSelect(dragSelect:Boolean):void;

  /**
   * Sets the value of <code>extensible</code>.
   * @param extensible The new value.
   * @see #extensible
   */
  public native function setExtensible(extensible:String):void;

  /**
   * This method is not currently supported by SpreadsheetModel.
   * @param locked True to lock, false to unlock.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-setLocked Original Ext JS documentation of 'setLocked'
   */
  override public native function setLocked(locked:Boolean):void;

  /**
   * Sets the value of <code>reducible</code>.
   * @param reducible The new value.
   * @see #reducible
   */
  public native function setReducible(reducible:Boolean):void;

  /**
   * Sets the value of <code>rowSelect</code>.
   * @param rowSelect The new value.
   * @see #rowSelect
   */
  public native function setRowSelect(rowSelect:Boolean):void;

  /**
   * Sets the value of <code>selected</code>.
   * @param selected The new value.
   * @see #selected
   */
  public native function setSelected(selected:Selection):void;

  /**
   * This method is not supported by SpreadsheetModel.
   * @param selMode 'SINGLE', 'MULTI' or 'SIMPLE'.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.grid.selection.SpreadsheetModel.html#method-setSelectionMode Original Ext JS documentation of 'setSelectionMode'
   */
  override public native function setSelectionMode(selMode:String):void;
}
}