package ext.layout.container {
import ext.Component;

[Native("Ext.layout.container.Card", require)]
/**
 * This layout manages multiple child Components, each fitted to the Container, where only
 * a single child Component can be visible at any given time. This layout style is most commonly
 * used for wizards, tab implementations, etc. This class is intended to be extended or created
 * via the <code>layout: 'card'</code> →<code>ext.container.Container.layout</code> config, and should generally
 * not need to be created directly via the new keyword.
 * <p>The CardLayout's focal method is →<code>setActiveItem()</code>. Since only one panel is displayed
 * at a time, the only way to move from one Component to the next is by calling setActiveItem,
 * passing the next panel to display (or its id or index). The layout itself does not provide
 * a user interface for handling this navigation, so that functionality must be provided
 * by the developer.</p>
 * <p>To change the active card of a container, call the setActiveItem method of its layout:</p>
 * <p><b>Example:</b></p>
 * <pre>
 * var p = Ext.create('Ext.panel.Panel', {
 *     layout: 'card',
 *     items: [
 *         { html: 'Card 1' },
 *         { html: 'Card 2' }
 *     ],
 *     renderTo: Ext.getBody()
 * });
 *
 * p.getLayout().setActiveItem(1);
 * </pre>
 * <p>The →<code>ext.Component.event:onBeforeDeactivate</code> and
 * →<code>ext.Component.event:onBeforeActivate</code> events can be used to prevent a card
 * from activating or deactivating by returning <code>false</code>.</p>
 * <p><b>Example:</b></p>
 * <pre>
 * var active = 0;
 * var main = Ext.create('Ext.panel.Panel', {
 *     renderTo: Ext.getBody(),
 *     width: 200,
 *     height: 200,
 *     layout: 'card',
 *     tbar: [{
 *         text: 'Next',
 *         handler: function(){
 *             var layout = main.getLayout();
 *             ++active;
 *             layout.setActiveItem(active);
 *             active = main.items.indexOf(layout.getActiveItem());
 *         }
 *     }],
 *     items: [{
 *         title: 'P1'
 *     }, {
 *         title: 'P2'
 *     }, {
 *         title: 'P3',
 *         listeners: {
 *             beforeactivate: function(){
 *                 return false;
 *             }
 *         }
 *     }]
 * });
 * </pre>
 * <p>In the following example, a simplistic wizard setup is demonstrated. A button bar is added
 * to the footer of the containing panel to provide navigation buttons. The buttons will be handled
 * by a common navigation routine. Note that other uses of a CardLayout (like a tab control)
 * would require a completely different implementation. For serious implementations,
 * a better approach would be to extend CardLayout to provide the custom functionality needed.</p>
 * <p><b>Example:</b></p>
 * <pre>
 * var navigate = function(panel, direction){
 *     // This routine could contain business logic required to manage the navigation steps.
 *     // It would call setActiveItem as needed, manage navigation button state, handle any
 *     // branching logic that might be required, handle alternate actions like cancellation
 *     // or finalization, etc.  A complete wizard implementation could get pretty
 *     // sophisticated depending on the complexity required, and should probably be
 *     // done as a subclass of CardLayout in a real-world implementation.
 *     var layout = panel.getLayout();
 *     layout[direction]();
 *     Ext.getCmp('move-prev').setDisabled(!layout.getPrev());
 *     Ext.getCmp('move-next').setDisabled(!layout.getNext());
 * };
 *
 * Ext.create('Ext.panel.Panel', {
 *     title: 'Example Wizard',
 *     width: 300,
 *     height: 200,
 *     layout: 'card',
 *     bodyStyle: 'padding:15px',
 *     defaults: {
 *         // applied to each contained panel
 *         border: false
 *     },
 *     // just an example of one possible navigation scheme, using buttons
 *     bbar: [
 *         {
 *             id: 'move-prev',
 *             text: 'Back',
 *             handler: function(btn) {
 *                 navigate(btn.up("panel"), "prev");
 *             },
 *             disabled: true
 *         },
 *         '-&gt;', // greedy spacer so that the buttons are aligned to each side
 *         {
 *             id: 'move-next',
 *             text: 'Next',
 *             handler: function(btn) {
 *                 navigate(btn.up("panel"), "next");
 *             }
 *         }
 *     ],
 *     // the panels (or "cards") within the layout
 *     items: [{
 *         id: 'card-0',
 *         html: '&lt;h1&gt;Welcome to the Wizard!&lt;/h1&gt;&lt;p&gt;Step 1 of 3&lt;/p&gt;'
 *     },{
 *         id: 'card-1',
 *         html: '&lt;p&gt;Step 2 of 3&lt;/p&gt;'
 *     },{
 *         id: 'card-2',
 *         html: '&lt;h1&gt;Congratulations!&lt;/h1&gt;&lt;p&gt;Step 3 of 3 - Complete&lt;/p&gt;'
 *     }],
 *     renderTo: Ext.getBody()
 * });
 * </pre>
 * @see ext.container.Container#layout
 * @see #setActiveItem()
 * @see ext.Component#event:onBeforeDeactivate
 * @see ext.Component#event:onBeforeActivate
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.layout.container.Card.html Original Ext JS documentation of 'Ext.layout.container.Card'
 */
public class CardLayout extends FitLayout {
  /**
   * @param config @inheritDoc
   */
  public function CardLayout(config:CardLayout = null) {
    super();
  }

  [ExtConfig]
  /**
   * <code>true</code> to render each contained item at the time it becomes active, <code>false</code> to render
   * all contained items as soon as the layout is rendered (defaults to false). If there is
   * a significant amount of content or a lot of heavy controls being rendered into panels
   * that are not displayed by default, setting this to <code>true</code> might improve performance.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.layout.container.Card.html#cfg-deferredRender Original Ext JS documentation of 'deferredRender'
   */
  public native function get deferredRender():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set deferredRender(value:Boolean):void;

  /**
   * Return the active (visible) component in the layout.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.layout.container.Card.html#method-getActiveItem Original Ext JS documentation of 'getActiveItem'
   */
  public native function getActiveItem():Component;

  /**
   * Return the active (visible) component in the layout to the next card
   * @return The next component or false.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.layout.container.Card.html#method-getNext Original Ext JS documentation of 'getNext'
   */
  public native function getNext():Component;

  /**
   * Return the active (visible) component in the layout to the previous card
   * @return The previous component or false.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.layout.container.Card.html#method-getPrev Original Ext JS documentation of 'getPrev'
   */
  public native function getPrev():Component;

  /**
   * Sets the active (visible) component in the layout to the next card
   * @return the activated component or false when nothing activated.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.layout.container.Card.html#method-next Original Ext JS documentation of 'next'
   */
  public native function next():Component;

  /**
   * Sets the active (visible) component in the layout to the previous card
   * @return the activated component or false when nothing activated.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.layout.container.Card.html#method-prev Original Ext JS documentation of 'prev'
   */
  public native function prev():Component;

  /**
   * Makes the given card active.
   * <pre>
   * var card1 = Ext.create('Ext.panel.Panel', {itemId: 'card-1'});
   * var card2 = Ext.create('Ext.panel.Panel', {itemId: 'card-2'});
   * var panel = Ext.create('Ext.panel.Panel', {
   *     layout: 'card',
   *     activeItem: 0,
   *     items: [card1, card2]
   * });
   * // These are all equivalent
   * panel.getLayout().setActiveItem(card2);
   * panel.getLayout().setActiveItem('card-2');
   * panel.getLayout().setActiveItem(1);
   * </pre>
   * @param newCard The component, component
   * →<code>ext.Component.id</code>, →<code>ext.Component.itemId</code>, or index of component.
   * @return the activated component or false when nothing activated.
   * False is returned also when trying to activate an already active card.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.layout.container.Card.html#method-setActiveItem Original Ext JS documentation of 'setActiveItem'
   * @see ext.Component#id
   * @see ext.Component#itemId
   */
  public native function setActiveItem(newCard:*):Component;
}
}