package ext.mixin {
import ext.ClassManager;
import ext.Component;
import ext.ObjectUtil;
import ext.container.Container;
import ext.dom.Element;
import ext.event.Event;
import ext.form.field.BaseField;
import ext.layout.container.BoxLayout;
import ext.util.KeyMap;

import joo.getQualifiedObject;

[Rename("Ext.mixin.AdvancedFocusableContainerMixin")]
public class AdvancedFocusableContainerMixin extends FocusableContainer implements IAdvancedFocusableContainerMixin {

  /**
   * Tab navigation disabled.
   *
   * @see #tabNavigationMode
   */
  public static const TAB_NAVIGATION_MODE_DISABLE:int = -1;
  /**
   * Tab navigation enabled (possibly in addition to arrow key navigation).
   *
   * @see #tabNavigationMode
   */
  public static const TAB_NAVIGATION_MODE_ENABLE:int = 0;
  /**
   * Only tab navigation allowed, arrow key navigation disabled.
   *
   * @see #tabNavigationMode
   */
  public static const TAB_NAVIGATION_MODE_ONLY:int = 1;

  /** @inheritDoc */
  [ExtConfig]
  public native function get excludeInputFields():Boolean;

  /** @private */
  [ExtConfig]
  public native function set excludeInputFields(nested:Boolean):void;

  /** @inheritDoc */
  [ExtConfig]
  public native function get getFocusablesFn():Function;

  /** @private */
  [ExtConfig]
  public native function set getFocusablesFn(fn:Function):void;

  /** @inheritDoc */
  [ExtConfig]
  public native function get ignoreInputFields():Boolean;

  /** @private */
  [ExtConfig]
  public native function set ignoreInputFields(nested:Boolean):void;

  /** @inheritDoc */
  [ExtConfig]
  public native function get nestedFocusableItems():Boolean;

  /** @private */
  [ExtConfig]
  public native function set nestedFocusableItems(nested:Boolean):void;

  /** @inheritDoc */
  [ExtConfig]
  public native function get tabNavigationMode():int;

  /** @private */
  [ExtConfig]
  public native function set tabNavigationMode(mode:int):void;

  [ExtPrivate]
  [ArrayElementType("ext.Component")]
  override internal function getFocusables():Array {
    if (nestedFocusableItems !== false) {
      // Not too nice. But the BoxLayout puts the menuTrigger directly into the DOM without letting the container know anything of it.
      // So we have to make the menuTrigger known to the getFocusables function.
      var mixinClient:ext.container.Container = ext.container.Container(this);
      var focusables:Array = getFocusablesFn ? getFocusablesFn() : mixinClient.query();
      var layout:BoxLayout = mixinClient.layout as BoxLayout;
      if (layout && layout.overflowHandler && layout.overflowHandler.menuTrigger) {
        focusables.push(layout.overflowHandler.menuTrigger);
      }

      return focusables;
    }
    return super.getFocusables();
  }

  [ExtPrivate]
  override internal function processFocusableContainerKeyEvent(e:Event):Event {
    if (ignoreInputFields === false) {
      return e;
    }
    return super.processFocusableContainerKeyEvent(e);
  }

  private function onFocusableContainerTabKeyNavigate(keyCode:Number, event:Event):* {
    event.preventDefault();
    return moveChildFocus(event, !event.shiftKey);
  }

  [ExtPrivate]
  override internal function initFocusableContainerKeyNav(el:Element):void {
    super.initFocusableContainerKeyNav(el);

    if (tabNavigationMode === undefined) {
      return;
    }

    var keyMap:KeyMap = focusableKeyNav['map'];
    keyMap.ignoreInputFields = ignoreInputFields;
    var bindings:Array = keyMap['bindings'];
    function removeBindings(keyCode:String):void {
      for each (var binding:Object in bindings) {
        if (binding.key === keyCode) {
          keyMap.removeBinding(binding);
        }
      }
    }

    removeBindings(Event.TAB);

    if (tabNavigationMode ===TAB_NAVIGATION_MODE_ONLY || tabNavigationMode === TAB_NAVIGATION_MODE_ENABLE) {
      // Tab navigation enabled (possibly in addition to arrow key navigation)
      keyMap.addBinding({
        key: Event.TAB,
        handler: onFocusableContainerTabKeyNavigate,
        ctrl: false,
        shift: false,
        alt: false
      });
      keyMap.addBinding({
        key: Event.TAB,
        handler: onFocusableContainerTabKeyNavigate,
        ctrl: false,
        shift: true,
        alt: false
      });
      if (tabNavigationMode ===TAB_NAVIGATION_MODE_ONLY) {
        // Tab navigation only (no arrow key navigation)
        ([Event.UP, Event.DOWN, Event.LEFT, Event.RIGHT]).forEach(removeBindings);
      }
    } else if (tabNavigationMode !== TAB_NAVIGATION_MODE_DISABLE) {
      throw new Error("Fatal: Unknown value for tabNavigationMode: " + tabNavigationMode);
    }
  }

  [MixinHook(after="constructor")]
  private function afterConstructor():void {
    if (!focusableContainer) {
      return;
    }

    // Listen to add / remove of items at any depth and call onFocusableChildAdd / onFocusableChildRemove (FocusableContainer mixin).
    getFocusables().forEach(function (item:Component):void {
      onFocusableChildAdd(item);
    });
    ext.container.Container(this).on("add", function (ct:ext.container.Container, item:Component):void {
      onFocusableChildAdd(item);

      var containerItem:ext.container.Container = item as ext.container.Container;
      if (containerItem) {
        containerItem.query().forEach(function (child:Component):void {
          onFocusableChildAdd(child);
        });
      }
    });
  }

  [MixinHook(after="onBeforeAdd")]
  private function beforeAdd(component:Component):void {
    if (excludeInputFields) {
      component['__needArrowKeys'] = component['needArrowKeys'];
      component['needArrowKeys'] = false;
    }
  }

  [MixinHook(after="onAdd")]
  private function afterAdd(component:Component):void {
    if (excludeInputFields) {
      component['needArrowKeys'] = component['__needArrowKeys'];
    }
  }

  [ExtPrivate]
  override internal function findNextFocusableChild(options:Object):Component {
    var focusableChild:Component = super.findNextFocusableChild(options);
    if (options.hasOwnProperty("child") && options.hasOwnProperty("step") && excludeInputFields && focusableChild is BaseField) {
      var step:int = options.step === true ? 1 : options.step === false ? -1 : options.step;
      focusableChild = options.child;
      while (true) {
        var previousChild:Component = super.findNextFocusableChild({
          child: focusableChild,
          step: -step // search in opposite direction!
        });
        if (!previousChild || (previousChild is BaseField)) {
          break;
        }
        focusableChild = previousChild;
      }
    }
    return focusableChild;
  }

  [ExtPrivate]
  override internal function deactivateFocusable(child:Component):void {
    if (excludeInputFields && (child is BaseField)) {
      return;
    }
    super.deactivateFocusable(child);
  }

  [ExtPrivate]
  override internal function beforeFocusableChildFocus(child:Component):void {
    super.beforeFocusableChildFocus(child);

    if (child['needArrowKeys']) {
      guardFocusableChild(child);
      if (child is BaseField) {
        ext.container.Container(this).mon(child, "change", listenToInputFieldChanges);
      }
    }
  }

  [ExtPrivate]
  override internal function afterFocusableChildBlur(child:Component):void {
    super.afterFocusableChildBlur(child);

    if (child['needArrowKeys']) {
      if (child is BaseField) {
        ext.container.Container(this).mun(child, "change", listenToInputFieldChanges);
      }
    }
  }

  private function guardFocusableChild(child:Component):void {
    tryGuardFocusableChild(child, false);
    tryGuardFocusableChild(child, true);
  }

  private function tryGuardFocusableChild(child:Component, forward:Boolean):void {
    var guard:Component = findNextFocusableChild({child: child, step: forward});
    if (guard) {
      guard.setTabIndex(activeChildTabIndex);
    }
  }

  private function listenToInputFieldChanges(component:Component):void {
    function handleInputFieldChanges():void {
      guardFocusableChild(component);
    }

    // AsyncObserver here: We need to wait until all other items of the
    // FocusableContainer have updated their status in (possible) reaction
    // to the change event from the input field component.
    // Sorry about the forward reference to a CoreMedia utility class. :-(
    // At least, it is robust against not being available.
    var AsyncObserver:* = getQualifiedObject("com.coremedia.ui.util.AsyncObserver");
    if (AsyncObserver) {
      AsyncObserver.complete(handleInputFieldChanges);
    } else {
      window.setTimeout(handleInputFieldChanges, 10);
    }
  }

  private static function patchOverriddenMixinMethods(...patchedMethods):void {
    var proto:* = AdvancedFocusableContainerMixin["prototype"];
    var mixinId:String = proto.mixinId;

    // find and patch all classes that already have the mixin we extend:
    ObjectUtil.each(ClassManager['classes'], function (name:String, cls:Class):void {
      var clsProto:* = cls && cls.prototype;
      if (hasMixin(clsProto, mixinId)) {
        patchedMethods.forEach(function (patchedMethod:String):void {
          if (clsProto.hasOwnProperty(patchedMethod)) {
            // Overwrite the original method with our mixin method:
            clsProto[patchedMethod] = proto[patchedMethod];
          }
        });
      }
    });
  }

  private static function hasMixin(clsProto:*, mixinId:String):Boolean {
    while (clsProto) {
      if (clsProto.mixins && clsProto.mixins.hasOwnProperty(mixinId)) {
        return true;
      }
      clsProto = clsProto.superclass;
    }
    return false;
  }

  patchOverriddenMixinMethods(
          "getFocusables",
          "processFocusableContainerKeyEvent",
          "initFocusableContainerKeyNav",
          "findNextFocusableChild",
          "deactivateFocusable",
          "beforeFocusableChildFocus",
          "afterFocusableChildBlur");
}
}
