package ext.mixin {
import ext.Component;
import ext.dom.Element;

[Mixin("ext.mixin.Focusable")]
/**
 * This mixin implements focus management functionality in Widgets and Components
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Focusable.html Original Ext JS documentation of 'Ext.mixin.Focusable'
 * @see ext.mixin.Focusable
 */
public interface IFocusable {
  /**
   * <code>true</code> if this currently focused element
   * is within this Component's or Container's hierarchy. This property is set separately
   * from →<code>hasFocus</code>, and can be <code>true</code> when <code>→hasFocus</code> is <code>false</code>.
   * <p>Examples:</p>
   * <ul>
   * <li>
   * <p>Text field with input element focused would be:
   * focusable: true,
   * hasFocus: true,
   * containsFocus: true</p>
   * </li>
   * <li>
   * <p>Date field with drop-down picker currently focused would be:
   * focusable: true,
   * hasFocus: false,
   * containsFocus: true</p>
   * </li>
   * <li>
   * <p>Form Panel with a child input field currently focused would be:
   * focusable: false,
   * hasFocus: false,
   * containsFocus: true</p>
   * </li>
   * </ul>
   * <p>See also →<code>hasFocus</code>.</p>
   * @default false
   * @see #hasFocus
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Focusable.html#property-containsFocus Original Ext JS documentation of 'containsFocus'
   */
  function get containsFocus():Boolean;

  [ExtConfig] /**
   * CSS class that will be added to focused
   * component's →<code>focusClsEl</code>, and removed when component blurs.
   * @default 'x-focused'
   * @see #focusClsEl
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Focusable.html#cfg-focusCls Original Ext JS documentation of 'focusCls'
   */
  function get focusCls():String;

  [ExtConfig] function set focusCls(value:String):void;

  /**
   * The element that will have the
   * →<code>focusCls</code> applied when component's →<code>focusEl</code> is focused.
   * @see #focusCls
   * @see #focusEl
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Focusable.html#property-focusClsEl Original Ext JS documentation of 'focusClsEl'
   */
  function get focusClsEl():Element;

  function set focusClsEl(value:Element):void;

  /**
   * The element that will be focused
   * when →<code>focus()</code> method is called on this component. Usually this is
   * the same element that receives focus via mouse clicks, taps, and pressing
   * Tab key.
   * @default 'el'
   * @see #focus()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Focusable.html#property-focusEl Original Ext JS documentation of 'focusEl'
   */
  function get focusEl():Element;

  function set focusEl(value:Element):void;

  /**
   * <code>true</code> for keyboard interactive Components or Widgets, <code>false</code> otherwise.
   * For Containers, this property reflects interactiveness of the
   * Container itself, not its children. See →<code>isFocusable()</code>.
   * <p><b>Note:</b> It is not enough to set this property to <code>true</code> to make
   * a component keyboard interactive. You also need to make sure that
   * the component's →<code>focusEl</code> is reachable via Tab key (tabbable).
   * See also →<code>tabIndex</code>.</p>
   * @default false
   * @see #isFocusable()
   * @see #focusEl
   * @see #tabIndex
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Focusable.html#property-focusable Original Ext JS documentation of 'focusable'
   */
  function get focusable():Boolean;

  /**
   * <code>true</code> if this component's →<code>focusEl</code> is focused.
   * See also →<code>containsFocus</code>.
   * @default false
   * @see #focusEl
   * @see #containsFocus
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Focusable.html#property-hasFocus Original Ext JS documentation of 'hasFocus'
   */
  function get hasFocus():Boolean;

  /**
   * Try to focus this component.
   * <p>If this component is disabled or otherwise not focusable, a close relation
   * will be targeted for focus instead to keep focus localized for keyboard users.</p>
   * @param selectText If applicable, <code>true</code> to also select all the text in this
   * component, or an array consisting of start and end (defaults to start) position of selection.
   * @param delay Delay the focus this number of milliseconds (true for
   * 10 milliseconds).
   * @param callback Only needed if the <code>delay</code> parameter is used. A function to call
   * upon focus.
   * @param scope Only needed if the <code>delay</code> parameter is used. The scope (<code>this</code>
   * reference) in which to execute the callback.
   * @return The focused Component. Usually <code>this</code> Component. Some Containers may
   * delegate focus to a descendant Component (→<code>ext.window.Window</code>s can do this
   * through their →<code>ext.window.Window.defaultFocus</code> config option. If this
   * component is disabled, a closely related component will be focused and that will be returned.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Focusable.html#method-focus Original Ext JS documentation of 'focus'
   * @see ext.window.Window
   * @see ext.window.Window#defaultFocus
   */
  [Return("this")]
  function focus(selectText:* = undefined, delay:* = undefined, callback:Function = null, scope:Function = null):Component;

  /**
   * Return the actual tabIndex for this Focusable.
   * @return tabIndex attribute value
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Focusable.html#method-getTabIndex Original Ext JS documentation of 'getTabIndex'
   */
  function getTabIndex():Number;

  /**
   * Determine if this Focusable can receive focus at this time.
   * <p>Note that Containers can be non-focusable themselves while delegating
   * focus treatment to a child Component; see <i>defaultFocus</i> (→<code>ext.container.Container</code>)
   * for more information.</p>
   * @param deep Optionally determine if the container itself
   * is focusable, or if container's focus is delegated to a child component
   * and that child is focusable.
   * @default false
   * @return True if component is focusable, false if not.
   * @see ext.container.Container
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Focusable.html#method-isFocusable Original Ext JS documentation of 'isFocusable'
   */
  function isFocusable(deep:Boolean = false):Boolean;

  /**
   * Set the tabIndex property for this Focusable. If the focusEl
   * is available, set tabIndex attribute on it, too.
   * @param newTabIndex new tabIndex to set
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Focusable.html#method-setTabIndex Original Ext JS documentation of 'setTabIndex'
   */
  function setTabIndex(newTabIndex:Number):void;
}
}