package ext.mixin {
import ext.IMixin;
import ext.event.Event;

[Mixin("ext.mixin.Keyboard")]
/**
 * A mixin for components that need to interact with the keyboard. The primary config
 * for this class is the <code>→keyMap</code> config. This config is an object
 * with key names as its properties and with values that describe how the key event
 * should be handled.
 * <p>Key names may key name as documented in <code>→ext.event.Event</code>, numbers (which are treated
 * as <code>keyCode</code> values), single characters (for those that are not defined in
 * <code>→ext.event.Event</code>) or <code>charCode</code> values prefixed by '#' (e.g., "#65" for <code>charCode=65</code>).</p>
 * <p>Entries that use a <code>keyCode</code> will be processed in a <code>keydown</code> event listener, while
 * those that use a <code>charCode</code> will be processed in <code>keypress</code>. This can be overridden
 * if the <code>→keyMap</code> entry specifies an <code>event</code> property.</p>
 * <p>Key names may be preceded by key modifiers. The modifier keys can be specified
 * by prepending the modifier name to the key name separated by <code>+</code> or <code>-</code> (e.g.,
 * "Ctrl+A" or "Ctrl-A"). Only one of these delimiters can be used in a given
 * entry.</p>
 * <p>Valid modifier names are:</p>
 * <ul>
 * <li>Alt</li>
 * <li>Shift</li>
 * <li>Control (or "Ctrl" for short)</li>
 * <li>Command (or "Cmd" or "Meta")</li>
 * <li>CommandOrControl (or "CmdOrCtrl") for Cmd on Mac, Ctrl otherwise.</li>
 * </ul>
 * <p><i>All these names are case insensitive and will be stored in upper case internally.</i></p>
 * <p>For example:</p>
 * <pre>
 *  Ext.define('MyChartPanel', {
 *      extend: 'Ext.panel.Panel',
 *
 *      mixins: [
 *          'Ext.mixin.Keyboard'
 *      ],
 *
 *      controller: 'mycontroller',
 *
 *      // Map keys to methods (typically in a ViewController):
 *      keyMap: {
 *          ENTER: 'onEnterKey',
 *
 *          "ALT+PRINT_SCREEN": 'doScreenshot',
 *
 *          // Cmd on Mac OS X, Ctrl on Windows/Linux.
 *          "CmdOrCtrl+C": 'doCopy',
 *
 *          // This one is handled by a class method.
 *          ESC: {
 *              handler: 'destroy',
 *              scope: 'this',
 *              event: 'keypress'  // default would be keydown
 *          },
 *
 *          "ALT+DOWN": 'openExpander',
 *
 *          // Match any key modifiers and invoke before any other DOWN keys
 *          // handlers with lower or default priority.
 *          "&#42;+DOWN": {
 *              handler: 'preprocessDownKey',
 *              priority: 100
 *          }
 *      }
 *  });
 * </pre>
 * <p>The method names are interpreted in the same way that event listener names are
 * interpreted.</p>
 * @see ext.mixin.Keyboard#keyMap
 * @see ext.event.Event
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Keyboard.html Original Ext JS documentation of 'Ext.mixin.Keyboard'
 * @see ext.mixin.Keyboard
 */
public interface IKeyboard extends IMixin {
  [ExtConfig]
  [Bindable] /**
   * An object containing handlers for keyboard events. The property names of this
   * object are the key name and any modifiers. The values of the properties are the
   * descriptors of how to handle each event.
   * <p>The handler descriptor can be simply the handler function(either the
   * literal function or the method name), or it can be an object with these
   * properties:</p>
   * <ul>
   * <li><code>handler</code>: The function or its name to call to handle the event.</li>
   * <li><code>scope</code>: The this pointer context (can be "this" or "controller").</li>
   * <li><code>event</code>: An optional override of the key event to which to listen.</li>
   * </ul>
   * <p><b>Important:</b> Calls to <code>setKeyMap</code> do not replace the entire <code>→keyMap</code> but
   * instead update the provided mappings. That is, unless <code>null</code> is passed as the
   * value of the <code>→keyMap</code> which will clear the <code>→keyMap</code> of all entries.</p>
   * <ul>
   * <li><code>scope:String</code> (optional) —
   * The default scope to apply to key handlers
   * which do not specify a scope. This is processed the same way as the scope of
   * →<code>listeners</code>. It defaults to the <code>"controller"</code>, but using <code>'this'</code>
   * means that an instance method will be used.
   * </li>
   * </ul>
   * @see #keyMap
   * @see #listeners
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Keyboard.html#cfg-keyMap Original Ext JS documentation of 'keyMap'
   * @see #getKeyMap()
   * @see #setKeyMap()
   */
  function get keyMap():Object;

  [ExtConfig]
  [Bindable] function set keyMap(value:Object):void;

  [ExtConfig]
  [Bindable] /**
   * Enables or disables processing keys in the <code>→keyMap</code>. This value starts as
   * <code>null</code> and if it is <code>null</code> when <code>→initKeyMap()</code> is called, it will automatically
   * be set to <code>true</code>. Since <code>→initKeyMap()</code> is called by <code>→ext.Component</code> at the
   * proper time, this is not something application code normally handles.
   * @default null
   * @see #keyMap
   * @see #initKeyMap()
   * @see ext.Component
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Keyboard.html#cfg-keyMapEnabled Original Ext JS documentation of 'keyMapEnabled'
   * @see #getKeyMapEnabled()
   * @see #setKeyMapEnabled()
   */
  function get keyMapEnabled():Boolean;

  [ExtConfig]
  [Bindable] function set keyMapEnabled(value:Boolean):void;

  /**
   * The last key event processed is cached on the component for use in subsequent
   * event handlers.
   * @since 6.6.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Keyboard.html#property-lastKeyMapEvent Original Ext JS documentation of 'lastKeyMapEvent'
   */
  function get lastKeyMapEvent():Event;

  function set lastKeyMapEvent(value:Event):void;

  /**
   * Returns the value of <code>keyMap</code>.
   * @see #keyMap
   */
  function getKeyMap():Object;

  /**
   * Returns the value of <code>keyMapEnabled</code>.
   * @see #keyMapEnabled
   */
  function getKeyMapEnabled():Boolean;

  /**
   * Sets the value of <code>keyMap</code>.
   * Important: Calls to setKeyMap do not replace the entire keyMap but instead update the provided mappings.
   * That is, unless null is passed as the value of the keyMap which will clear the keyMap of all entries.
   * @param keyMap The new value.
   * @see #keyMap
   */
  function setKeyMap(keyMap:Object):void;

  /**
   * Sets the value of <code>keyMapEnabled</code>.
   * @param keyMapEnabled The new value.
   * @see #keyMapEnabled
   */
  function setKeyMapEnabled(keyMapEnabled:Boolean):void;
}
}