package ext.mixin {
import ext.Component;
import ext.Mixin;

[Native("Ext.mixin.Inheritable", require)]
/**
 * <b>NOTE: This is a private utility class for internal use by the framework. Don't rely on its existence.</b>
 * <p>A mixin that provides the functionality for inheritable configs. This allows linking
 * components and containers via a prototype-chained object for accessing inherited
 * values.</p>
 * <p><b><i>Getting Inherited Properties</i></b></p>
 * <p>A component's inherited state is used to keep track of aspects of a component's state
 * that might be influenced by its ancestors like "collapsed" and "hidden". For example:</p>
 * <pre>
 *  var hidden = this.getInheritedConfig('hidden');
 * </pre>
 * <p>The above will produce <code>true</code> if this or any ancestor component has its <code>hidden</code> config
 * set to <code>true</code>.</p>
 * <p><b><i>Chained Objects</i></b></p>
 * <p>Inheritable properties are implemented by chaining each component's inherited state
 * object to its parent container's inherited state object via the prototype. The result
 * is such that if a component's <code>inheritedState</code> does not have it's own property, it
 * inherits the property from the nearest ancestor that does.</p>
 * <p>In the case of a Container, two state objects are created. The primary ("outer") object
 * is used for reading inherited properties. It is also what a child will prototype chain
 * to if that child is not part of the container's <code>items</code> collection. Anything in the
 * <code>items</code> collection will chain to the inheritedStateInner object instead. This object is
 * prototype chained to inheritedState but allows for Container's layout to set inherited
 * properties that specifically apply only to children of the container. This inner object
 * is unlikely to be needed by user code.</p>
 * <p><b><i>Publishing Inherited Properties</i></b></p>
 * <p>The first step to publishing inherited properties is to override <code>→initInheritedState()</code>
 * and add properties that have inheritable values.</p>
 * <pre>
 *  initInheritedState: function (inheritedState) {
 *      this.callParent(arguments);
 *
 *      if (this.getHidden()) {
 *          inheritedState.hidden = true;
 *      }
 *  }
 * </pre>
 * <p>The above is important because <code>→initInheritedState()</code> is called whenever the object needs
 * to be repopulated. As you can see, only <code>true</code> values are added to <code>inheritedState</code> in
 * this case because <code>false</code> would mask a <code>hidden</code> value of <code>true</code> from an ancestor.</p>
 * <p>If these values change dynamically, these properties must be maintained. For example:</p>
 * <pre>
 *  updateHidden: function (hidden) {
 *      var inherited = this.getInherited();
 *
 *      if (hidden) {
 *          inherited.hidden = true;
 *      } else {
 *          // Unmask whatever may be inherited:
 *          delete inherited.hidden;
 *      }
 *  }
 * </pre>
 * <p><b><i>Proper Usage</i></b></p>
 * <p>ALWAYS access inherited state using <code>→getInherited()</code> or <code>→getInheritedConfig()</code>, not by
 * accessing <code>inheritedState</code> directly.</p>
 * <p>The <code>inheritedState</code> property does not exist until the first call to <code>→getInherited()</code>. At
 * that point <code>→getInherited()</code> walks up the component tree to establish the <code>inheritedState</code>
 * prototype chain. Additionally the <code>inheritedState</code> property should NOT be relied upon
 * even after the initial call to <code>→getInherited()</code> because it is possible for it to become
 * invalid.</p>
 * <p>Invalidation typically happens when a component is moved to a new container. In such a
 * case the <code>inheritedState</code> remains invalid until the next time <code>→getInherited()</code> is called
 * on the component or one of its descendants.</p>
 * @see #initInheritedState()
 * @see #getInherited()
 * @see #getInheritedConfig()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Inheritable.html Original Ext JS documentation of 'Ext.mixin.Inheritable'
 */
public class Inheritable extends Mixin implements IInheritable {
  /**
   * Bubbles up the →<code>getRefOwner()</code> hierarchy, calling the specified function
   * with each component. The scope (<code>this</code> reference) of the function call will be the
   * scope provided or the current component. The arguments to the function will
   * be the args provided or the current component. If the function returns false at any
   * point, the bubble is stopped.
   * @param fn The function to call
   * @param scope The scope of the function. Defaults to current node.
   * @param args The args to call the function with. Defaults to passing the current
   * component.
   * @see #getRefOwner()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Inheritable.html#method-bubble Original Ext JS documentation of 'bubble'
   */
  public native function bubble(fn:Function, scope:Object = null, args:Array = null):void;

  /**
   * This method returns an object containing the inherited properties for this instance.
   * @since 5.0.0
   * @param inner Pass <code>true</code> to return <code>inheritedStateInner</code> instead
   * of the normal <code>inheritedState</code> object. This is only needed internally and should
   * not be passed by user code.
   * @default false
   * @return The <code>inheritedState</code> object containing inherited properties.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Inheritable.html#method-getInherited Original Ext JS documentation of 'getInherited'
   */
  public native function getInherited(inner:Boolean = false):Object;

  /**
   * This method returns the value of a config property that may be inherited from some
   * ancestor.
   * <p>In some cases, a config may be explicitly set on a component with the intent of
   * <i>only</i> being presented to its children while that component should act upon the
   * inherited value (see <code>referenceHolder</code> for example). In these cases the <code>skipThis</code>
   * parameter should be specified as <code>true</code>.</p>
   * @since 5.0.0
   * @param property The name of the config property to return.
   * @param skipThis Pass <code>true</code> if the property should be ignored if
   * found on this instance. In other words, <code>true</code> means the property must be inherited
   * and not explicitly set on this instance.
   * @default false
   * @return The value of the requested <code>property</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Inheritable.html#method-getInheritedConfig Original Ext JS documentation of 'getInheritedConfig'
   */
  public native function getInheritedConfig(property:String, skipThis:Boolean = false):*;

  /**
   * Used by →<code>ext.ComponentQuery</code>, and the →<code>ext.Component.up()</code>
   * method to find the owning Component in the linkage hierarchy.
   * <p>By default this returns the Container which contains this Component.</p>
   * <p>This may be overridden by Component authors who implement ownership hierarchies
   * which are not based upon ownerCt, such as BoundLists being owned by Fields or Menus
   * being owned by Buttons.</p>
   * @see ext.#ComponentQuery ext.ComponentQuery
   * @see ext.Component#up()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Inheritable.html#method-getRefOwner Original Ext JS documentation of 'getRefOwner'
   */
  protected native function getRefOwner():Component;

  /**
   * This method is called to initialize the <code>inheritedState</code> objects for this instance.
   * This amounts to typically copying certain properties from the instance to the given
   * object.
   * @since 5.0.0
   * @param inheritedState The state object for this instance.
   * @param inheritedStateInner This object is only provided for containers.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Inheritable.html#method-initInheritedState Original Ext JS documentation of 'initInheritedState'
   */
  protected native function initInheritedState(inheritedState:Object, inheritedStateInner:Object = null):void;

  /**
   * Determines whether <b>this Component</b> is an ancestor of the passed Component.
   * This will return <code>true</code> if the passed Component is anywhere within the subtree
   * beneath this Component.
   * @param possibleDescendant The Component to test for presence
   * within this Component's subtree.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Inheritable.html#method-isAncestor Original Ext JS documentation of 'isAncestor'
   */
  public native function isAncestor(possibleDescendant:Component):Boolean;

  /**
   * Determines whether this component is the descendant of a passed component.
   * @param ancestor A Component which may contain this Component.
   * @return <code>true</code> if the component is the descendant of the passed component,
   * otherwise <code>false</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Inheritable.html#method-isDescendantOf Original Ext JS documentation of 'isDescendantOf'
   */
  public native function isDescendantOf(ancestor:Component):Boolean;

  /**
   * Gets the Controller or Component that is used as the event root for this view.
   * @since 5.0.0
   * @param defaultScope (Default this) The default scope to return if none is found.
   * @default this
   * @return The default listener scope.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Inheritable.html#method-resolveListenerScope Original Ext JS documentation of 'resolveListenerScope'
   */
  protected native function resolveListenerScope(defaultScope:Object = null):*;

  /**
   * Returns the default listener scope for a "satellite" of this component.
   * Used for resolving scope for observable objects that are not part of the normal
   * Container/Component hierarchy (for example, plugins)
   * @since 5.1.1
   * @return The listener scope
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Inheritable.html#method-resolveSatelliteListenerScope Original Ext JS documentation of 'resolveSatelliteListenerScope'
   */
  protected native function resolveSatelliteListenerScope(satellite:Observable, defaultScope:Object = null):Object;
}
}