package ext.mixin {
import ext.Base;
import ext.Mixin;

[Native("Ext.mixin.Observable", require)]
/**
 * Base class that provides a common interface for publishing events. Subclasses are
 * expected to have a property "events" which is populated as event listeners register,
 * and, optionally, a property "listeners" with configured listeners defined.
 * <p><i>Note</i>: This mixin requires the constructor to be called, which is typically done
 * during the construction of your object. The Observable constructor will call
 * →<code>initConfig()</code>, so it does not need to be called a second time.</p>
 * <p>For example:</p>
 * <pre>
 * Ext.define('Employee', {
 *     mixins: ['Ext.mixin.Observable'],
 *
 *     config: {
 *         name: ''
 *     },
 *
 *     constructor: function (config) {
 *         // The `→<code>listeners</code>` property is processed to add listeners and the config
 *         // is applied to the object.
 *         this.mixins.observable.constructor.call(this, config);
 *         // Config has been initialized
 *         console.log(this.getEmployeeName());
 *     }
 * });
 * </pre>
 * <p>This could then be used like this:</p>
 * <pre>
 * var newEmployee = new Employee({
 *     name: employeeName,
 *     listeners: {
 *         quit: function() {
 *             // By default, "this" will be the object that fired the event.
 *             alert(this.getName() + " has quit!");
 *         }
 *     }
 * });
 * </pre>
 * @see #initConfig()
 * @see #listeners
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html Original Ext JS documentation of 'Ext.mixin.Observable'
 */
public class Observable extends Mixin implements IObservable, IIdentifiable {
  /**
   * @param config @inheritDoc
   */
  public function Observable(config:Observable = null) {
    super();
  }

  public native function get events():Object;

  /**
   * This object holds a key for any event that has a listener. The listener may be set
   * directly on the instance, or on its class or a super class (via →<code>observe()</code>) or
   * on the <i>MVC EventBus</i> (→<code>null</code>). The values of this object are truthy
   * (a non-zero number) and falsy (0 or undefined). They do not represent an exact count
   * of listeners. The value for an event is truthy if the event must be fired and is
   * falsy if there is no need to fire the event.
   * <p>The intended use of this property is to avoid the expense of fireEvent calls when
   * there are no listeners. This can be particularly helpful when one would otherwise
   * have to call fireEvent hundreds or thousands of times. It is used like this:</p>
   * <pre>
   *  if (this.hasListeners.foo) {
   *      this.fireEvent('foo', this, arg1);
   *  }
   * </pre>
   * @see #observe()
   * @see null
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#property-hasListeners Original Ext JS documentation of 'hasListeners'
   */
  public native function get hasListeners():Object;

  /**
   * <code>true</code> in this class to identify an object as an instantiated Observable, or subclass
   * thereof.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#property-isObservable Original Ext JS documentation of 'isObservable'
   */
  public native function get isObservable():Boolean;

  /**
   * @private
   */
  public native function set isObservable(value:Boolean):void;

  [ExtConfig]
  /**
   * A config object containing one or more event handlers to be added to this object during
   * initialization. This should be a valid listeners config object as specified in the
   * →<code>ext.util.Observable.addListener()</code> example for attaching
   * multiple handlers at once.
   * <p><b>DOM events from Ext JS <i>Components</i> (→<code>ext.Component</code>)</b></p>
   * <p>While <i>some</i> Ext JS Component classes export selected DOM events (e.g. "click",
   * "mouseover" etc), this is usually only done when extra value can be added. For example
   * the →<code>ext.view.DataView</code>'s <b><code>→ext.view.DataView.event:onItemClick</code></b>
   * event passing the node clicked on. To access DOM events directly from a child element
   * of a Component, we need to specify the <code>element</code> option to identify the Component
   * property to add a DOM listener to:</p>
   * <pre>
   * new Ext.panel.Panel({
   *     width: 400,
   *     height: 200,
   *     dockedItems: [{
   *         xtype: 'toolbar'
   *     }],
   *     listeners: {
   *         click: {
   *             element: 'el', //bind to the underlying el property on the panel
   *             fn: function(){ console.log('click el'); }
   *         },
   *         dblclick: {
   *             element: 'body', //bind to the underlying body property on the panel
   *             fn: function(){ console.log('dblclick body'); }
   *         }
   *     }
   * });
   * </pre>
   * @see ext.util.Observable#addListener()
   * @see ext.Component
   * @see ext.view.DataView
   * @see ext.view.DataView#event:onItemClick
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#cfg-listeners Original Ext JS documentation of 'listeners'
   * @see #setListeners()
   */
  public native function get listeners():Object;

  [ExtConfig]
  [Bindable(style="methods")]
  /**
   * @private
   */
  public native function set listeners(value:Object):void;

  /**
   * Alias for →<code>onAfter()</code>.
   * @see #onAfter()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-addAfterListener Original Ext JS documentation of 'addAfterListener'
   */
  public native function addAfterListener():void;

  /**
   * Alias for →<code>onBefore()</code>.
   * @see #onBefore()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-addBeforeListener Original Ext JS documentation of 'addBeforeListener'
   */
  public native function addBeforeListener():void;

  /**
   * @inheritDoc
   */
  public native function addEventListener(type:String, listener:Function, options:Object = null):void;

  [Deprecated]
  protected native function addEvents(...events):void;

  /**
   * The →<code>on()</code> method is shorthand for
   * →<code>ext.util.Observable.addListener()</code>.
   * <p>Appends an event handler to this object. For example:</p>
   * <pre>
   * myGridPanel.on("itemclick", this.onItemClick, this);
   * </pre>
   * <p>The method also allows for a single argument to be passed which is a config object
   * containing properties which specify multiple events. For example:</p>
   * <pre>
   * myGridPanel.on({
   *     cellclick: this.onCellClick,
   *     select: this.onSelect,
   *     viewready: this.onViewReady,
   *     scope: this // Important. Ensure "this" is correct during handler execution
   * });
   * </pre>
   * <p>One can also specify options for each event handler separately:</p>
   * <pre>
   * myGridPanel.on({
   *     cellclick: {fn: this.onCellClick, scope: this, single: true},
   *     viewready: {fn: panel.onViewReady, scope: panel}
   * });
   * </pre>
   * <p><i>Names</i> of methods in a specified scope may also be used:</p>
   * <pre>
   * myGridPanel.on({
   *     cellclick: {fn: 'onCellClick', scope: this, single: true},
   *     viewready: {fn: 'onViewReady', scope: panel}
   * });
   * </pre>
   * @param eventName The name of the event to listen for.
   * May also be an object who's property names are event names.
   * @param fn The method the event invokes or the <i>name</i> of
   * the method within the specified <code>scope</code>. Will be called with arguments
   * given to →<code>ext.util.Observable.fireEvent()</code> plus the <code>options</code> parameter described
   * below.
   * @param scope The scope (<code>this</code> reference) in which the handler function is
   * executed. <b>If omitted, defaults to the object which fired the event.</b>
   * @param options An object containing handler configuration.
   * <p><b>Note:</b> The options object will also be passed as the last argument to every
   * event handler.</p>
   * <p>This object may contain any of the following properties:</p>
   * <ul>
   * <li><code>scope:Object</code> (optional) —
   * The scope (<code>this</code> reference) in which the handler function is executed. <b>If omitted,
   * defaults to the object which fired the event.</b>
   * </li>
   * <li><code>delay:Number</code> (optional) —
   * The number of milliseconds to delay the invocation of the handler after the event
   * fires.
   * </li>
   * <li><code>single:Boolean</code> (optional) —
   * True to add a handler to handle just the next firing of the event, and then remove
   * itself.
   * </li>
   * <li><code>buffer:Number</code> (optional) —
   * Causes the handler to be scheduled to run in an →<code>ext.util.DelayedTask</code> delayed
   * by the specified number of milliseconds. If the event fires again within that time,
   * the original handler is <i>not</i> invoked, but the new handler is scheduled in its place.
   * </li>
   * <li><code>onFrame:Number</code> (optional) —
   * Causes the handler to be scheduled to run at the next
   * <i>animation frame event</i> (→<code>ext.FunctionUtil.requestAnimationFrame()</code>). If the
   * event fires again before that time, the handler is not rescheduled - the handler
   * will only be called once when the next animation frame is fired, with the last set
   * of arguments passed.
   * </li>
   * <li><code>target:ext.util.Observable</code> (optional) —
   * Only call the handler if the event was fired on the target Observable, <i>not</i> if the
   * event was bubbled up from a child Observable.
   * </li>
   * <li><code>element:String</code> (optional) —
   * <b>This option is only valid for listeners bound to <i>Components</i> (→<code>ext.Component</code>).</b>
   * The name of a Component property which references an <i>element</i> (→<code>ext.dom.Element</code>)
   * to add a listener to.
   * <p>This option is useful during Component construction to add DOM event listeners to
   * elements of <i>Components</i> (→<code>ext.Component</code>) which will exist only after the
   * Component is rendered.</p>
   * <p>For example, to add a click listener to a Panel's body:</p>
   * <pre>
   *   var panel = new Ext.panel.Panel({
   *       title: 'The title',
   *       listeners: {
   *           click: this.handlePanelClick,
   *           element: 'body'
   *       }
   *   });
   * </pre>
   * <p>In order to remove listeners attached using the element, you'll need to reference
   * the element itself as seen below.</p>
   * <pre>
   *  panel.body.un(...)
   * </pre>
   * </li>
   * <li><code>delegate:String</code> (optional) —
   * A simple selector to filter the event target or look for a descendant of the target.
   * <p>The "delegate" option is only available on Ext.dom.Element instances (or
   * when attaching a listener to a Ext.dom.Element via a Component using the
   * element option).</p>
   * <p>See the <i>delegate</i> example below.</p>
   * </li>
   * <li><code>capture:Boolean</code> (optional) —
   * When set to <code>true</code>, the listener is fired in the capture phase of the event propagation
   * sequence, instead of the default bubble phase.
   * <p>The <code>→capture()</code> option is only available on Ext.dom.Element instances (or
   * when attaching a listener to a Ext.dom.Element via a Component using the
   * element option).</p>
   * </li>
   * <li><code>stopPropagation:Boolean</code> (optional) —
   * <b>This option is only valid for listeners bound to <i>Elements</i> (→<code>ext.dom.Element</code>).</b>
   * <code>true</code> to call →<code>ext.event.Event.stopPropagation()</code> on the event
   * object before firing the handler.
   * </li>
   * <li><code>preventDefault:Boolean</code> (optional) —
   * <b>This option is only valid for listeners bound to <i>Elements</i> (→<code>ext.dom.Element</code>).</b>
   * <code>true</code> to call →<code>ext.event.Event.preventDefault()</code> on the event
   * object before firing the handler.
   * </li>
   * <li><code>stopEvent:Boolean</code> (optional) —
   * <b>This option is only valid for listeners bound to <i>Elements</i> (→<code>ext.dom.Element</code>).</b>
   * <code>true</code> to call →<code>ext.event.Event.stopEvent()</code> on the event object
   * before firing the handler.
   * </li>
   * <li><code>args:Array</code> (optional) —
   * Optional set of arguments to pass to the handler function before the actual
   * fired event arguments. For example, if <code>args</code> is set to <code>['foo', 42]</code>,
   * the event handler function will be called with an arguments list like this:
   * <pre>
   *  handler('foo', 42, &lt;actual event arguments&gt;...);
   * </pre>
   * </li>
   * <li><code>destroyable:Boolean</code> (default = <code>false</code>) —
   * When specified as <code>true</code>, the function returns a <code>destroyable</code> object. An object
   * which implements the <code>→destroy()</code> method which removes all listeners added in this call.
   * This syntax can be a helpful shortcut to using →<code>un()</code>; particularly when
   * removing multiple listeners. <i>NOTE</i> - not compatible when using the <i>element</i>
   * option. See →<code>un()</code> for the proper syntax for removing listeners added using the
   * <i>element</i> config.
   * </li>
   * <li><code>priority:Number</code> (optional) —
   * An optional numeric priority that determines the order in which event handlers
   * are run. Event handlers with no priority will be run as if they had a priority
   * of 0. Handlers with a higher priority will be prioritized to run sooner than
   * those with a lower priority. Negative numbers can be used to set a priority
   * lower than the default. Internally, the framework uses a range of 1000 or
   * greater, and -1000 or lesser for handlers that are intended to run before or
   * after all others, so it is recommended to stay within the range of -999 to 999
   * when setting the priority of event handlers in application-level code.
   * A priority must be an integer to be valid. Fractional values are reserved for
   * internal framework use.
   * </li>
   * <li><code>order:String</code> (default = <code>'current'</code>) —
   * A legacy option that is provided for backward compatibility.
   * It is recommended to use the <code>priority</code> option instead. Available options are:
   * <ul>
   * <li><code>'before'</code>: equal to a priority of <code>100</code></li>
   * <li><code>'current'</code>: equal to a priority of <code>0</code> or default priority</li>
   * <li><code>'after'</code>: equal to a priority of <code>-100</code></li>
   * </ul>
   * </li>
   * </ul>
   * @param order A shortcut for the <code>order</code> event option. Provided for backward compatibility.
   * Please use the <code>priority</code> event option instead.
   * @default 'current'
   * @return <b>Only when the <code>destroyable</code> option is specified.</b>
   * <p>A <code>Destroyable</code> object. An object which implements the <code>→destroy()</code> method which removes
   * all listeners added in this call. For example:</p>
   * <pre>
   * this.btnListeners =  = myButton.on({
   *     destroyable: true
   *     mouseover:   function() { console.log('mouseover'); },
   *     mouseout:    function() { console.log('mouseout'); },
   *     click:       function() { console.log('click'); }
   * });
   * </pre>
   * <p>And when those listeners need to be removed:</p>
   * <pre>
   * Ext.destroy(this.btnListeners);
   * </pre>
   * <p>or</p>
   * <pre>
   * this.btnListeners.destroy();
   * </pre>
   * @see #on()
   * @see ext.util.Observable#addListener()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-addListener Original Ext JS documentation of 'addListener'
   * @see ext.util.Observable#fireEvent()
   * @see ext.util.DelayedTask
   * @see ext.SFunctionUtil#requestAnimationFrame() ext.FunctionUtil.requestAnimationFrame()
   * @see ext.Component
   * @see ext.dom.Element
   * @see ext.mixin.Observable#capture()
   * @see ext.event.Event#stopPropagation()
   * @see ext.event.Event#preventDefault()
   * @see ext.event.Event#stopEvent()
   * @see ext.mixin.Observable#destroy()
   * @see ext.mixin.Observable#un()
   * @see #destroy()
   */
  [Return("this")]
  public native function addListener(eventName:String, fn:Function, scope:Object = null, options:Object = null, order:String = 'current'):Base;

  /**
   * The addManagedListener method is used when some object (call it "A") is listening
   * to an event on another observable object ("B") and you want to remove that listener
   * from "B" when "A" is destroyed. This is not an issue when "B" is destroyed because
   * all of its listeners will be removed at that time.
   * <p>Example:</p>
   * <pre>
   * Ext.define('Foo', {
   *     extend: 'Ext.Component',
   *
   *     initComponent: function () {
   *         this.addManagedListener(MyApp.SomeSharedMenu, 'show', this.doSomething);
   *         this.callParent();
   *     }
   * });
   * </pre>
   * <p>As you can see, when an instance of Foo is destroyed, it ensures that the 'show'
   * listener on the menu (<code>MyApp.SomeGlobalSharedMenu</code>) is also removed.</p>
   * <p>As of version 5.1 it is no longer necessary to use this method in most cases because
   * listeners are automatically managed if the scope object provided to
   * →<code>ext.util.Observable.addListener()</code> is an Observable instance.
   * However, if the observable instance and scope are not the same object you
   * still need to use <code>→mon()</code> or <code>→addManagedListener()</code> if you want the listener to be
   * managed.</p>
   * @param item The item to which to add
   * a listener/listeners.
   * @param ename The event name, or an object containing event name
   * properties.
   * @param fn If the <code>ename</code> parameter was an event
   * name, this is the handler function or the name of a method on the specified
   * <code>scope</code>.
   * @param scope If the <code>ename</code> parameter was an event name, this is
   * the scope (<code>this</code> reference) in which the handler function is executed.
   * @param options If the <code>ename</code> parameter was an event name, this is
   * the →<code>ext.util.Observable.addListener()</code> options.
   * @return <b>Only when the <code>destroyable</code> option is specified.</b>
   * <p>A <code>Destroyable</code> object. An object which implements the <code>→destroy()</code> method which removes
   * all listeners added in this call. For example:</p>
   * <pre>
   * this.btnListeners = myButton.mon({
   *     destroyable: true
   *     mouseover:   function() { console.log('mouseover'); },
   *     mouseout:    function() { console.log('mouseout'); },
   *     click:       function() { console.log('click'); }
   * });
   * </pre>
   * <p>And when those listeners need to be removed:</p>
   * <pre>
   * Ext.destroy(this.btnListeners);
   * </pre>
   * <p>or</p>
   * <pre>
   * this.btnListeners.destroy();
   * </pre>
   * @see ext.util.Observable#addListener()
   * @see #mon()
   * @see #addManagedListener()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-addManagedListener Original Ext JS documentation of 'addManagedListener'
   * @see #destroy()
   */
  public native function addManagedListener(item:*, ename:*, fn:* = undefined, scope:Object = null, options:Object = null):Object;

  /**
   * Removes all listeners for this object including the managed listeners
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-clearListeners Original Ext JS documentation of 'clearListeners'
   */
  public native function clearListeners():void;

  /**
   * Removes all managed listeners for this object.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-clearManagedListeners Original Ext JS documentation of 'clearManagedListeners'
   */
  public native function clearManagedListeners():void;

  /**
   * Enables events fired by this Observable to bubble up an owner hierarchy by calling
   * <code>this.getBubbleTarget()</code> if present. There is no implementation in the Observable
   * base class.
   * <p>This is commonly used by Ext.Components to bubble events to owner Containers.
   * See →<code>ext.Component.getBubbleTarget()</code>. The default implementation in Ext.Component
   * returns the Component's immediate owner. But if a known target is required, this can be
   * overridden to access the required target more quickly.</p>
   * <p>Example:</p>
   * <pre>
   * Ext.define('Ext.overrides.form.field.Base', {
   *     override: 'Ext.form.field.Base',
   *
   *     //  Add functionality to Field's initComponent to enable
   *     // the change event to bubble
   *     initComponent: function () {
   *         this.callParent();
   *         this.enableBubble('change');
   *     }
   * });
   *
   * var myForm = Ext.create('Ext.form.Panel', {
   *     title: 'User Details',
   *     items: [{
   *         ...
   *     }],
   *     listeners: {
   *         change: function() {
   *             // Title goes red if form has been modified.
   *             myForm.header.setStyle('color', 'red');
   *         }
   *     }
   * });
   * </pre>
   * @param eventNames The event name to bubble, or an Array of event names.
   * @see ext.Component#getBubbleTarget()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-enableBubble Original Ext JS documentation of 'enableBubble'
   */
  public native function enableBubble(eventNames:*):void;

  [Deprecated(replacement="fireEventedAction", since="5.5")]
  /**
   * Fires the specified event with the passed parameters and executes a function (action).
   * By default, the action function will be executed after any "before" event handlers
   * (as specified using the <code>order</code> option of
   * <code>→ext.util.Observable.addListener()</code>), but before any other
   * handlers are fired. This gives the "before" handlers an opportunity to
   * cancel the event by returning <code>false</code>, and prevent the action function from
   * being called.
   * <p>The action can also be configured to run after normal handlers, but before any "after"
   * handlers (as specified using the <code>order</code> event option) by passing <code>'after'</code>
   * as the <code>order</code> parameter. This configuration gives any event handlers except
   * for "after" handlers the opportunity to cancel the event and prevent the action
   * function from being called.</p>
   * @param eventName The name of the event to fire.
   * @param args Arguments to pass to handlers and to the action function.
   * @param fn The action function.
   * @param scope The scope (<code>this</code> reference) in which the handler function is
   * executed. <b>If omitted, defaults to the object which fired the event.</b>
   * @param options Event options for the action function. Accepts any
   * of the options of <code>→ext.util.Observable.addListener()</code>
   * @param order The order to call the action function relative
   * too the event handlers (<code>'before'</code> or <code>'after'</code>). Note that this option is
   * simply used to sort the action function relative to the event handlers by "priority".
   * An order of <code>'before'</code> is equivalent to a priority of <code>99.5</code>, while an order of
   * <code>'after'</code> is equivalent to a priority of <code>-99.5</code>. See the <code>priority</code> option
   * of <code>→ext.util.Observable.addListener()</code> for more details.
   * @default 'before'
   * @see ext.util.Observable#addListener()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-fireAction Original Ext JS documentation of 'fireAction'
   */
  public native function fireAction(eventName:String, args:Array, fn:Function, scope:Object = null, options:Object = null, order:String = 'before'):void;

  /**
   * Fires the specified event with the passed parameters (minus the event name, plus
   * the <code>options</code> object passed to →<code>ext.util.Observable.addListener()</code>).
   * <p>An event may be set to bubble up an Observable parent hierarchy (See
   * →<code>ext.Component.getBubbleTarget()</code>) by calling →<code>enableBubble()</code>.</p>
   * @param eventName The name of the event to fire.
   * @param args Variable number of parameters are passed to handlers.
   * @return returns false if any of the handlers return false otherwise it
   * returns true.
   * @see ext.util.Observable#addListener()
   * @see ext.Component#getBubbleTarget()
   * @see #enableBubble()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-fireEvent Original Ext JS documentation of 'fireEvent'
   */
  public native function fireEvent(eventName:String, ...args):Boolean;

  /**
   * Fires the specified event with the passed parameter list.
   * <p>An event may be set to bubble up an Observable parent hierarchy
   * (See →<code>ext.Component.getBubbleTarget()</code>) by calling →<code>enableBubble()</code>.</p>
   * @param eventName The name of the event to fire.
   * @param args An array of parameters which are passed to handlers.
   * @return returns false if any of the handlers return false otherwise
   * it returns true.
   * @see ext.Component#getBubbleTarget()
   * @see #enableBubble()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-fireEventArgs Original Ext JS documentation of 'fireEventArgs'
   */
  public native function fireEventArgs(eventName:String, args:Array):Boolean;

  /**
   * Fires the specified event with the passed parameters and executes a function (action).
   * Evented Actions will automatically dispatch a 'before' event passing. This event will
   * be given a special controller that allows for pausing/resuming of the event flow.
   * <p>By pausing the controller the updater and events will not run until resumed. Pausing,
   * however, will not stop the processing of any other before events.</p>
   * @param eventName The name of the event to fire.
   * @param args Arguments to pass to handlers and to the action function.
   * @param fn The action function.
   * @param scope The scope (<code>this</code> reference) in which the handler function is
   * executed. <b>If omitted, defaults to the object which fired the event.</b>
   * @param fnArgs Optional arguments for the action <code>fn</code>. If not
   * given, the normal <code>args</code> will be used to call <code>fn</code>. If <code>false</code> is passed, the
   * <code>args</code> are used but if the first argument is this instance it will be removed
   * from the args passed to the action function.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-fireEventedAction Original Ext JS documentation of 'fireEventedAction'
   */
  public native function fireEventedAction(eventName:String, args:Array, fn:*, scope:Object = null, fnArgs:* = undefined):Boolean;

  /**
   * @inheritDoc
   */
  public native function getId():String;

  public native function getOptimizedId():String;

  public native function getUniqueId():String;

  /**
   * Checks to see if this object has any listeners for a specified event, or whether
   * the event bubbles. The answer indicates whether the event needs firing or not.
   * @param eventName The name of the event to check for
   * @return <code>true</code> if the event is being listened for or bubbles, else <code>false</code>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-hasListener Original Ext JS documentation of 'hasListener'
   */
  public native function hasListener(eventName:String):Boolean;

  /**
   * Checks if all events, or a specific event, is suspended.
   * @param event The name of the specific event to check
   * @return <code>true</code> if events are suspended
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-isSuspended Original Ext JS documentation of 'isSuspended'
   */
  public native function isSuspended(event:String = null):Boolean;

  /**
   * Shorthand for →<code>addManagedListener()</code>.
   * The addManagedListener method is used when some object (call it "A") is listening
   * to an event on another observable object ("B") and you want to remove that listener
   * from "B" when "A" is destroyed. This is not an issue when "B" is destroyed because
   * all of its listeners will be removed at that time.
   * <p>Example:</p>
   * <pre>
   * Ext.define('Foo', {
   *     extend: 'Ext.Component',
   *
   *     initComponent: function () {
   *         this.addManagedListener(MyApp.SomeSharedMenu, 'show', this.doSomething);
   *         this.callParent();
   *     }
   * });
   * </pre>
   * <p>As you can see, when an instance of Foo is destroyed, it ensures that the 'show'
   * listener on the menu (<code>MyApp.SomeGlobalSharedMenu</code>) is also removed.</p>
   * <p>As of version 5.1 it is no longer necessary to use this method in most cases because
   * listeners are automatically managed if the scope object provided to
   * →<code>ext.util.Observable.addListener()</code> is an Observable instance.
   * However, if the observable instance and scope are not the same object you
   * still need to use <code>→mon()</code> or <code>→addManagedListener()</code> if you want the listener to be
   * managed.</p>
   * @param item The item to which to add
   * a listener/listeners.
   * @param ename The event name, or an object containing event name
   * properties.
   * @param fn If the <code>ename</code> parameter was an event
   * name, this is the handler function or the name of a method on the specified
   * <code>scope</code>.
   * @param scope If the <code>ename</code> parameter was an event name, this is
   * the scope (<code>this</code> reference) in which the handler function is executed.
   * @param options If the <code>ename</code> parameter was an event name, this is
   * the →<code>ext.util.Observable.addListener()</code> options.
   * @return <b>Only when the <code>destroyable</code> option is specified.</b>
   * <p>A <code>Destroyable</code> object. An object which implements the <code>→destroy()</code> method which removes
   * all listeners added in this call. For example:</p>
   * <pre>
   * this.btnListeners = myButton.mon({
   *     destroyable: true
   *     mouseover:   function() { console.log('mouseover'); },
   *     mouseout:    function() { console.log('mouseout'); },
   *     click:       function() { console.log('click'); }
   * });
   * </pre>
   * <p>And when those listeners need to be removed:</p>
   * <pre>
   * Ext.destroy(this.btnListeners);
   * </pre>
   * <p>or</p>
   * <pre>
   * this.btnListeners.destroy();
   * </pre>
   * @see #addManagedListener()
   * @see ext.util.Observable#addListener()
   * @see #mon()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-mon Original Ext JS documentation of 'mon'
   * @see #destroy()
   */
  public native function mon(item:*, ename:*, fn:* = undefined, scope:Object = null, options:Object = null):Object;

  /**
   * Shorthand for →<code>removeManagedListener()</code>.
   * Removes listeners that were added by the →<code>mon()</code> method.
   * @param item The item from which to remove
   * a listener/listeners.
   * @param ename The event name, or an object containing event name
   * properties.
   * @param fn If the <code>ename</code> parameter was an event name, this is
   * the handler function.
   * @param scope If the <code>ename</code> parameter was an event name, this is
   * the scope (<code>this</code> reference) in which the handler function is executed.
   * @see #removeManagedListener()
   * @see #mon()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-mun Original Ext JS documentation of 'mun'
   */
  public native function mun(item:*, ename:*, fn:Function = null, scope:Object = null):void;

  /**
   * The →<code>on()</code> method is shorthand for
   * →<code>ext.util.Observable.addListener()</code>.
   * <p>Appends an event handler to this object. For example:</p>
   * <pre>
   * myGridPanel.on("itemclick", this.onItemClick, this);
   * </pre>
   * <p>The method also allows for a single argument to be passed which is a config object
   * containing properties which specify multiple events. For example:</p>
   * <pre>
   * myGridPanel.on({
   *     cellclick: this.onCellClick,
   *     select: this.onSelect,
   *     viewready: this.onViewReady,
   *     scope: this // Important. Ensure "this" is correct during handler execution
   * });
   * </pre>
   * <p>One can also specify options for each event handler separately:</p>
   * <pre>
   * myGridPanel.on({
   *     cellclick: {fn: this.onCellClick, scope: this, single: true},
   *     viewready: {fn: panel.onViewReady, scope: panel}
   * });
   * </pre>
   * <p><i>Names</i> of methods in a specified scope may also be used:</p>
   * <pre>
   * myGridPanel.on({
   *     cellclick: {fn: 'onCellClick', scope: this, single: true},
   *     viewready: {fn: 'onViewReady', scope: panel}
   * });
   * </pre>
   * @param eventName The name of the event to listen for.
   * May also be an object who's property names are event names.
   * @param fn The method the event invokes or the <i>name</i> of
   * the method within the specified <code>scope</code>. Will be called with arguments
   * given to →<code>ext.util.Observable.fireEvent()</code> plus the <code>options</code> parameter described
   * below.
   * @param scope The scope (<code>this</code> reference) in which the handler function is
   * executed. <b>If omitted, defaults to the object which fired the event.</b>
   * @param options An object containing handler configuration.
   * <p><b>Note:</b> The options object will also be passed as the last argument to every
   * event handler.</p>
   * <p>This object may contain any of the following properties:</p>
   * <ul>
   * <li><code>scope:Object</code> (optional) —
   * The scope (<code>this</code> reference) in which the handler function is executed. <b>If omitted,
   * defaults to the object which fired the event.</b>
   * </li>
   * <li><code>delay:Number</code> (optional) —
   * The number of milliseconds to delay the invocation of the handler after the event
   * fires.
   * </li>
   * <li><code>single:Boolean</code> (optional) —
   * True to add a handler to handle just the next firing of the event, and then remove
   * itself.
   * </li>
   * <li><code>buffer:Number</code> (optional) —
   * Causes the handler to be scheduled to run in an →<code>ext.util.DelayedTask</code> delayed
   * by the specified number of milliseconds. If the event fires again within that time,
   * the original handler is <i>not</i> invoked, but the new handler is scheduled in its place.
   * </li>
   * <li><code>onFrame:Number</code> (optional) —
   * Causes the handler to be scheduled to run at the next
   * <i>animation frame event</i> (→<code>ext.FunctionUtil.requestAnimationFrame()</code>). If the
   * event fires again before that time, the handler is not rescheduled - the handler
   * will only be called once when the next animation frame is fired, with the last set
   * of arguments passed.
   * </li>
   * <li><code>target:ext.util.Observable</code> (optional) —
   * Only call the handler if the event was fired on the target Observable, <i>not</i> if the
   * event was bubbled up from a child Observable.
   * </li>
   * <li><code>element:String</code> (optional) —
   * <b>This option is only valid for listeners bound to <i>Components</i> (→<code>ext.Component</code>).</b>
   * The name of a Component property which references an <i>element</i> (→<code>ext.dom.Element</code>)
   * to add a listener to.
   * <p>This option is useful during Component construction to add DOM event listeners to
   * elements of <i>Components</i> (→<code>ext.Component</code>) which will exist only after the
   * Component is rendered.</p>
   * <p>For example, to add a click listener to a Panel's body:</p>
   * <pre>
   *   var panel = new Ext.panel.Panel({
   *       title: 'The title',
   *       listeners: {
   *           click: this.handlePanelClick,
   *           element: 'body'
   *       }
   *   });
   * </pre>
   * <p>In order to remove listeners attached using the element, you'll need to reference
   * the element itself as seen below.</p>
   * <pre>
   *  panel.body.un(...)
   * </pre>
   * </li>
   * <li><code>delegate:String</code> (optional) —
   * A simple selector to filter the event target or look for a descendant of the target.
   * <p>The "delegate" option is only available on Ext.dom.Element instances (or
   * when attaching a listener to a Ext.dom.Element via a Component using the
   * element option).</p>
   * <p>See the <i>delegate</i> example below.</p>
   * </li>
   * <li><code>capture:Boolean</code> (optional) —
   * When set to <code>true</code>, the listener is fired in the capture phase of the event propagation
   * sequence, instead of the default bubble phase.
   * <p>The <code>→capture()</code> option is only available on Ext.dom.Element instances (or
   * when attaching a listener to a Ext.dom.Element via a Component using the
   * element option).</p>
   * </li>
   * <li><code>stopPropagation:Boolean</code> (optional) —
   * <b>This option is only valid for listeners bound to <i>Elements</i> (→<code>ext.dom.Element</code>).</b>
   * <code>true</code> to call →<code>ext.event.Event.stopPropagation()</code> on the event
   * object before firing the handler.
   * </li>
   * <li><code>preventDefault:Boolean</code> (optional) —
   * <b>This option is only valid for listeners bound to <i>Elements</i> (→<code>ext.dom.Element</code>).</b>
   * <code>true</code> to call →<code>ext.event.Event.preventDefault()</code> on the event
   * object before firing the handler.
   * </li>
   * <li><code>stopEvent:Boolean</code> (optional) —
   * <b>This option is only valid for listeners bound to <i>Elements</i> (→<code>ext.dom.Element</code>).</b>
   * <code>true</code> to call →<code>ext.event.Event.stopEvent()</code> on the event object
   * before firing the handler.
   * </li>
   * <li><code>args:Array</code> (optional) —
   * Optional set of arguments to pass to the handler function before the actual
   * fired event arguments. For example, if <code>args</code> is set to <code>['foo', 42]</code>,
   * the event handler function will be called with an arguments list like this:
   * <pre>
   *  handler('foo', 42, &lt;actual event arguments&gt;...);
   * </pre>
   * </li>
   * <li><code>destroyable:Boolean</code> (default = <code>false</code>) —
   * When specified as <code>true</code>, the function returns a <code>destroyable</code> object. An object
   * which implements the <code>→destroy()</code> method which removes all listeners added in this call.
   * This syntax can be a helpful shortcut to using →<code>un()</code>; particularly when
   * removing multiple listeners. <i>NOTE</i> - not compatible when using the <i>element</i>
   * option. See →<code>un()</code> for the proper syntax for removing listeners added using the
   * <i>element</i> config.
   * </li>
   * <li><code>priority:Number</code> (optional) —
   * An optional numeric priority that determines the order in which event handlers
   * are run. Event handlers with no priority will be run as if they had a priority
   * of 0. Handlers with a higher priority will be prioritized to run sooner than
   * those with a lower priority. Negative numbers can be used to set a priority
   * lower than the default. Internally, the framework uses a range of 1000 or
   * greater, and -1000 or lesser for handlers that are intended to run before or
   * after all others, so it is recommended to stay within the range of -999 to 999
   * when setting the priority of event handlers in application-level code.
   * A priority must be an integer to be valid. Fractional values are reserved for
   * internal framework use.
   * </li>
   * <li><code>order:String</code> (default = <code>'current'</code>) —
   * A legacy option that is provided for backward compatibility.
   * It is recommended to use the <code>priority</code> option instead. Available options are:
   * <ul>
   * <li><code>'before'</code>: equal to a priority of <code>100</code></li>
   * <li><code>'current'</code>: equal to a priority of <code>0</code> or default priority</li>
   * <li><code>'after'</code>: equal to a priority of <code>-100</code></li>
   * </ul>
   * </li>
   * </ul>
   * @param order A shortcut for the <code>order</code> event option. Provided for backward compatibility.
   * Please use the <code>priority</code> event option instead.
   * @default 'current'
   * @return <b>Only when the <code>destroyable</code> option is specified.</b>
   * <p>A <code>Destroyable</code> object. An object which implements the <code>→destroy()</code> method which removes
   * all listeners added in this call. For example:</p>
   * <pre>
   * this.btnListeners =  = myButton.on({
   *     destroyable: true
   *     mouseover:   function() { console.log('mouseover'); },
   *     mouseout:    function() { console.log('mouseout'); },
   *     click:       function() { console.log('click'); }
   * });
   * </pre>
   * <p>And when those listeners need to be removed:</p>
   * <pre>
   * Ext.destroy(this.btnListeners);
   * </pre>
   * <p>or</p>
   * <pre>
   * this.btnListeners.destroy();
   * </pre>
   * @see #on()
   * @see ext.util.Observable#addListener()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-on Original Ext JS documentation of 'on'
   * @see ext.util.Observable#fireEvent()
   * @see ext.util.DelayedTask
   * @see ext.SFunctionUtil#requestAnimationFrame() ext.FunctionUtil.requestAnimationFrame()
   * @see ext.Component
   * @see ext.dom.Element
   * @see ext.mixin.Observable#capture()
   * @see ext.event.Event#stopPropagation()
   * @see ext.event.Event#preventDefault()
   * @see ext.event.Event#stopEvent()
   * @see ext.mixin.Observable#destroy()
   * @see ext.mixin.Observable#un()
   * @see #destroy()
   */
  [Return("this")]
  public native function on(eventName:String, fn:Function, scope:Object = null, options:Object = null, order:String = 'current'):Base;

  /**
   * Appends an after-event handler.
   * <p>Same as →<code>ext.util.Observable.addListener()</code> with <code>order</code> set
   * to <code>'after'</code>.</p>
   * @param eventName The name of the event to listen for.
   * @param fn The method the event invokes.
   * @param scope The scope for <code>fn</code>.
   * @param options An object containing handler configuration.
   * @see ext.util.Observable#addListener()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-onAfter Original Ext JS documentation of 'onAfter'
   */
  public native function onAfter(eventName:String, fn:Function, scope:Object = null, options:Object = null):void;

  /**
   * Appends a before-event handler. Returning <code>false</code> from the handler will stop the event.
   * <p>Same as →<code>ext.util.Observable.addListener()</code> with <code>order</code> set
   * to <code>'before'</code>.</p>
   * @param eventName The name of the event to listen for.
   * @param fn The method the event invokes.
   * @param scope The scope for <code>fn</code>.
   * @param options An object containing handler configuration.
   * @see ext.util.Observable#addListener()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-onBefore Original Ext JS documentation of 'onBefore'
   */
  public native function onBefore(eventName:String, fn:Function, scope:Object = null, options:Object = null):void;

  /**
   * Relays selected events from the specified Observable as if the events were fired
   * by <code>this</code>.
   * <p>For example if you are extending Grid, you might decide to forward some events from
   * store. So you can do this inside your initComponent:</p>
   * <pre>
   * this.relayEvents(this.getStore(), ['load']);
   * </pre>
   * <p>The grid instance will then have an observable 'load' event which will be passed
   * the parameters of the store's load event and any function fired with the grid's
   * load event would have access to the grid using the this keyword (unless the event
   * is handled by a controller's control/listen event listener in which case 'this'
   * will be the controller rather than the grid).</p>
   * @param origin The Observable whose events this object is to relay.
   * @param events Array of event names to relay or an Object with key/value
   * pairs translating to ActualEventName/NewEventName respectively. For example:
   * this.relayEvents(this, {add:'push', remove:'pop'});
   * <p>Would now redispatch the add event of this as a push event and the remove event
   * as a pop event.</p>
   * @param prefix A common prefix to prepend to the event names. For example:
   * <pre>
   * this.relayEvents(this.getStore(), ['load', 'clear'], 'store');
   * </pre>
   * <p>Now the grid will forward 'load' and 'clear' events of store as 'storeload' and
   * 'storeclear'.</p>
   * @return A <code>Destroyable</code> object. An object which implements the <code>→destroy()</code> method
   * which, when destroyed, removes all relayers. For example:
   * <pre>
   * this.storeRelayers = this.relayEvents(this.getStore(), ['load', 'clear'], 'store');
   * </pre>
   * <p>Can be undone by calling</p>
   * <pre>
   * Ext.destroy(this.storeRelayers);
   * </pre>
   * <p>or
   * this.store.relayers.destroy();</p>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-relayEvents Original Ext JS documentation of 'relayEvents'
   * @see #destroy()
   */
  public native function relayEvents(origin:Object, events:*, prefix:String = null):Object;

  /**
   * Alias for →<code>unAfter()</code>.
   * @see #unAfter()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-removeAfterListener Original Ext JS documentation of 'removeAfterListener'
   */
  public native function removeAfterListener():void;

  /**
   * Alias for →<code>unBefore()</code>.
   * @see #unBefore()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-removeBeforeListener Original Ext JS documentation of 'removeBeforeListener'
   */
  public native function removeBeforeListener():void;

  /**
   * Removes an event handler.
   * @param eventName The type of event the handler was associated with.
   * @param fn The handler to remove. <b>This must be a reference to the function
   * passed into the
   * →<code>ext.util.Observable.addListener()</code> call.</b>
   * @param scope The scope originally specified for the handler. It
   * must be the same as the scope argument specified in the original call to
   * →<code>ext.util.Observable.addListener()</code> or the listener will not be removed.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-removeListener Original Ext JS documentation of 'removeListener'
   * @see ext.util.Observable#addListener()
   */
  [Return("this")]
  public native function removeListener(eventName:String, fn:Function, scope:Object = null):void;

  /**
   * Removes listeners that were added by the →<code>mon()</code> method.
   * @param item The item from which to remove
   * a listener/listeners.
   * @param ename The event name, or an object containing event name
   * properties.
   * @param fn If the <code>ename</code> parameter was an event name, this is
   * the handler function.
   * @param scope If the <code>ename</code> parameter was an event name, this is
   * the scope (<code>this</code> reference) in which the handler function is executed.
   * @see #mon()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-removeManagedListener Original Ext JS documentation of 'removeManagedListener'
   */
  public native function removeManagedListener(item:*, ename:*, fn:Function = null, scope:Object = null):void;

  /**
   * Gets the default scope for firing late bound events (string names with
   * no scope attached) at runtime.
   * @param defaultScope (Default this) The default scope to return if none is found.
   * @default this
   * @return The default event scope
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-resolveListenerScope Original Ext JS documentation of 'resolveListenerScope'
   */
  protected native function resolveListenerScope(defaultScope:Object = null):Object;

  /**
   * Resumes firing of the named event(s).
   * <p>After calling this method to resume events, the events will fire when requested to fire.</p>
   * <p><b>Note that if the →<code>suspendEvent()</code> method is called multiple times for a certain
   * event, this converse method will have to be called the same number of times for it
   * to resume firing.</b></p>
   * @param eventName Multiple event names to resume.
   * @see #suspendEvent()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-resumeEvent Original Ext JS documentation of 'resumeEvent'
   */
  public native function resumeEvent(...eventName):void;

  /**
   * Resumes firing events (see →<code>suspendEvents()</code>).
   * <p>If events were suspended using the <code>queueSuspended</code> parameter, then all events fired
   * during event suspension will be sent to any listeners now.</p>
   * @param discardQueue <code>true</code> to prevent any previously queued events from firing
   * while we were suspended. See →<code>suspendEvents()</code>.
   * @see #suspendEvents()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-resumeEvents Original Ext JS documentation of 'resumeEvents'
   */
  public native function resumeEvents(discardQueue:Boolean = false):void;

  public native function setId(id:String):void;

  /**
   * An alias for →<code>ext.util.Observable.addListener()</code>. In
   * versions prior to 5.1, →<code>listeners</code> had a generated setter which could
   * be called to add listeners. In 5.1 the listeners config is not processed
   * using the config system and has no generated setter, so this method is
   * provided for backward compatibility. The preferred way of adding listeners
   * is to use the →<code>on()</code> method.
   * @param listeners The listeners
   * @see ext.util.Observable#addListener()
   * @see #listeners
   * @see #on()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-setListeners Original Ext JS documentation of 'setListeners'
   */
  public native function setListeners(listeners:Object):void;

  /**
   * Suspends firing of the named event(s).
   * <p>After calling this method to suspend events, the events will no longer fire when
   * requested to fire.</p>
   * <p><b>Note that if this is called multiple times for a certain event, the converse method
   * →<code>resumeEvent()</code> will have to be called the same number of times for it to resume
   * firing.</b></p>
   * @param eventName Multiple event names to suspend.
   * @see #resumeEvent()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-suspendEvent Original Ext JS documentation of 'suspendEvent'
   */
  public native function suspendEvent(...eventName):void;

  // TODO: these methods are inherited from private API Ext.mixin.Identifiable

  /**
   * Suspends the firing of all events. (see →<code>resumeEvents()</code>)
   * @param queueSuspended <code>true</code> to queue up suspended events to be fired
   * after the →<code>resumeEvents()</code> call instead of discarding all suspended events.
   * @see #resumeEvents()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-suspendEvents Original Ext JS documentation of 'suspendEvents'
   */
  public native function suspendEvents(queueSuspended:Boolean):void;

  /**
   * Shorthand for →<code>removeListener()</code>.
   * Removes an event handler.
   * @param eventName The type of event the handler was associated with.
   * @param fn The handler to remove. <b>This must be a reference to the function
   * passed into the
   * →<code>ext.util.Observable.addListener()</code> call.</b>
   * @param scope The scope originally specified for the handler. It
   * must be the same as the scope argument specified in the original call to
   * →<code>ext.util.Observable.addListener()</code> or the listener will not be removed.
   * @see #removeListener()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-un Original Ext JS documentation of 'un'
   * @see ext.util.Observable#addListener()
   */
  [Return("this")]
  public native function un(eventName:String, fn:Function, scope:Object = null):*;

  /**
   * Removes a before-event handler.
   * <p>Same as →<code>removeListener()</code> with <code>order</code> set to <code>'after'</code>.</p>
   * @param eventName The name of the event the handler
   * was associated with.
   * @param fn The handler to remove.
   * @param scope The scope originally specified for <code>fn</code>.
   * @param options Extra options object.
   * @see #removeListener()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-unAfter Original Ext JS documentation of 'unAfter'
   */
  public native function unAfter(eventName:*, fn:*, scope:Object = null, options:Object = null):void;

  /**
   * Removes a before-event handler.
   * <p>Same as →<code>removeListener()</code> with <code>order</code> set to <code>'before'</code>.</p>
   * @param eventName The name of the event the handler
   * was associated with.
   * @param fn The handler to remove.
   * @param scope The scope originally specified for <code>fn</code>.
   * @param options Extra options object.
   * @see #removeListener()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Observable.html#method-unBefore Original Ext JS documentation of 'unBefore'
   */
  public native function unBefore(eventName:*, fn:*, scope:Object = null, options:Object = null):void;
}
}