package ext.mixin {
import ext.Base;
import ext.plugin.AbstractPlugin;

[Native("Ext.mixin.Pluggable", require)]
/**
 * This mixin provides support for a <code>→plugins</code> config and related API's.
 * <p>If this mixin is used for non-Components, the statements regarding the host being a
 * Component can be translated accordingly. The only requirement on the user of this class
 * is that the plugins actually used be appropriate for their host.</p>
 * <p>While <code>→ext.Component</code> in the Classic Toolkit supports <code>→plugins</code>, it does not use this
 * class to provide that support. This is due to backwards compatibility in regard to
 * timing changes this implementation would present.</p>
 * <p><b>Important:</b> To ensure plugins are destroyed, call <code>setPlugins(null)</code>.</p>
 * @see #plugins
 * @see ext.Component
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Pluggable.html Original Ext JS documentation of 'Ext.mixin.Pluggable'
 */
public class Pluggable extends Base implements IPluggable {
  [ExtConfig]
  [Bindable]
  /**
   * This config describes one or more plugin config objects used to create plugin
   * instances for this component.
   * <p>Plugins are a way to bundle and reuse custom functionality. Plugins should extend
   * <code>→ext.plugin.AbstractPlugin</code> but technically the only requirement for a valid plugin
   * is that it contain an <code>init</code> method that accepts a reference to its owner. Once
   * a plugin is created, the owner will call the <code>init</code> method, passing a reference
   * to itself. Each plugin can then call methods or respond to events on its owner
   * as needed to provide its functionality.</p>
   * <p>This config's value can take several different forms.</p>
   * <p>The value can be a single string with the plugin's <i>alias</i> (→<code>null</code>):</p>
   * <pre>
   *  var list = Ext.create({
   *      xtype: 'list',
   *      itemTpl: '&lt;div class="item"&gt;{title}&lt;/div&gt;',
   *      store: 'Items',
   *
   *      plugins: 'listpaging'
   *  });
   * </pre>
   * <p>In the above examples, the string "listpaging" is the type alias for
   * <code>Ext.dataview.plugin.ListPaging</code>. The full alias includes the "plugin." prefix
   * (i.e., 'plugin.listpaging').</p>
   * <p>The preferred form for multiple plugins or to configure plugins is the
   * keyed-object form (new in version 6.5):</p>
   * <pre>
   *  var list = Ext.create({
   *      xtype: 'list',
   *      itemTpl: '&lt;div class="item"&gt;{title}&lt;/div&gt;',
   *      store: 'Items',
   *
   *      plugins: {
   *          pullrefresh: true,
   *          listpaging: {
   *              autoPaging: true,
   *              weight: 10
   *          }
   *      }
   *  });
   * </pre>
   * <p>The object keys are the <code>id</code>'s as well as the default type alias. This form
   * allows the value of the <code>→plugins</code> to be merged from base class to derived class
   * and finally with the instance configuration. This allows classes to define a
   * set of plugins that derived classes or instantiators can further configure or
   * disable. This merge behavior is a feature of the
   * <i>config system</i> (→<code>ext.ExtClass.config</code>).</p>
   * <p>The <code>→plugins</code> config can also be an array of plugin aliases (arrays are not
   * merged so this form does not respect plugins defined by the class author):</p>
   * <pre>
   *  var list = Ext.create({
   *      xtype: 'list',
   *      itemTpl: '&lt;div class="item"&gt;{title}&lt;/div&gt;',
   *      store: 'Items',
   *
   *      plugins: ['listpaging', 'pullrefresh']
   *  });
   * </pre>
   * <p>An array can also contain elements that are config objects with a <code>type</code>
   * property holding the type alias:</p>
   * <pre>
   *  var list = Ext.create({
   *      xtype: 'list',
   *      itemTpl: '&lt;div class="item"&gt;{title}&lt;/div&gt;',
   *      store: 'Items',
   *
   *      plugins: ['pullrefresh', {
   *          type: 'listpaging',
   *          autoPaging: true
   *      }]
   *  });
   * </pre>
   * @default null
   * @see ext.plugin.AbstractPlugin
   * @see null
   * @see #plugins
   * @see ext.ExtClass#config
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Pluggable.html#cfg-plugins Original Ext JS documentation of 'plugins'
   * @see #getPlugins()
   * @see #setPlugins()
   */
  public native function get plugins():Array;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set plugins(value:Array):void;

  /**
   * Adds a plugin. For example:
   * <pre>
   *  list.addPlugin('pullrefresh');
   * </pre>
   * <p>Or:</p>
   * <pre>
   *  list.addPlugin({
   *      type: 'pullrefresh',
   *      pullRefreshText: 'Pull to refresh...'
   *  });
   * </pre>
   * @since 6.2.0
   * @param plugin The plugin or config object or
   * alias to add.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Pluggable.html#method-addPlugin Original Ext JS documentation of 'addPlugin'
   */
  public native function addPlugin(plugin:*):void;

  /**
   * Removes and destroys a plugin.
   * <p><b>Note:</b> Not all plugins are designed to be removable. Consult the documentation
   * for the specific plugin in question to be sure.</p>
   * @since 6.2.0
   * @param plugin The plugin or its <code>id</code> to remove.
   * @return plugin instance or <code>null</code> if not found.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Pluggable.html#method-destroyPlugin Original Ext JS documentation of 'destroyPlugin'
   */
  public native function destroyPlugin(plugin:*):AbstractPlugin;

  /**
   * Retrieves plugin by its <code>type</code> alias. For example:
   * <pre>
   *  var list = Ext.create({
   *      xtype: 'list',
   *      itemTpl: '&lt;div class="item"&gt;{title}&lt;/div&gt;',
   *      store: 'Items',
   *
   *      plugins: ['listpaging', 'pullrefresh']
   *  });
   *
   *  list.findPlugin('pullrefresh').setPullRefreshText('Pull to refresh...');
   * </pre>
   * <p><b>Note:</b> See also →<code>getPlugin()</code>.</p>
   * @since 6.2.0
   * @param type The Plugin's <code>type</code> as specified by the class's
   * →<code>ext.ExtClass.alias</code> configuration.
   * @return plugin instance or <code>null</code> if not found.
   * @see #getPlugin()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Pluggable.html#method-findPlugin Original Ext JS documentation of 'findPlugin'
   * @see ext.ExtClass#alias
   */
  public native function findPlugin(type:String):AbstractPlugin;

  /**
   * Retrieves a plugin by its <code>id</code>.
   * <pre>
   *  var list = Ext.create({
   *      xtype: 'list',
   *      itemTpl: '&lt;div class="item"&gt;{title}&lt;/div&gt;',
   *      store: 'Items',
   *
   *      plugins: [{
   *          type: 'pullrefresh',
   *          id: 'foo'
   *      }]
   *  });
   *
   *  list.getPlugin('foo').setPullRefreshText('Pull to refresh...');
   * </pre>
   * <p><b>Note:</b> See also →<code>findPlugin()</code>.</p>
   * @since 6.2.0
   * @param id The <code>id</code> of the plugin.
   * @return plugin instance or <code>null</code> if not found.
   * @see #findPlugin()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Pluggable.html#method-getPlugin Original Ext JS documentation of 'getPlugin'
   */
  public native function getPlugin(id:String):AbstractPlugin;

  /**
   * Returns the value of <code>plugins</code>.
   * @see #plugins
   */
  public native function getPlugins():Array;

  /**
   * Removes and (optionally) destroys a plugin.
   * <p><b>Note:</b> Not all plugins are designed to be removable. Consult the documentation
   * for the specific plugin in question to be sure.</p>
   * @since 6.2.0
   * @param plugin The plugin or its <code>id</code> to remove.
   * @param destroy Pass <code>true</code> to not call <code>destroy()</code> on the plugin.
   * @return plugin instance or <code>null</code> if not found.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.mixin.Pluggable.html#method-removePlugin Original Ext JS documentation of 'removePlugin'
   */
  public native function removePlugin(plugin:*, destroy:Boolean = false):AbstractPlugin;

  /**
   * Sets the value of <code>plugins</code>.
   * @param plugins The new value.
   * @see #plugins
   */
  public native function setPlugins(plugins:Array):void;
}
}