package ext.plugin {
import ext.Component;
import ext.dom.Element;

[Native("Ext.plugin.AbstractClipboard", require)]
/**
 * This base class manages clipboard data transfer for a component. As an abstract class,
 * applications use derived classes such as <code>→ext.grid.plugin.GridClipboardPlugin</code> instead
 * and seldom use this class directly.
 * <p><b><i>Operation</i></b></p>
 * <p>Components that interact with the clipboard do so in two directions: copy and paste.
 * When copying to the clipboard, a component will often provide multiple data formats.
 * On paste, the consumer of the data can then decide what format it prefers and ignore
 * the others.</p>
 * <p><i>Copy (and Cut)</i></p>
 * <p>There are two storage locations provided for holding copied data:</p>
 * <ul>
 * <li>The system clipboard, used to exchange data with other applications running
 * outside the browser.</li>
 * <li>A memory space in the browser page that can hold data for use only by other
 * components on the page. This allows for richer formats to be transferred.</li>
 * </ul>
 * <p>A component can copy (or cut) data in multiple formats as controlled by the
 * <code>→memory</code> and <code>→system</code> configs.</p>
 * <p><i>Paste</i></p>
 * <p>While there may be many formats available, when a component is ready to paste, only
 * one format can ultimately be used. This is specified by the <code>→source</code>
 * config.</p>
 * <p><b><i>Browser Limitations</i></b></p>
 * <p>At the current time, browsers have only a limited ability to interact with the system
 * clipboard. The only reliable, cross-browser, plugin-in-free technique for doing so is
 * to use invisible elements and focus tricks <b>during</b> the processing of clipboard key
 * presses like CTRL+C (on Windows/Linux) or CMD+C (on Mac).</p>
 * @see ext.grid.plugin.GridClipboardPlugin
 * @see #memory
 * @see #system
 * @see #source
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.AbstractClipboard.html Original Ext JS documentation of 'Ext.plugin.AbstractClipboard'
 */
public class AbstractClipboardPlugin extends AbstractPlugin {
  /**
   * @param config @inheritDoc
   */
  public function AbstractClipboardPlugin(config:AbstractClipboardPlugin = null) {
    super();
  }

  [ExtConfig]
  [Bindable]
  /**
   * This object is keyed by the names of the data formats supported by this plugin.
   * The property values of this object are objects with <code>get</code> and <code>put</code> properties
   * that name the methods for getting data from (copy) and putting to into (paste)
   * the associated component.
   * <p>For example:</p>
   * <pre>
   *  formats: {
   *      html: {
   *          get: 'getHtmlData',
   *          put: 'putHtmlData'
   *      }
   *  }
   * </pre>
   * <p>This declares support for the "html" data format and indicates that the
   * <code>getHtmlData</code> method should be called to copy HTML data from the component,
   * while the <code>putHtmlData</code> should be called to paste HTML data into the component.</p>
   * <p>By default, all derived classes must support a "text" format:</p>
   * <pre>
   *  formats: {
   *      text: {
   *          get: 'getTextData',
   *          put: 'putTextData'
   *      }
   *  }
   * </pre>
   * <p>To understand the method signatures required to implement a data format, see the
   * documentation for <code>→getTextData()</code> and <code>→putTextData()</code>.</p>
   * <p>The format name "system" is not allowed.</p>
   * @default {
   *     text: {
   *         get: 'getTextData',
   *         put: 'putTextData'
   *     }
   * }
   * @see #getTextData()
   * @see #putTextData()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.AbstractClipboard.html#cfg-formats Original Ext JS documentation of 'formats'
   * @see #getFormats()
   * @see #setFormats()
   */
  public native function get formats():Object;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set formats(value:Object):void;

  [ExtConfig]
  [Bindable]
  /**
   * The data format(s) to copy to the private, memory clipboard. By default, data
   * is not saved to the memory clipboard. Specify <code>true</code> to include all formats
   * of data, or a string to copy a single format, or an array of strings to copy
   * multiple formats.
   * @default null
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.AbstractClipboard.html#cfg-memory Original Ext JS documentation of 'memory'
   * @see #getMemory()
   * @see #setMemory()
   */
  public native function get memory():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set memory(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * The format or formats in order of preference when pasting data. This list can
   * be any of the valid formats, plus the name "system". When a paste occurs, this
   * config is consulted. The first format specified by this config that has data
   * available in the private memory space is used. If "system" is encountered in
   * the list, whatever data is available on the system clipboard is chosen. At
   * that point, no further source formats will be considered.
   * @default "system"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.AbstractClipboard.html#cfg-source Original Ext JS documentation of 'source'
   * @see #getSource()
   * @see #setSource()
   */
  public native function get source():*;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set source(value:*):void;

  [ExtConfig]
  [Bindable]
  /**
   * The data format to set in the system clipboard. By default, the "text"
   * format is used. Based on the type of derived class, other formats may be
   * possible.
   * @default "text"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.AbstractClipboard.html#cfg-system Original Ext JS documentation of 'system'
   * @see #getSystem()
   * @see #setSystem()
   */
  public native function get system():String;

  [ExtConfig]
  [Bindable]
  /**
   * @private
   */
  public native function set system(value:String):void;

  /**
   * Returns the value of <code>formats</code>.
   * @see #formats
   */
  public native function getFormats():Object;

  /**
   * Returns the value of <code>memory</code>.
   * @see #memory
   */
  public native function getMemory():*;

  /**
   * Returns the value of <code>source</code>.
   * @see #source
   */
  public native function getSource():*;

  /**
   * Returns the value of <code>system</code>.
   * @see #system
   */
  public native function getSystem():String;

  /**
   * Returns the element target to listen to copy/paste.
   * @param comp The component this plugin is initialized on.
   * @return The element target.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.AbstractClipboard.html#method-getTarget Original Ext JS documentation of 'getTarget'
   */
  public native function getTarget(comp:Component):Element;

  /**
   * This method returns the selected data in text format.
   * @param format The name of the format (i.e., "text").
   * @param erase Pass <code>true</code> to erase (cut) the data, <code>false</code> to just copy.
   * @return The data in text format.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.AbstractClipboard.html#method-getTextData Original Ext JS documentation of 'getTextData'
   */
  public native function getTextData(format:String, erase:Boolean):String;

  /**
   * This method pastes the given text data.
   * @param data The data in the indicated <code>format</code>.
   * @param format The name of the format (i.e., "text").
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.AbstractClipboard.html#method-putTextData Original Ext JS documentation of 'putTextData'
   */
  public native function putTextData(data:Object, format:String):void;

  /**
   * Sets the value of <code>formats</code>.
   * @param formats The new value.
   * @see #formats
   */
  public native function setFormats(formats:Object):void;

  /**
   * Sets the value of <code>memory</code>.
   * @param memory The new value.
   * @see #memory
   */
  public native function setMemory(memory:*):void;

  /**
   * Sets the value of <code>source</code>.
   * @param source The new value.
   * @see #source
   */
  public native function setSource(source:*):void;

  /**
   * Sets the value of <code>system</code>.
   * @param system The new value.
   * @see #system
   */
  public native function setSystem(system:String):void;
}
}