package ext.plugin {
import ext.Component;
import ext.Plugin;
import ext.mixin.IIdentifiable;

[Mixin("ext.plugin.AbstractPlugin")]
/**
 * This is the base class from which all plugins should extend.
 * <p>This class defines the essential API of plugins as used by Components by defining the
 * following methods:</p>
 * <ul>
 * <li>
 * <p><code>→init()</code> : The plugin initialization method which the host Component calls during
 * Component initialization. The Component passes itself as the sole parameter.
 * Subclasses should set up bidirectional links between the plugin and its host
 * Component here.</p>
 * </li>
 * <li>
 * <p><code>→destroy()</code> : The plugin cleanup method which the host Component calls at Component
 * destruction time. Use this method to break links between the plugin and the
 * Component and to free any allocated resources.</p>
 * </li>
 * </ul>
 * @see ext.plugin.AbstractPlugin#init()
 * @see ext.plugin.AbstractPlugin#destroy()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html Original Ext JS documentation of 'Ext.plugin.Abstract'
 * @see ext.plugin.AbstractPlugin
 */
public interface IAbstractPlugin extends Plugin, IIdentifiable {
  [ExtConfig] /**
   * An identifier for the plugin that can be set at creation time to later retrieve the
   * plugin using the →<code>getPlugin</code> method. For example:
   * <pre>
   *  var panel = Ext.create({
   *      xtype: 'panel',
   *
   *      plugins: {
   *          foobar: {
   *              id: 'foo',
   *              ...
   *          }
   *      }
   *  });
   *
   *  // later on:
   *  var plugin = panel.getPlugin('foo');
   * </pre>
   * @since 6.2.0
   * @see #getPlugin
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#cfg-id Original Ext JS documentation of 'id'
   */
  function get id():String;

  [ExtConfig] function set id(value:String):void;

  /**
   * The value <code>true</code> to identify objects of this class or a subclass thereof.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#property-isPlugin Original Ext JS documentation of 'isPlugin'
   */
  function get isPlugin():Boolean;

  [Deprecated(replacement="id", since="6.2.0")]
  [ExtConfig] /**
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#cfg-pluginId Original Ext JS documentation of 'pluginId'
   */
  function get pluginId():String;

  [ExtConfig] function set pluginId(value:String):void;

  [ExtConfig] /**
   * The configured list of stateEvents used to (optionally) participate in Owner Component's
   * state management.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#cfg-stateEvents Original Ext JS documentation of 'stateEvents'
   */
  function get stateEvents():*;

  [ExtConfig] function set stateEvents(value:*):void;

  /**
   * The applyState method is invoked by the client Component's State mixin after initComponent
   * method has been run for the client.
   * <p>The supplied implementation is empty. If plugin Subclasses are to (optionally) participate
   * in the client Component's state management, implementers should provide a suitable method
   * to utilize it.</p>
   * @param state The current plugin state object to be applied.
   * @param allState The current aggregate state of the Component and all plugins.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#method-applyState Original Ext JS documentation of 'applyState'
   */
  function applyState(state:Object, allState:Object):void;

  /**
   * Creates clone of the plugin.
   * @param overrideCfg Additional config for the derived plugin.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#method-clonePlugin Original Ext JS documentation of 'clonePlugin'
   */
  function clonePlugin(overrideCfg:Object = null):void;

  /**
   * The destroy method is invoked by the owning Component at the time the Component is
   * being destroyed.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#method-destroy Original Ext JS documentation of 'destroy'
   */
  function destroy(...params):void;

  /**
   * Plugins that can be disconnected from their host component should implement
   * this method.
   * @since 6.2.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#method-detachCmp Original Ext JS documentation of 'detachCmp'
   */
  function detachCmp():void;

  /**
   * The base implementation just sets the plugin's <code>disabled</code> flag to <code>true</code>.
   * <p>Plugin subclasses which need more complex processing may implement an overriding
   * implementation.</p>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#method-disable Original Ext JS documentation of 'disable'
   */
  function disable():void;

  /**
   * The base implementation just sets the plugin's <code>disabled</code> flag to <code>false</code>.
   * <p>Plugin subclasses which need more complex processing may implement an overriding
   * implementation.</p>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#method-enable Original Ext JS documentation of 'enable'
   */
  function enable():void;

  /**
   * Returns the component to which this plugin is attached.
   * @return The owning host component.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#method-getCmp Original Ext JS documentation of 'getCmp'
   */
  function getCmp():Component;

  /**
   * The getState method is invoked by the client Component's State mixin when one or more of the
   * specified →<code>stateEvents</code> are raised.
   * <p>The supplied implementation is empty. If plugin Subclasses are to (optionally) participate
   * in the client Component's state management, implementers should provide a suitable method
   * which returns a state object.</p>
   * @return state
   * @see ext.plugin.AbstractPlugin#stateEvents
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#method-getState Original Ext JS documentation of 'getState'
   */
  function getState():Object;

  /**
   * Sets the host component to which this plugin is attached. For a plugin to be
   * removable without being destroyed, this method should be provided and be prepared
   * to receive <code>null</code> for the component.
   * @param host The owning host component.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.plugin.Abstract.html#method-setCmp Original Ext JS documentation of 'setCmp'
   */
  function setCmp(host:Component):void;
}
}