package ext.promise {
import ext.Base;
import ext.IPromise;

[Native("Ext.promise.Promise", require)]
/**
 * Promises represent a future value; i.e., a value that may not yet be available.
 * <p>Users should <b>not</b> create instances of this class directly. Instead user code should
 * use <code>new →ext.Promise()</code> or <code>new →ext.Deferred()</code> to create and manage
 * promises. If the browser supports the standard <code>Promise</code> constructor, this class will
 * not be used by <code>→ext.Promise</code>. This class will always be used by <code>→ext.Deferred</code> in order
 * to provide enhanced capabilities beyond standard promises.</p>
 * <p>A Promise's <code>→then()</code> method is used to specify onFulfilled and onRejected
 * callbacks that will be notified when the future value becomes available. Those callbacks
 * can subsequently transform the value that was resolved or the reason that was rejected.
 * Each call to <code>→then()</code> returns a new Promise of that transformed value; i.e., a Promise
 * that is resolved with the callback return value or rejected with any error thrown by
 * the callback.</p>
 * <p><b><i>Basic Usage</i></b></p>
 * <pre>
 *  this.companyService.loadCompanies().then(
 *      function(records) {
 *          // Do something with result.
 *      },
 *      function(error) {
 *          // Do something on failure.
 *      }).
 *  always(function() {
 *      // Do something whether call succeeded or failed
 *  });
 * </pre>
 * <p>The above code uses the <code>Promise</code> returned from the <code>companyService.loadCompanies()</code>
 * method and uses <code>then()</code> to attach success and failure handlers. Finally, an <code>always()</code>
 * method call is chained onto the returned promise. This specifies a callback function
 * that will run whether the underlying call succeeded or failed.</p>
 * <p>See <code>→ext.Deferred</code> for an example of using the returned Promise.</p>
 * @see ext.Promise
 * @see ext.Deferred
 * @see #then()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.promise.Promise.html Original Ext JS documentation of 'Ext.promise.Promise'
 */
public class ExtPromise extends Base implements IPromise {
  /**
   * The type of <code>Error</code> propagated by the <code>→cancel()</code> method. If
   * the browser provides a native <code>→CancellationError</code> then that type is used. If
   * not, a basic <code>Error</code> type is used.
   * @default Ext.global.CancellationError || Error
   * @see #cancel()
   * @see #CancellationError
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.promise.Promise.html#static-property-CancellationError Original Ext JS documentation of 'CancellationError'
   */
  public static native function get CancellationError():*;

  /**
   * @private
   */
  public static native function set CancellationError(value:*):void;

  /**
   * Attaches an onCompleted callback that will be notified when this Promise is completed.
   * <p>Similar to <code>finally</code> in <code>try... catch... finally</code>.</p>
   * <p>NOTE: The specified callback does not affect the resulting Promise's outcome; any
   * return value is ignored and any Error is rethrown.</p>
   * @param onCompleted Callback to execute when the Promise is resolved or
   * rejected.
   * @param scope Optional scope for the callback.
   * @return A new "pass-through" Promise that is resolved with
   * the original value or rejected with the original reason.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.promise.Promise.html#method-always Original Ext JS documentation of 'always'
   */
  public native function always(onCompleted:Function, scope:Object = null):IPromise;

  /**
   * Cancels this Promise if it is still pending, triggering a rejection with a
   * <code>→CancellationError</code> that will propagate to any Promises originating from
   * this Promise.
   * <p>NOTE: Cancellation only propagates to Promises that branch from the target Promise.
   * It does not traverse back up to parent branches, as this would reject nodes from
   * which other Promises may have branched, causing unintended side-effects.</p>
   * @param reason Cancellation reason.
   * @see #CancellationError
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.promise.Promise.html#method-cancel Original Ext JS documentation of 'cancel'
   */
  public native function cancel(reason:Error):void;

  /**
   * Attaches an onRejected callback that will be notified if this Promise is rejected.
   * <p>The callback can subsequently transform the reason that was rejected. Each call to
   * <code>→otherwise()</code> returns a new Promise of that transformed value; i.e., a Promise that
   * is resolved with the original resolved value, or resolved with the callback return
   * value or rejected with any error thrown by the callback.</p>
   * @since 6.5.0
   * @param onRejected Callback to execute to transform a rejection reason.
   * @param scope Optional scope for the callback.
   * @return Promise of the transformed future value.
   * @see #otherwise()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.promise.Promise.html#method-catch Original Ext JS documentation of 'catch'
   */
  public native function catch_(onRejected:Function, scope:Object = null):IPromise;

  /**
   * Terminates a Promise chain, ensuring that unhandled rejections will be rethrown as
   * Errors.
   * <p>One of the pitfalls of interacting with Promise-based APIs is the tendency for
   * important errors to be silently swallowed unless an explicit rejection handler is
   * specified.</p>
   * <p>For example:</p>
   * <pre>
   *  promise.then(function() {
   *      // logic in your callback throws an error and it is interpreted as a
   *      // rejection. throw new Error("Boom!");
   *  });
   *
   *  // The Error was not handled by the Promise chain and is silently swallowed.
   * </pre>
   * <p>This problem can be addressed by terminating the Promise chain with the done()
   * method:</p>
   * <pre>
   *  promise.then(function() {
   *      // logic in your callback throws an error and it is interpreted as a
   *      // rejection. throw new Error("Boom!");
   *  }).done();
   *
   * // The Error was not handled by the Promise chain and is rethrown by done() on
   * // the next tick.
   * </pre>
   * <p>The <code>done()</code> method ensures that any unhandled rejections are rethrown as Errors.</p>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.promise.Promise.html#method-done Original Ext JS documentation of 'done'
   */
  public native function done():void;

  /**
   * Logs the resolution or rejection of this Promise with the specified category and
   * optional identifier. Messages are logged via all registered custom logger functions.
   * @param identifier An optional identifier to incorporate into the
   * resulting log entry.
   * @return A new "pass-through" Promise that is resolved with
   * the original value or rejected with the original reason.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.promise.Promise.html#method-log Original Ext JS documentation of 'log'
   */
  public native function log(identifier:String):IPromise;

  /**
   * An alias for the →<code>catch_()</code> method. To be used for browsers
   * where catch cannot be used as a method name.
   * @see #catch_()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.promise.Promise.html#method-otherwise Original Ext JS documentation of 'otherwise'
   */
  public native function otherwise(onRejected:Function, scope:Object = null):void;

  /**
   * Attaches onFulfilled and onRejected callbacks that will be notified when the future
   * value becomes available.
   * <p>Those callbacks can subsequently transform the value that was fulfilled or the error
   * that was rejected. Each call to <code>→then()</code> returns a new Promise of that transformed
   * value; i.e., a Promise that is fulfilled with the callback return value or rejected
   * with any error thrown by the callback.</p>
   * @param onFulfilled Optional callback to execute to transform a
   * fulfillment value.
   * @param onRejected Optional callback to execute to transform a rejection
   * reason.
   * @param onProgress Optional callback function to be called with progress
   * updates.
   * @return Promise that is fulfilled with the callback return
   * value or rejected with any error thrown by the callback.
   * @see #then()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.promise.Promise.html#method-then Original Ext JS documentation of 'then'
   */
  public native function then(onFulfilled:Function = null, onRejected:Function = null, onProgress:Function = null):IPromise;
}
}