package ext.route {
import ext.IMixin;

[Mixin("ext.route.RouteMixin")]
/**
 * A mixin to allow any class to configure and listen to routes and also change the hash.
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.route.Mixin.html Original Ext JS documentation of 'Ext.route.Mixin'
 * @see ext.route.RouteMixin
 */
public interface IRouteMixin extends IMixin {
  [ExtConfig]
  [Bindable] /**
   * An object of routes to handle hash changes. A route can be defined in a simple way:
   * <pre>
   * routes: {
   *     'foo/bar': 'handleFoo',
   *     'user/:id': 'showUser'
   * }
   * </pre>
   * <p>Where the property is the hash (which can accept a parameter defined by a colon)
   * and the value is the method on the controller to execute. The parameters will get sent
   * in the action method.</p>
   * <p>If no routes match a given hash, an →<code>ext.GlobalEvents.unmatchedroute</code> event
   * will be fired. This can be listened to in four ways:</p>
   * <pre>
   * Ext.on('unmatchedroute', function(token) {});
   *
   * Ext.define('MyApp.controller.Foo', {
   *     extend: 'Ext.app.Controller',
   *
   *     listen: {
   *         global: {
   *             unmatchedroute: 'onUnmatchedRoute'
   *         }
   *     },
   *
   *     onUnmatchedRoute: function(token) {}
   * });
   *
   * Ext.application({
   *     name: 'MyApp',
   *
   *     listen: {
   *         global: {
   *             unmatchedroute: 'onUnmatchedRoute'
   *         }
   *     },
   *
   *     onUnmatchedRoute: function(token) {}
   * });
   *
   * Ext.application({
   *     name: 'MyApp',
   *
   *     listeners: {
   *         unmatchedroute: 'onUnmatchedRoute'
   *     },
   *
   *     onUnmatchedRoute: function(token) {}
   * });
   * </pre>
   * <p>There is also a complex means of defining a route where you can use a before action
   * and even specify your own RegEx for the parameter:</p>
   * <pre>
   * routes: {
   *     'foo/bar': {
   *         action: 'handleFoo',
   *         before: 'beforeHandleFoo'
   *     },
   *     'user/:id': {
   *         action: 'showUser',
   *         before: 'beforeShowUser',
   *         conditions: {
   *             ':id': '([0-9]+)'
   *         }
   *     }
   * }
   * </pre>
   * <p>This will only match if the <code>id</code> parameter is a number.</p>
   * <p>The before action allows you to cancel an action. Every before action will get passed
   * an <code>action</code> argument with a <code>resume</code> and <code>stop</code> methods as the last argument of the
   * method and you <i>MUST</i> execute either method:</p>
   * <pre>
   * beforeHandleFoo: function (action) {
   *     // some logic here
   *
   *     // this will allow the handleFoo action to be executed
   *     action.resume();
   * },
   * handleFoo: function () {
   *     // will get executed due to true being passed in callback in beforeHandleFoo
   * },
   * beforeShowUser: function (id, action) {
   *     // allows for async process like an Ajax
   *     Ext.Ajax.request({
   *         url: 'foo.php',
   *         success: function () {
   *             // will not allow the showUser method to be executed
   *             // but will continue other queued actions.
   *             action.stop();
   *         },
   *         failure: function () {
   *             // will not allow the showUser method to be executed
   *             // and will not allow other queued actions to be executed.
   *             action.stop(true);
   *         }
   *     });
   * },
   * showUser: function (id) {
   *     // will not get executed due to false being passed in callback in beforeShowUser
   * }
   * </pre>
   * <p>You <b>MUST</b> execute the <code>→ext.route.RouteAction.resume()</code> or
   * <code>→ext.route.RouteAction.stop()</code> method on the <code>action</code> argument. Executing
   * <code>action.resume();</code> will continue the action, <code>action.stop();</code> will prevent
   * further execution.</p>
   * <p>The default RegEx that will be used is <code>([%a-zA-Z0-9\\-\\_\\s,]+)</code> but you can specify
   * any that may suit what you need to accomplish. An example of an advanced condition
   * may be to make a parameter optional and case-insensitive:</p>
   * <pre>
   * routes: {
   *     'user:id': {
   *         action: 'showUser',
   *         before: 'beforeShowUser',
   *         conditions: {
   *             ':id': '(?:(?:\/){1}([%a-z0-9_,\s\-]+))?'
   *         }
   *     }
   * }
   * </pre>
   * <p>Each route can be named; this allows for the route to be looked up by name instead of
   * url. By default, the route's name will be the url you configure but you can provide
   * the <code>→ext.route.Route.name</code> config to override the default:</p>
   * <pre>
   * routes: {
   *     'user:id': {
   *         action: 'showUser',
   *         before: 'beforeShowUser',
   *         name: 'user',
   *         conditions: {
   *             ':id': '(?:(?:\/){1}([%a-z0-9_,\s\-]+))?'
   *         }
   *     }
   * }
   * </pre>
   * <p>The <code>user:id</code> route can not be looked up via the <code>user</code> name which is useful when using
   * <code>→redirectTo()</code>.</p>
   * <p>A wildcard route can also be defined which works exactly like any other route but will
   * always execute before any other route. To specify a wildcard route, use the <code>&#42;</code>
   * as the url:</p>
   * <pre>
   * routes: {
   *     '&#42;': 'onToken'
   * }
   * </pre>
   * <p>Since a wildcard route will execute before any other route, it can delay the execution
   * of other routes allowing for such things like a user session to be retrieved:</p>
   * <pre>
   * routes: {
   *     '&#42;': {
   *         before: 'onBeforeToken'
   *     }
   * },
   *
   * onBeforeToken: function () {
   *     return Ext.Ajax.request({
   *         url: '/user/session'
   *     });
   * }
   * </pre>
   * <p>In the above example, no other route will execute unless that
   * →<code>ext.Ajax.request()</code> returns successfully.</p>
   * <p>You can also use a wildcard route if you need to defer routes until a store has been
   * loaded when an application first starts up:</p>
   * <pre>
   * routes: {
   *     '&#42;': {
   *         before: 'onBeforeToken'
   *     }
   * },
   *
   * onBeforeToken: function (action) {
   *     var store = Ext.getStore('Settings');
   *
   *     if (store.loaded) {
   *         action.resume();
   *     } else {
   *         store.on('load', action.resume, action, { single: true });
   *     }
   * }
   * </pre>
   * <p>The valid options are configurations from →<code>ext.route.RouteHandler</code> and
   * →<code>ext.route.Route</code>.</p>
   * @see ext.SGlobalEvents#unmatchedroute ext.GlobalEvents.unmatchedroute
   * @see ext.route.RouteAction#resume()
   * @see ext.route.RouteAction#stop()
   * @see ext.route.Route#name
   * @see #redirectTo()
   * @see ext.SAjax#request() ext.Ajax.request()
   * @see ext.route.RouteHandler
   * @see ext.route.Route
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.route.Mixin.html#cfg-routes Original Ext JS documentation of 'routes'
   * @see #getRoutes()
   * @see #setRoutes()
   */
  function get routes():Object;

  [ExtConfig]
  [Bindable] function set routes(value:Object):void;

  /**
   * Returns the value of <code>routes</code>.
   * @see #routes
   */
  function getRoutes():Object;

  /**
   * Update the hash. By default, it will not execute the routes if the current token and the
   * token passed are the same.
   * @param hash The hash to redirect to. The hash can be
   * of several values:
   * <ul>
   * <li>
   * <p><b>String</b> The hash to exactly be set to.</p>
   * </li>
   * <li>
   * <p><b>Number</b> If <code>1</code> is passed, →<code>ext.util.History.forward()</code> function will be
   * executed. If <code>-1</code> is passed, <i>back</i> (→<code>ext.util.History.bck</code>) function will be executed.</p>
   * </li>
   * <li>
   * <p><b>Ext.data.Model</b> If a model instance is passed, the Model's
   * →<code>ext.data.Model.toUrl()</code> function will be executed to convert it into a String
   * value to set the hash to.</p>
   * </li>
   * <li>
   * <p><b>Object</b> An Object can be passed to control individual tokens in the full hash.
   * The key should be an associated →<code>ext.route.Route</code>'s
   * →<code>ext.route.Route.name</code> and the value should be the value of that token
   * in the complete hash. For example, if you have two routes configured, each token in the
   * hash that can be matched for each route can be individually controlled:</p>
   * <p>routes: {
   * 'foo/bar': 'onFooBar',
   * 'baz/:id': {
   * action: 'onBaz',
   * name: 'baz'
   * }
   * }</p>
   * </li>
   * </ul>
   * <p>If you pass in a hash of <code>#foo/bar|baz/1</code>, each route will execute in response. If you want
   * to change only the <code>baz</code> route but leave the <code>foo/bar</code> route in the hash, you can pass only
   * the <code>baz</code> key in an object:</p>
   * <pre>
   * this.redirectTo({
   *     baz : 'baz/5'
   * });
   * </pre>
   * <p>and the resulting hash will be <code>#foo/bar/|baz/5</code> and only the <code>baz</code> route will execute
   * in reaction but the <code>foo/bar</code> will not react since it's associated token in the hash
   * remained the same. If you wanted to update the <code>baz</code> route and remove <code>foo/bar</code>
   * from the hash, you can set the value to <code>null</code>:</p>
   * <pre>
   * this.redirectTo({
   *     'foo/bar': null,
   *     baz: 'baz/3'
   * });
   * </pre>
   * <p>and the resulting hash will be <code>#baz/3</code>. Like before, the <code>baz</code> route will execute
   * in reaction.</p>
   * @param opt An optional <code>→Object</code> describing how to enact the hash being passed in.
   * Valid options are:
   * <ul>
   * <li><code>force</code> Even if the hash will not change, setting this to <code>true</code> will force the
   * →<code>ext.route.Router</code> to react.</li>
   * <li><code>replace</code> When set to <code>true</code>, this will replace the current resource in the history stack
   * with the hash being set.</li>
   * </ul>
   * <p>For backwards compatibility, if <code>true</code> is passed instead of an <code>→Object</code>, this will set
   * the <code>force</code> option to <code>true</code>.</p>
   * @return Will return <code>true</code> if the token was updated.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.route.Mixin.html#method-redirectTo Original Ext JS documentation of 'redirectTo'
   * @see ext.util.SHistory#forward() ext.util.History.forward()
   * @see ext.util.SHistory#bck ext.util.History.bck
   * @see ext.data.Model#toUrl()
   * @see ext.route.Route
   * @see ext.route.Route#name
   * @see Object
   * @see ext.route.#Router ext.route.Router
   */
  function redirectTo(hash:*, opt:Object):Boolean;

  /**
   * Sets the value of <code>routes</code>.
   * @param routes The new value.
   * @see #routes
   */
  function setRoutes(routes:Object):void;
}
}