package ext.selection {
import ext.grid.column.Column;

[Native("Ext.selection.CheckboxModel", require)]
/**
 * A selection model that renders a column of checkboxes that can be toggled to
 * select or deselect rows. The default mode for this selection model is MULTI.
 * <p><b>Example:</b></p>
 * <pre>
 *   var store = Ext.create('Ext.data.Store', {
 *       fields: ['name', 'email', 'phone'],
 *       data: [{
 *           name: 'Lisa',
 *           email: 'lisa&#64;simpsons.com',
 *           phone: '555-111-1224'
 *       }, {
 *           name: 'Bart',
 *           email: 'bart&#64;simpsons.com',
 *           phone: '555-222-1234'
 *       }, {
 *           name: 'Homer',
 *           email: 'homer&#64;simpsons.com',
 *           phone: '555-222-1244'
 *       }, {
 *           name: 'Marge',
 *           email: 'marge&#64;simpsons.com',
 *           phone: '555-222-1254'
 *       }]
 *   });
 *
 *   Ext.create('Ext.grid.Panel', {
 *       title: 'Simpsons',
 *       store: store,
 *       columns: [{
 *           text: 'Name',
 *           dataIndex: 'name'
 *       }, {
 *           text: 'Email',
 *           dataIndex: 'email',
 *           flex: 1
 *       }, {
 *           text: 'Phone',
 *           dataIndex: 'phone'
 *       }],
 *       height: 200,
 *       width: 400,
 *       renderTo: Ext.getBody(),
 *       selModel: {
 *           selType: 'checkboxmodel'
 *       }
 *   });
 * </pre>
 * <p>The selection model will inject a header for the checkboxes in the first view
 * and according to the →<code>injectCheckbox</code> configuration.</p>
 * @see #injectCheckbox
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html Original Ext JS documentation of 'Ext.selection.CheckboxModel'
 */
public class CheckboxSelectionModel extends RowSelectionModel {
  /**
   * @param config @inheritDoc
   */
  public function CheckboxSelectionModel(config:CheckboxSelectionModel = null) {
    super();
  }

  [ExtConfig]
  /**
   * True if rows can only be selected by clicking on the checkbox column, not by clicking
   * on the row itself. Note that this only refers to selection via the UI, programmatic
   * selection will still occur regardless.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#cfg-checkOnly Original Ext JS documentation of 'checkOnly'
   */
  public native function get checkOnly():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set checkOnly(value:Boolean):void;

  [ExtConfig]
  /**
   * A renderer to be used in conjunction with
   * →<code>ext.grid.plugin.RowEditingPlugin</code>. This renderer is used to display a
   * custom value for non-editable fields.
   * <p><b>Note:</b> The editRenderer is called when the roweditor is initially shown.
   * Changes to the record during editing will not call editRenderer.</p>
   * <pre>
   * var store = Ext.create('Ext.data.Store', {
   *     fields: ['name', 'email'],
   *     data: [{
   *         "name": "Finn",
   *         "email": "finn&#64;adventuretime.com"
   *     }, {
   *         "name": "Jake",
   *         "email": "jake&#64;adventuretime.com"
   *     }]
   * });
   *
   * Ext.create('Ext.grid.Panel', {
   *     title: 'Land Of Ooo',
   *     store: store,
   *     columns: [{
   *         text: 'Name',
   *         dataIndex: 'name',
   *         editRenderer: function(value){
   *             return '&lt;span style="color:gray;"&gt;' + value + '&lt;/span&gt;';
   *           }
   *     }, {
   *         text: 'Email',
   *         dataIndex: 'email',
   *         flex: 1,
   *         editor: {
   *             xtype: 'textfield',
   *             allowBlank: false
   *         }
   *     }],
   *     plugins: {
   *         rowediting: {
   *             clicksToEdit: 1
   *         }
   *     },
   *     height: 200,
   *     width: 400,
   *     renderTo: document.body
   * });
   * </pre>
   * <ul>
   * <li><code>value:Object</code> —
   * The data value for the current cell
   * <pre>
   * editRenderer: function(value){
   *     // evaluates `value` to append either `person' or `people`
   *     return Ext.util.Format.plural(value, 'person', 'people');
   * }
   * </pre>
   * </li>
   * <li><code>metaData:Object</code> —
   * <b>Note:</b> The metadata param is passed to the
   * editRenderer, but is not used.
   * </li>
   * <li><code>record:ext.data.Model</code> —
   * The record for the current row
   * <pre>
   * editRenderer: function (value, metaData, record) {
   *     // evaluate the record's `updated` field and if truthy return the value
   *     // from the `newVal` field, else return value
   *     var updated = record.get('updated');
   *     return updated ? record.get('newVal') : value;
   * }
   * </pre>
   * </li>
   * <li><code>rowIndex:Number</code> —
   * The index of the current row
   * <pre>
   * editRenderer: function (value, metaData, record, rowIndex) {
   *     // style the value differently for even / odd values
   *     var odd = (rowIndex % 2 === 0),
   *         color = (odd ? 'gray' : 'red');
   *     return '&lt;span style="color:' + color + ';"&gt;' + value + '&lt;/span&gt;';
   * }
   * </pre>
   * </li>
   * <li><code>colIndex:Number</code> —
   * The index of the current column
   * </li>
   * <li><code>store:ext.data.Store</code> —
   * The data store
   * <pre>
   * editRenderer: function (value, metaData, record, rowIndex, colIndex, store) {
   *     // style the cell differently depending on how the value relates to the
   *     // average of all values
   *     var average = store.average('grades'),
   *         status = (value &lt; average) ? 'needsImprovement' : 'satisfactory';
   *     return '&lt;span class="' + status + '"&gt;' + value + '&lt;/span&gt;';
   * }
   * </pre>
   * </li>
   * <li><code>view:ext.view.DataView</code> —
   * The data view
   * <pre>
   * editRenderer: function (value, metaData, record, rowIndex, colIndex, store, view) {
   *     // style the value using the dataIndex of the column
   *     var headerCt = this.getHeaderContainer(),
   *         column = headerCt.getHeaderAtIndex(colIndex);
   *
   *     return '&lt;span class="app-' + column.dataIndex + '"&gt;' + value + '&lt;/span&gt;';
   * }
   * </pre>
   * </li>
   * <li><code>null:String</code> —
   * The HTML string to be rendered.
   * </li>
   * </ul>
   * @see ext.grid.plugin.RowEditingPlugin
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#cfg-editRenderer Original Ext JS documentation of 'editRenderer'
   */
  public native function get editRenderer():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set editRenderer(value:*):void;

  [ExtConfig]
  /**
   * ARIA label for screen readers to announce for the check column's header when it is focused.
   * Note that this label will not be visible on screen.
   * @default 'Row selector'
   * @since 6.2.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#cfg-headerAriaLabel Original Ext JS documentation of 'headerAriaLabel'
   */
  public native function get headerAriaLabel():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set headerAriaLabel(value:String):void;

  [ExtConfig]
  /**
   * ARIA description text to announce for the check column's header when it is focused,
   * →<code>showHeaderCheckbox</code> is shown, and all rows are selected.
   * @default 'Press Space to deselect all rows'
   * @see #showHeaderCheckbox
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#cfg-headerDeselectText Original Ext JS documentation of 'headerDeselectText'
   */
  public native function get headerDeselectText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set headerDeselectText(value:String):void;

  [ExtConfig]
  /**
   * ARIA description text to announce for the check column's header when it is focused,
   * →<code>showHeaderCheckbox</code> is shown, and not all rows are selected.
   * @default 'Press Space to select all rows'
   * @since 6.2.0
   * @see #showHeaderCheckbox
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#cfg-headerSelectText Original Ext JS documentation of 'headerSelectText'
   */
  public native function get headerSelectText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set headerSelectText(value:String):void;

  [ExtConfig]
  /**
   * Displays the configured text in the check column's header.
   * <p>if →<code>showHeaderCheckbox</code> is <code>true</code>, the text is shown <i>above</i> the checkbox.</p>
   * @default undefined
   * @since 6.0.1
   * @see #showHeaderCheckbox
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#cfg-headerText Original Ext JS documentation of 'headerText'
   */
  public native function get headerText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set headerText(value:String):void;

  [ExtConfig]
  /**
   * The index at which to insert the checkbox column.
   * Supported values are a numeric index, and the strings 'first' and 'last'.
   * @default 0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#cfg-injectCheckbox Original Ext JS documentation of 'injectCheckbox'
   */
  public native function get injectCheckbox():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set injectCheckbox(value:*):void;

  [ExtConfig]
  /**
   * Modes of selection.
   * Valid values are <code>"SINGLE"</code>, <code>"SIMPLE"</code>, and <code>"MULTI"</code>.
   * @default 'MULTI'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#cfg-mode Original Ext JS documentation of 'mode'
   */
  override public native function get mode():String;

  [ExtConfig]
  /**
   * @private
   */
  override public native function set mode(value:String):void;

  [ExtConfig]
  /**
   * A renderer is an 'interceptor' method which can be used to transform data (value,
   * appearance, etc.) before it is rendered. Example:
   * <p><b>NOTE:</b> In previous releases, a string was treated as a method on
   * <code>→ext.util.Format</code> but that is now handled by the →<code>formatter</code> config.</p>
   * @default false
   * <ul>
   * <li><code>value:Object</code> —
   * The data value for the current cell
   * <pre>
   * renderer: function(value){
   *     // evaluates `value` to append either `person' or `people`
   *     return Ext.util.Format.plural(value, 'person', 'people');
   * }
   * </pre>
   * </li>
   * <li><code>metaData:Object</code> —
   * A collection of metadata about the current cell; can be
   * used or modified by the renderer. Recognized properties are: <code>tdCls</code>, <code>tdAttr</code>,
   * and <code>tdStyle</code>.
   * <p>To add style attributes to the <code>&amp;lt;td&gt;</code> element, you must use the <code>tdStyle</code>
   * property. Using a style attribute in the <code>tdAttr</code> property will override the
   * styles the column sets, such as the width which will break the rendering.</p>
   * <p>You can see an example of using the metaData parameter below.</p>
   * <pre>
   *  Ext.create('Ext.data.Store', {
   *       storeId: 'simpsonsStore',
   *       fields: ['class', 'attr', 'style'],
   *       data: {
   *           'class': 'red-bg',
   *           'attr': 'lightyellow',
   *           'style': 'red'
   *       }
   *  });
   *
   *  Ext.create('Ext.grid.Panel', {
   *       title: 'Simpsons',
   *       store: Ext.data.StoreManager.lookup('simpsonsStore'),
   *       columns: [{
   *           text: 'Name',
   *           dataIndex: 'class',
   *           renderer: function (value, metaData) {
   *               metaData.tdCls = value;
   *               return value;
   *           }
   *       }, {
   *           text: 'Email',
   *           dataIndex: 'attr',
   *           flex: 1,
   *           renderer: function (value, metaData) {
   *               metaData.tdAttr = 'bgcolor="' + value + '"';
   *               return value;
   *           }
   *       }, {
   *           text: 'Phone',
   *           dataIndex: 'style',
   *           renderer: function (value, metaData) {
   *               metaData.tdStyle = 'color:' + value;
   *               return value;
   *           }
   *       }],
   *       height: 200,
   *       width: 400,
   *       renderTo: Ext.getBody()
   *   });
   * </pre>
   * </li>
   * <li><code>record:ext.data.Model</code> —
   * The record for the current row
   * <pre>
   * renderer: function (value, metaData, record) {
   *     // evaluate the record's `updated` field and if truthy return the value
   *     // from the `newVal` field, else return value
   *     var updated = record.get('updated');
   *     return updated ? record.get('newVal') : value;
   * }
   * </pre>
   * </li>
   * <li><code>rowIndex:Number</code> —
   * The index of the current row
   * <pre>
   * renderer: function (value, metaData, record, rowIndex) {
   *     // style the cell differently for even / odd values
   *     var odd = (rowIndex % 2 === 0);
   *     metaData.tdStyle = 'color:' + (odd ? 'gray' : 'red');
   * }
   * </pre>
   * </li>
   * <li><code>colIndex:Number</code> —
   * The index of the current column
   * <pre>
   * var myRenderer = function(value, metaData, record, rowIndex, colIndex) {
   *     if (colIndex === 0) {
   *         metaData.tdAttr = 'data-qtip=' + value;
   *     }
   *     // additional logic to apply to values in all columns
   *     return value;
   * }
   *
   * // using the same renderer on all columns you can process the value for
   * // each column with the same logic and only set a tooltip on the first column
   * renderer: myRenderer
   * </pre>
   * <p><i>See also →<code>ext.tip.QuickTipManager</code></i></p>
   * </li>
   * <li><code>store:ext.data.Store</code> —
   * The data store
   * <pre>
   * renderer: function (value, metaData, record, rowIndex, colIndex, store) {
   *     // style the cell differently depending on how the value relates to the
   *     // average of all values
   *     var average = store.average('grades');
   *     metaData.tdCls = (value &lt; average) ? 'needsImprovement' : 'satisfactory';
   *     return value;
   * }
   * </pre>
   * </li>
   * <li><code>view:ext.view.DataView</code> —
   * The data view
   * <pre>
   * renderer: function (value, metaData, record, rowIndex, colIndex, store, view) {
   *     // style the cell using the dataIndex of the column
   *     var headerCt = this.getHeaderContainer(),
   *         column = headerCt.getHeaderAtIndex(colIndex);
   *
   *     metaData.tdCls = 'app-' + column.dataIndex;
   *     return value;
   * }
   * </pre>
   * </li>
   * <li><code>null:String</code> —
   * The HTML string to be rendered.
   * </li>
   * </ul>
   * @see ext.util.#Format ext.util.Format
   * @see #formatter
   * @see ext.tip.#QuickTipManager ext.tip.QuickTipManager
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#cfg-renderer Original Ext JS documentation of 'renderer'
   */
  public native function get renderer():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set renderer(value:*):void;

  [ExtConfig]
  /**
   * ARIA description text to announce when check column cell is focused and the row
   * is selected.
   * @default 'Press Space to deselect this row'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#cfg-rowDeselectText Original Ext JS documentation of 'rowDeselectText'
   */
  public native function get rowDeselectText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set rowDeselectText(value:String):void;

  [ExtConfig]
  /**
   * ARIA description text to announce when check column cell is focused and the row
   * is not selected.
   * @default 'Press Space to select this row'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#cfg-rowSelectText Original Ext JS documentation of 'rowSelectText'
   */
  public native function get rowSelectText():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set rowSelectText(value:String):void;

  [ExtConfig]
  /**
   * Configure as <code>false</code> to not display the header checkbox at the top of the column.
   * When the store is a →<code>ext.data.BufferedStore</code>, this configuration will
   * not be available because the buffered data set does not always contain all data.
   * @default false
   * @see ext.data.BufferedStore
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#cfg-showHeaderCheckbox Original Ext JS documentation of 'showHeaderCheckbox'
   */
  public native function get showHeaderCheckbox():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set showHeaderCheckbox(value:Boolean):void;

  /**
   * Retrieve a configuration to be used in a HeaderContainer.
   * This is called when injectCheckbox is not <code>false</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.selection.CheckboxModel.html#method-getHeaderConfig Original Ext JS documentation of 'getHeaderConfig'
   */
  public native function getHeaderConfig():Column;

  /**
   * Toggle between selecting all and deselecting all when clicking on
   * a checkbox header.
   */
  override protected native function onHeaderClick():void;
}
}