package ext.state {

[Mixin("ext.state.Stateful")]
/**
 * A mixin for being able to save the state of an object to an underlying
 * →<code>ext.state.Provider</code>.
 * @see ext.state.Provider
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.state.Stateful.html Original Ext JS documentation of 'Ext.state.Stateful'
 * @see ext.state.Stateful
 */
public interface IStateful {
  [ExtConfig] /**
   * A buffer to be applied if many state events are fired within a short period.
   * @default 100
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.state.Stateful.html#cfg-saveDelay Original Ext JS documentation of 'saveDelay'
   */
  function get saveDelay():Number;

  [ExtConfig] function set saveDelay(value:Number):void;

  [ArrayElementType("String")]
  [ExtConfig] /**
   * An array of events that, when fired, should trigger this object to
   * save its state. <code>→stateEvents</code> defaults to the <code>→stateEvents</code> associated with the
   * component you are using. Any events you statically set will be appended to that list.
   * <code>→stateEvents</code> may be any type of event supported by this object, including
   * browser or custom events (e.g., <code>['click', 'customerchange']</code>).
   * <p>See <code>→stateful</code> for an explanation of saving and
   * restoring object state.</p>
   * @see #stateEvents
   * @see #stateful
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.state.Stateful.html#cfg-stateEvents Original Ext JS documentation of 'stateEvents'
   */
  function get stateEvents():Array;

  [ExtConfig] function set stateEvents(value:Array):void;

  [ExtConfig] /**
   * The unique id for this object to use for state management purposes.
   * <p>See →<code>stateful</code> for an explanation of saving and restoring state.</p>
   * @see #stateful
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.state.Stateful.html#cfg-stateId Original Ext JS documentation of 'stateId'
   */
  function get stateId():String;

  [ExtConfig] function set stateId(value:String):void;

  [ExtConfig]
  [Bindable] /**
   * A flag which causes the object to attempt to restore the state of
   * internal properties from a saved state on startup. The object must have
   * a →<code>stateId</code> for state to be managed.
   * <p>Auto-generated ids are not guaranteed to be stable across page loads and
   * cannot be relied upon to save and restore the same state for a object.</p>
   * <p>For state saving to work, the state manager's provider must have been
   * set to an implementation of →<code>ext.state.Provider</code> which overrides the
   * →<code>ext.state.Provider.set()</code> and →<code>ext.state.Provider.get()</code>
   * methods to save and recall name/value pairs. A built-in implementation,
   * →<code>ext.state.CookieProvider</code> is available.</p>
   * <p>To set the state provider for the current page:</p>
   * <pre>
   * Ext.state.Manager.setProvider(new Ext.state.CookieProvider({
   *     expires: new Date(new Date().getTime()+(1000&#42;60&#42;60&#42;24&#42;7)), // 7 days from now
   * }));
   * </pre>
   * <p>A stateful object attempts to save state when one of the events
   * listed in the →<code>stateEvents</code> configuration fires.</p>
   * <p>To save state, a stateful object first serializes its state by
   * calling <i>→<code>getState()</code></i>.</p>
   * <p>The Component base class implements →<code>getState()</code> to save its width and
   * height within the state only if they were initially configured, and have
   * changed from the configured value.</p>
   * <p>The Panel class saves its collapsed state in addition to that.</p>
   * <p>The Grid class saves its column state and store state (sorters and filters and grouper)
   * in addition to its superclass state.</p>
   * <p>If there is more application state to be save, the developer must provide
   * an implementation which first calls the superclass method to inherit the above behaviour,
   * and then injects new properties into the returned object.</p>
   * <p>The value yielded by <code>→getState()</code> is passed to →<code>ext.state.StateManager.set()</code>
   * which uses the configured →<code>ext.state.Provider</code> to save the object
   * keyed by the →<code>stateId</code>.</p>
   * <p>During construction, a stateful object attempts to <i>restore</i> its state by calling
   * →<code>ext.state.StateManager.get()</code> passing the →<code>stateId</code></p>
   * <p>The resulting object is passed to →<code>applyState()</code>&#42;. The default implementation of
   * →<code>applyState()</code> simply copies properties into the object, but a developer may
   * override this to support restoration of more complex application state.</p>
   * <p>You can perform extra processing on state save and restore by attaching
   * handlers to the →<code>event:onBeforeStateRestore</code>, →<code>event:onStateRestore</code>,
   * →<code>event:onBeforeStateSave</code> and →<code>event:onStateSave</code> events. In some simple cases,
   * passing an object for the <code>→stateful</code> config may suffice. If an object is
   * provided, the properties of that object are used to include or exclude stateful
   * properties returned by <code>→getState()</code>. For example:</p>
   * <pre>
   *  stateful: {
   *      height: false, // never persist the height
   *      width: true    // always persist the width
   *  }
   * </pre>
   * <p>The above is roughly equivalent to the following:</p>
   * <pre>
   *  getState: function () {
   *      var state = this.callParent();
   *
   *      delete state.height;
   *      state.width = this.width;
   *
   *      return state;
   *  }
   * </pre>
   * @default false
   * @see #stateId
   * @see ext.state.Provider
   * @see ext.state.Provider#set()
   * @see ext.state.Provider#get()
   * @see ext.state.CookieProvider
   * @see #stateEvents
   * @see #getState()
   * @see ext.state.SStateManager#set() ext.state.StateManager.set()
   * @see ext.state.SStateManager#get() ext.state.StateManager.get()
   * @see #applyState()
   * @see #event:onBeforeStateRestore
   * @see #event:onStateRestore
   * @see #event:onBeforeStateSave
   * @see #event:onStateSave
   * @see #stateful
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.state.Stateful.html#cfg-stateful Original Ext JS documentation of 'stateful'
   * @see #getStateful()
   * @see #setStateful()
   */
  function get stateful():*;

  [ExtConfig]
  [Bindable] function set stateful(value:*):void;

  /**
   * Add events that will trigger the state to be saved. If the first argument is an
   * array, each element of that array is the name of a state event. Otherwise, each
   * argument passed to this method is the name of a state event.
   * @param events The event name or an array of event names.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.state.Stateful.html#method-addStateEvents Original Ext JS documentation of 'addStateEvents'
   */
  function addStateEvents(events:*):void;

  /**
   * Applies the state to the object. This should be overridden in subclasses to do
   * more complex state operations. By default it applies the state properties onto
   * the current object.
   * @param state The state
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.state.Stateful.html#method-applyState Original Ext JS documentation of 'applyState'
   */
  function applyState(state:Object):void;

  /**
   * Destroys this stateful object.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.state.Stateful.html#method-destroy Original Ext JS documentation of 'destroy'
   */
  function destroy(...params):void;

  /**
   * Gets the current state of the object. By default this function returns null,
   * it should be overridden in subclasses to implement methods for getting the state.
   * @return The current state
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.state.Stateful.html#method-getState Original Ext JS documentation of 'getState'
   */
  function getState():Object;

  /**
   * Returns the value of <code>stateful</code>.
   * @see #stateful
   */
  function getStateful():*;

  /**
   * Conditionally saves a single property from this object to the given state object.
   * The idea is to only save state which has changed from the initial state so that
   * current software settings do not override future software settings. Only those
   * values that are user-changed state should be saved.
   * @param propName The name of the property to save.
   * @param state The state object in to which to save the property.
   * @param stateName The name to use for the property in state.
   * @return True if the property was saved, false if not.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.state.Stateful.html#method-savePropToState Original Ext JS documentation of 'savePropToState'
   */
  function savePropToState(propName:String, state:Object, stateName:String = null):Boolean;

  /**
   * Gathers additional named properties of the instance and adds their current values
   * to the passed state object.
   * @param propNames The name (or array of names) of the property to save.
   * @param state The state object in to which to save the property values.
   * @return state
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.state.Stateful.html#method-savePropsToState Original Ext JS documentation of 'savePropsToState'
   */
  function savePropsToState(propNames:*, state:Object):Object;

  /**
   * Saves the state of the object to the persistence store.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.state.Stateful.html#method-saveState Original Ext JS documentation of 'saveState'
   */
  function saveState():void;

  /**
   * Sets the value of <code>stateful</code>.
   * @param stateful The new value.
   * @see #stateful
   */
  function setStateful(stateful:*):void;
}
}