package ext.system {
import ext.ArrayUtil;
import ext.Deferred;
import ext.Ext;
import ext.IPromise;
import ext.Loader;
import ext.Promise;

import joo.localeSupport;

import js.XMLHttpRequest;

[Rename("Ext.system.PackageEntry")]
internal class PackageEntry {

  private static const PACKAGE_JSON:String = "/package.json";
  private static const PACKAGE_PREFIX:String = "package.";

  private var _name:String;
  private var _manifestPromise:IPromise;
  private var _transitiveScriptsPromise:IPromise;

  public function PackageEntry(name:String) {
    _name = name;
  }

  internal function get name():String {
    return _name;
  }

  internal function get manifest():Object {
    return Ext.manifest.packages[name];
  }

  internal function loadManifest():IPromise {
    if (!_manifestPromise) {
      if (manifest) {
        manifest.name = name; // may be missing in pre-loaded packages!
        _manifestPromise = Deferred.resolved(manifest);
      } else {
        _manifestPromise = doLoadPackageJson("packages/" + _name);
      }
    }
    return _manifestPromise;
  }

  private function doLoadPackageJson(packageDir:String):IPromise {
    return PackageLoader.loadJson(packageDir + PACKAGE_JSON)
            .then(function (json:Object):Object {
                      var manifest:Object = Ext.manifest.packages[name] = json;
                      manifest.dir = packageDir;
                      manifest.locale = localeSupport.getLocale();
                      manifest.loaded = false;
                      return manifest;
                    },
                    function (response:XMLHttpRequest):void {
                      throw new Error("Failed to load package " + name + ".");
                    }
            );
  }

  private static function selectName(dependency:Object):String {
    return dependency.name || (typeof dependency === "string" ? String(dependency) : null);
  }

  internal function get dependencies():Array {
    return (name !== "core" && manifest.requires || []).map(selectName).filter(function(dependency:String):Boolean {
      return !!dependency;
    });
  }

  internal function loadScripts():IPromise {
    if (!_transitiveScriptsPromise) {
      _transitiveScriptsPromise = loadManifest().then(function (manifest:Object):Object {
        if (manifest && manifest.loaded === false) {
          return Promise.all(dependencies.map(PackageLoader.loadScripts))
                  .then(loadJsAndCss)
                  .then(loadClasspath)
                  .then(loadOverrides)
                  .then(allScriptsLoaded, allScriptsLoaded); // ignore errors when loading overrides!
        }
        return null;
      });
    }
    return _transitiveScriptsPromise;
  }

  private function allScriptsLoaded():void {
    manifest.loaded = true;
  }

  [ArrayElementType("String")]
  private static function splitAtComma(str:String):Array {
    return str.split(",");
  }

  [ArrayElementType("String")]
  private function convertToUrlArray(path:Array):Array {
    return ArrayUtil.flatten((path || []).map(PlaceholderReplacer.substitutePlaceholders(manifest, PACKAGE_PREFIX)).map(splitAtComma));
  }

  private function loadJsAndCss():IPromise {
    return loadManifest().then(function(manifest:Object):IPromise {

      // JavaScript descriptors where bundled scripts are loaded after non-bundled ones.
      var jsDescriptors:Array = ((manifest.development || {})['js'] || []).concat(manifest.js || []);
      // Likewise for CSS.
      var cssDescriptors:Array = ((manifest.development || {})['css'] || []).concat(manifest.css || []);

      var descriptors:Array = jsDescriptors.concat(cssDescriptors);

      var urls:Array = descriptors
              .map(function (jsDescriptor:Object):String {
                return manifest.dir + "/" + jsDescriptor.path;
              });
      return loadScriptsSequentially(urls, false);
    });
  }

  private function loadClasspath():IPromise {
    var urls:Array = convertToUrlArray(manifest.classpath);
    return Promise.all(urls.map(loadPackageDefines)).then(function():IPromise {
      return loadScriptsSequentially(urls, false);
    });
  }

  private function loadPackageDefines(jsUrl:String):IPromise {
    var jsonUrl:String = jsUrl.replace(/\.js$/, ".json");
    return PackageLoader.loadJson(jsonUrl).then(function (definesList:Array):void {
              definesList.forEach(function (define:String):void {
                Loader.setPath(define, jsUrl);
              });
            }, Ext.emptyFn // ignore missing defines lists (module without TOC)
    );
  }

  private static const OVERRIDES_FILTER_PATTERN:RegExp = new RegExp("overrides(-" + localeSupport.getDefaultLocale() + ")?\\.js$");

  //additional loading of overrides
  private function loadOverrides():IPromise {
    var urls:Array = convertToUrlArray(manifest.overrides);
    urls = urls.filter(function(url:String):Boolean {
      return !url.match(OVERRIDES_FILTER_PATTERN);
    });
    return loadScriptsSequentially(urls, true);
  }

  private static function loadScriptsSequentially(urls:Array, ignoreErrors:Boolean):IPromise {
    var result:IPromise = Promise.resolve(true);
    urls.forEach(function(url:String):void {
      function next():IPromise {
        return loadScript(url);
      }
      result = result.then(next, ignoreErrors ? next : null);
    });
    return result;
  }

  private static function loadScript(url:String):IPromise {
    var result:Deferred = new Deferred();
    Loader.loadScript({
      url: url,
      onLoad: result.resolve,
      onError: result.reject
    });
    return result;
  }

}
}
