package ext.system {
import ext.Ajax;
import ext.ArrayUtil;
import ext.Ext;
import ext.IPromise;
import ext.JSON;
import ext.Promise;

import js.XMLHttpRequest;

[Rename("Ext.system.PackageLoader")]
public class PackageLoader {

  private static var _packageListPromise:IPromise;

  private static const PACKAGE_ENTRIES_BY_NAME:Object = {};

  internal static function getPackageEntry(name:String):PackageEntry {
    var packageEntry:PackageEntry = PACKAGE_ENTRIES_BY_NAME[name];
    if (!packageEntry) {
      packageEntry = PACKAGE_ENTRIES_BY_NAME[name] = new PackageEntry(name);
    }
    return packageEntry;
  }

  private static const DYNAMIC_PACKAGES_JSON:String = "dynamic-packages.json";
  private static const ADDITIONAL_PACKAGES_URL_PATH:String = "additional-packages/";

  public static function loadPackageList(localPackagesOnly:Boolean = false):IPromise {
    if (!_packageListPromise) {
      _packageListPromise = (localPackagesOnly
              ? loadJson(DYNAMIC_PACKAGES_JSON + "?local-only", [])
              : Promise.all([
                loadJson(DYNAMIC_PACKAGES_JSON, []),
                loadJson(ADDITIONAL_PACKAGES_URL_PATH, [])
                    .then(directoryIndexToStringArray)
                    .then(loadFromDirectories)
                    .then(uniteAll)
              ]).then(uniteAll)
      ).then(concatStaticAndDynamicPackages);
    }
    return _packageListPromise;
  }

  private static function loadFromDirectories(directories: Array/*<String>*/): IPromise/*<String[]>*/ {
    return Promise.all(directories.map(function(directory: String): IPromise {
      return loadJson(ADDITIONAL_PACKAGES_URL_PATH + directory +"/", [])
          .then(directoryIndexToStringArray)
    }))
  }

  [ArrayElementType("String")]
  private static function concatStaticAndDynamicPackages(dynamicPackages:Array):Array {
    // start with statically linked packages:
    var packages:Array = Object.keys(Ext.manifest.packages);
    if (Ext.isArray(dynamicPackages)) {
      // add dynamic packages, avoiding duplicates:
      return ArrayUtil.union(packages, dynamicPackages);
    }
    return packages;
  }

  [ArrayElementType("String")]
  private static function directoryIndexToStringArray(directoryContents: Array): Array {
    return !Ext.isArray(directoryContents) ? []
        : directoryContents.filter(function(rec: Object): Boolean {
          return !!rec && !!rec.name && rec.type === "directory";
        }).map(function(rec: Object): String {
          return rec.name;
        });
  }

  private static function uniteAll(packagesArrays: Array): Array {
    return ArrayUtil.union.apply(ArrayUtil, packagesArrays);
  }

  internal static function loadJson(url:String, valueOnError: Object = null):IPromise {
    return Ajax.request({
      url: url,
      method: 'GET',
      headers: {
        'Accept': "application/json, application/octet-stream"
      }
    }).then(toJson, valueOnError ? function(): Object {
      return valueOnError;
    } : null);
  }

  private static function toJson(response:XMLHttpRequest):Object {
    var jsonString:String = response.responseText;
    // remove illegal comments:
    jsonString = jsonString.replace(/\/\*(.|\s)*?\*\//g, "");
    return JSON.decode(jsonString);
  }

  public static function loadManifest(packageName:String):IPromise {
    return getPackageEntry(packageName).loadManifest();
  }

  public static function getLoadedManifest(name:String):Object {
    return Ext.manifest.packages[name];
  }

  public static function loadManifests(packageNames:Array):IPromise {
    return Promise.all(packageNames.map(loadManifest));
  }

  public static function loadTransitiveManifests(packageNames:Array):IPromise {
    return loadManifests(packageNames).then(function(manifests:Array):* {
      var morePackageNames:Array = manifests.reduce(function(accumulator:Array, manifest:Object):Array {
        var dependencies:Array = getPackageEntry(String(manifest.name)).dependencies;
        return ArrayUtil.union(accumulator, dependencies);
      }, packageNames);
      return morePackageNames.length > packageNames.length
              ? loadTransitiveManifests(morePackageNames)
              : loadManifests(morePackageNames);
    });
  }

  public static function loadScripts(packageName:String):IPromise {
    return getPackageEntry(packageName).loadScripts();
  }

  public static function sortByDependencies(packageNames:Array):IPromise {
    return loadTransitiveManifests(packageNames).then(function():Array {

      function isInScope(packageName:String):Boolean {
        return packageNames.indexOf(packageName) !== -1;
      }

      var sortedPackages:Array = [];
      function isSorted(packageName:String):Boolean {
        return sortedPackages.indexOf(packageName) !== -1;
      }

      while (sortedPackages.length < packageNames.length) {
        // find all packages that are not yet sorted, but all of their dependencies are:
        var packagesWithSortedDependencies:Array = packageNames.filter(function(packageName:String):Boolean {
          return !isSorted(packageName) && // package is still unsorted and...
                  getPackageEntry(packageName).dependencies.every(function(dependency:String):Boolean {
                    // ...all of its dependencies are either out-of-scope or already sorted:
                    return !isInScope(dependency) || isSorted(dependency);
                  });
        });

        if (packagesWithSortedDependencies.length === 0) {
          throw new Error("Inconsistent dependency graph - aborting package loading.");
        }

        // These packages have no dependencies on each other, but to end up with a reproducible load order, we sort
        // them alphabetically:
        packagesWithSortedDependencies.sort();
        sortedPackages = sortedPackages.concat(packagesWithSortedDependencies);
      }
      return sortedPackages;
    });
  }

  patchExtGetResourcePath();

  private static function patchExtGetResourcePath():void {
    var originalGetResourcePath:Function = Ext.getResourcePath;
    Ext.getResourcePath = function (path:*, poolName:String, packageName:String):String {
      if (typeof path !== 'string') {
        poolName = path.pool;
        packageName = path.packageName;
        path = path.path;
      }
      if (!poolName && "loaded" in Ext.manifest.packages[packageName]) {
        // use different resource path for dynamically loaded packages:
        return "packages/" + packageName + "/resources/" + path;
      }
      return originalGetResourcePath.apply(this, arguments);
    };
  }

}
}
