package ext.util {
import ext.Base;
import ext.dom.Element;

[Native("Ext.util.Animate", require)]
/**
 * This animation class is a mixin.
 * <p>Ext.util.Animate provides an API for the creation of animated transitions of properties
 * and styles. This class is used as a mixin and currently applied to →<code>ext.dom.Element</code>,
 * →<code>ext.dom.CompositeElement</code>, →<code>ext.draw.sprite.Sprite</code>, →<code>ext.draw.sprite.CompositeSprite</code>,
 * and →<code>ext.Component</code>. Note that Components have a limited subset of what attributes
 * can be animated such as top, left, x, y, height, width, and opacity (color, paddings,
 * and margins can not be animated).</p>
 * <p><b><i>Animation Basics</i></b></p>
 * <p>All animations require three things - <code>easing</code>, <code>duration</code>, and <code>to</code> (the final end value
 * for each property) you wish to animate. Easing and duration are defaulted values specified below.
 * Easing describes how the intermediate values used during a transition will be calculated.
 * <i>Easing</i> (→<code>ext.fx.Anim.easing</code>) allows for a transition to change speed over its duration.
 * You may use the defaults for easing and duration, but you must always set a
 * →<code>ext.fx.Anim.to</code> property which is the end value for all animations.</p>
 * <p>Popular element 'to' configurations are:</p>
 * <ul>
 * <li>opacity</li>
 * <li>x</li>
 * <li>y</li>
 * <li>color</li>
 * <li>height</li>
 * <li>width</li>
 * </ul>
 * <p>Popular sprite 'to' configurations are:</p>
 * <ul>
 * <li>translation</li>
 * <li>path</li>
 * <li>scale</li>
 * <li>stroke</li>
 * <li>rotation</li>
 * </ul>
 * <p>The default duration for animations is 250 (which is a 1/4 of a second). Duration is denoted in
 * milliseconds. Therefore 1 second is 1000, 1 minute would be 60000, and so on. The default
 * easing curve used for all animations is 'ease'. Popular easing functions are included
 * and can be found in <i>Easing</i> (→<code>ext.fx.Anim.easing</code>).</p>
 * <p>For example, a simple animation to fade out an element with a default easing and duration:</p>
 * <pre>
 * var p1 = Ext.get('myElementId');
 *
 * p1.animate({
 *     to: {
 *         opacity: 0
 *     }
 * });
 * </pre>
 * <p>To make this animation fade out in a tenth of a second:</p>
 * <pre>
 * var p1 = Ext.get('myElementId');
 *
 * p1.animate({
 *    duration: 100,
 *     to: {
 *         opacity: 0
 *     }
 * });
 * </pre>
 * <p><b><i>Animation Queues</i></b></p>
 * <p>By default all animations are added to a queue which allows for animation via a chain-style API.
 * For example, the following code will queue 4 animations which occur sequentially (one right
 * after the other):</p>
 * <pre>
 * p1.animate({
 *     to: {
 *         x: 500
 *     }
 * }).animate({
 *     to: {
 *         y: 150
 *     }
 * }).animate({
 *     to: {
 *         backgroundColor: '#f00'  //red
 *     }
 * }).animate({
 *     to: {
 *         opacity: 0
 *     }
 * });
 * </pre>
 * <p>You can change this behavior by calling the →<code>ext.util.Animate.syncFx()</code> method and all
 * subsequent animations for the specified target will be run concurrently (at the same time).</p>
 * <pre>
 * p1.syncFx();  //this will make all animations run at the same time
 *
 * p1.animate({
 *     to: {
 *         x: 500
 *     }
 * }).animate({
 *     to: {
 *         y: 150
 *     }
 * }).animate({
 *     to: {
 *         backgroundColor: '#f00'  //red
 *     }
 * }).animate({
 *     to: {
 *         opacity: 0
 *     }
 * });
 * </pre>
 * <p>This works the same as:</p>
 * <pre>
 * p1.animate({
 *     to: {
 *         x: 500,
 *         y: 150,
 *         backgroundColor: '#f00'  //red
 *         opacity: 0
 *     }
 * });
 * </pre>
 * <p>The →<code>ext.util.Animate.stopAnimation()</code> method can be used to stop any
 * currently running animations and clear any queued animations.</p>
 * <p><b><i>Animation Keyframes</i></b></p>
 * <p>You can also set up complex animations with →<code>ext.fx.Anim.keyframes</code> which follow
 * the CSS3 Animation configuration pattern. Note rotation, translation, and scaling can only
 * be done for sprites. The previous example can be written with the following syntax:</p>
 * <pre>
 * p1.animate({
 *     duration: 1000,  //one second total
 *     keyframes: {
 *         25: {     //from 0 to 250ms (25%)
 *             x: 0
 *         },
 *         50: {   //from 250ms to 500ms (50%)
 *             y: 0
 *         },
 *         75: {  //from 500ms to 750ms (75%)
 *             backgroundColor: '#f00'  //red
 *         },
 *         100: {  //from 750ms to 1sec
 *             opacity: 0
 *         }
 *     }
 * });
 * </pre>
 * <p><b><i>Animation Events</i></b></p>
 * <p>Each animation you create has events for →<code>ext.fx.Anim.event:onBeforeAnimate</code>,
 * →<code>ext.fx.Anim.event:onAfterAnimate</code>, and →<code>ext.fx.Anim.event:onLastFrame</code>.
 * Keyframed animations adds an additional →<code>ext.fx.Animator.event:onKeyframe</code> event which
 * fires for each keyframe in your animation.</p>
 * <p>All animations support the →<code>ext.util.Observable.listeners</code> configuration
 * to attact functions to these events.</p>
 * <pre>
 * startAnimate: function() {
 *     var p1 = Ext.get('myElementId');
 *     p1.animate({
 *        duration: 100,
 *         to: {
 *             opacity: 0
 *         },
 *         listeners: {
 *             beforeanimate:  function() {
 *                 // Execute my custom method before the animation
 *                 this.myBeforeAnimateFn();
 *             },
 *             afteranimate: function() {
 *                 // Execute my custom method after the animation
 *                 this.myAfterAnimateFn();
 *             },
 *             scope: this
 *     });
 * },
 * myBeforeAnimateFn: function() {
 *   // My custom logic
 * },
 * myAfterAnimateFn: function() {
 *   // My custom logic
 * }
 * </pre>
 * <p>Due to the fact that animations run asynchronously, you can determine if an animation
 * is currently running on any target by using the
 * →<code>ext.util.Animate.getActiveAnimation()</code> method. This method will return
 * false if there are no active animations or return the currently running →<code>ext.fx.Anim</code>
 * instance.</p>
 * <p>In this example, we're going to wait for the current animation to finish, then stop any other
 * queued animations before we fade our element's opacity to 0:</p>
 * <pre>
 * var curAnim = p1.getActiveAnimation();
 * if (curAnim) {
 *     curAnim.on('afteranimate', function() {
 *         p1.stopAnimation();
 *         p1.animate({
 *             to: {
 *                 opacity: 0
 *             }
 *         });
 *     });
 * }
 * </pre>
 * @see ext.dom.Element
 * @see ext.dom.CompositeElement
 * @see ext.draw.sprite.Sprite
 * @see ext.draw.sprite.CompositeSprite
 * @see ext.Component
 * @see ext.fx.Anim#easing
 * @see ext.fx.Anim#to
 * @see ext.util.Animate#syncFx()
 * @see ext.util.Animate#stopAnimation()
 * @see ext.fx.Anim#keyframes
 * @see ext.fx.Anim#event:onBeforeAnimate
 * @see ext.fx.Anim#event:onAfterAnimate
 * @see ext.fx.Anim#event:onLastFrame
 * @see ext.fx.Animator#event:onKeyframe
 * @see ext.util.Observable#listeners
 * @see ext.util.Animate#getActiveAnimation()
 * @see ext.fx.Anim
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Animate.html Original Ext JS documentation of 'Ext.util.Animate'
 */
public class Animate extends Base implements IAnimate {
  /**
   * Performs custom animation on this object.
   * <p>This method is applicable to both the →<code>ext.Component</code> class and the
   * →<code>ext.draw.sprite.Sprite</code> class. It performs animated transitions of certain
   * properties of this object over a specified timeline.</p>
   * <p><i>Animating a Component (→<code>ext.Component</code>)</i></p>
   * <p>When animating a Component, the following properties may be specified in <code>from</code>, <code>to</code>,
   * and <code>keyframe</code> objects:</p>
   * <ul>
   * <li>
   * <p><code>x</code> - The Component's page X position in pixels.</p>
   * </li>
   * <li>
   * <p><code>y</code> - The Component's page Y position in pixels</p>
   * </li>
   * <li>
   * <p><code>left</code> - The Component's <code>left</code> value in pixels.</p>
   * </li>
   * <li>
   * <p><code>top</code> - The Component's <code>top</code> value in pixels.</p>
   * </li>
   * <li>
   * <p><code>width</code> - The Component's <code>width</code> value in pixels.</p>
   * </li>
   * <li>
   * <p><code>height</code> - The Component's <code>height</code> value in pixels.</p>
   * </li>
   * </ul>
   * <p>The following property may be set on the animation config root:</p>
   * <ul>
   * <li><code>dynamic</code> - Specify as true to update the Component's layout (if it is a Container)
   * at every frame of the animation. <i>Use sparingly as laying out on every intermediate
   * size change is an expensive operation.</i></li>
   * </ul>
   * <p>For example, to animate a Window to a new size, ensuring that its internal layout
   * and any shadow is correct:</p>
   * <pre>
   * myWindow = Ext.create('Ext.window.Window', {
   *     title: 'Test Component animation',
   *     width: 500,
   *     height: 300,
   *     layout: {
   *         type: 'hbox',
   *         align: 'stretch'
   *     },
   *     items: [{
   *         title: 'Left: 33%',
   *         margin: '5 0 5 5',
   *         flex: 1
   *     }, {
   *         title: 'Left: 66%',
   *         margin: '5 5 5 5',
   *         flex: 2
   *     }]
   * });
   * myWindow.show();
   * myWindow.header.el.on('click', function() {
   *     myWindow.animate({
   *         to: {
   *             width: (myWindow.getWidth() == 500) ? 700 : 500,
   *             height: (myWindow.getHeight() == 300) ? 400 : 300
   *         }
   *     });
   * });
   * </pre>
   * <p>For performance reasons, by default, the internal layout is only updated when the Window
   * reaches its final <code>"to"</code> size. If dynamic updating of the Window's child Components
   * is required, then configure the animation with <code>dynamic: true</code> and the two child items
   * will maintain their proportions during the animation.</p>
   * @param animObj Configuration for →<code>ext.fx.Anim</code>.
   * Note that the →<code>ext.fx.Anim.to</code> config is required.
   * @return this
   * @see ext.Component
   * @see ext.draw.sprite.Sprite
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Animate.html#method-animate Original Ext JS documentation of 'animate'
   * @see ext.fx.Anim
   * @see ext.fx.Anim#to
   */
  public native function animate(animObj:Object):Object;

  /**
   * Returns the current animation if this object has any effects actively running or queued,
   * else returns false.
   * @return Anim if element has active effects, else false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Animate.html#method-getActiveAnimation Original Ext JS documentation of 'getActiveAnimation'
   */
  public native function getActiveAnimation():*;

  [Deprecated(replacement="getActiveAnimation", since="4.0")]
  /**
   * Returns the current animation if this object has any effects actively running or queued,
   * else returns false.
   * @return Anim if element has active effects, else false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Animate.html#method-hasActiveFx Original Ext JS documentation of 'hasActiveFx'
   */
  public native function hasActiveFx():*;

  /**
   * Ensures that all effects queued after sequenceFx is called on this object are run
   * in sequence. This is the opposite of →<code>syncFx()</code>.
   * @return this
   * @see #syncFx()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Animate.html#method-sequenceFx Original Ext JS documentation of 'sequenceFx'
   */
  public native function sequenceFx():Object;

  /**
   * Stops any running effects and clears this object's internal effects queue if it contains
   * any additional effects that haven't started yet.
   * @return The Element
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Animate.html#method-stopAnimation Original Ext JS documentation of 'stopAnimation'
   */
  public native function stopAnimation(suppressEvent:* = undefined):Element;

  [Deprecated(replacement="stopAnimation", since="4.0")]
  /**
   * Stops any running effects and clears this object's internal effects queue if it contains
   * any additional effects that haven't started yet.
   * @return The Element
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Animate.html#method-stopFx Original Ext JS documentation of 'stopFx'
   */
  public native function stopFx():Element;

  /**
   * Ensures that all effects queued after syncFx is called on this object are run concurrently.
   * This is the opposite of →<code>sequenceFx()</code>.
   * @return this
   * @see #sequenceFx()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Animate.html#method-syncFx Original Ext JS documentation of 'syncFx'
   */
  public native function syncFx():Object;
}
}