package ext.util {
import ext.Base;

[Native("Ext.util.Color", require)]
/**
 * Represents an RGB color and provides helper functions on it e.g. to get
 * color components in HSL color space.
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html Original Ext JS documentation of 'Ext.util.Color'
 */
public class Color extends Base {
  /**
   * Convenience method for creating a color.
   * <p>Can be called with several different combinations of arguments:</p>
   * <pre>
   * // Ext.util.Color is returned unchanged.
   * Ext.util.Color.create(new Ext.util.color(255, 0, 0, 0));
   *
   * // CSS color string.
   * Ext.util.Color.create("red");
   *
   * // Array of red, green, blue, alpha
   * Ext.util.Color.create([255, 0, 0, 0]);
   *
   * // Separate arguments of red, green, blue, alpha
   * Ext.util.Color.create(255, 0, 0, 0);
   *
   * // Returns black when no arguments given.
   * Ext.util.Color.create();
   * </pre>
   * @param arg
   * <ul>
   * <li><code>red</code> (optional) —
   * Red component (0..255),
   * CSS color string or array of all components.
   * </li>
   * <li><code>green:Number</code> (optional) —
   * Green component (0..255)
   * </li>
   * <li><code>blue:Number</code> (optional) —
   * Blue component (0..255)
   * </li>
   * <li><code>alpha:Number</code> (default = <code>1</code>) —
   * Alpha component (0..1)
   * </li>
   * </ul>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#static-method-create Original Ext JS documentation of 'create'
   */
  public static native function create(arg:Array):Color;

  /**
   * Returns a flyweight instance of Ext.util.Color.
   * <p>Can be called with either a CSS color string or with separate
   * arguments for red, green, blue, alpha.</p>
   * @param red Red component (0..255) or CSS color string.
   * @param green Green component (0..255)
   * @param blue Blue component (0..255)
   * @param alpha Alpha component (0..1)
   * @default 1
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#static-method-fly Original Ext JS documentation of 'fly'
   */
  public static native function fly(red:*, green:Number = NaN, blue:Number = NaN, alpha:Number = 1):Color;

  /**
   * Create a new color based on the specified HSL values.
   * @param h Hue component [0..360)
   * @param s Saturation component [0..1]
   * @param l Lightness component [0..1]
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#static-method-fromHSL Original Ext JS documentation of 'fromHSL'
   */
  public static native function fromHSL(h:Number, s:Number, l:Number):Color;

  /**
   * Create a new color based on the specified HSV values.
   * @param h Hue component [0..360)
   * @param s Saturation component [0..1]
   * @param v Value component [0..1]
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#static-method-fromHSV Original Ext JS documentation of 'fromHSV'
   */
  public static native function fromHSV(h:Number, s:Number, v:Number):Color;

  /**
   * Parse the string and create a new color.
   * <p>Supported formats:</p>
   * <ul>
   * <li>'#rrggbb'</li>
   * <li>'#rgb', 'rgb(r,g,b)'</li>
   * <li>'rgba(r,g,b,a)'</li>
   * <li>supported CSS color names (e.g., 'black', 'white', etc).</li>
   * </ul>
   * <p>If the string is not recognized, fromString returns rgba(0,0,0,0).</p>
   * @param color Color as string.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#static-method-fromString Original Ext JS documentation of 'fromString'
   */
  public static native function fromString(color:String):Color;

  /**
   * @param red Red component (0..255)
   * @param green Green component (0..255)
   * @param blue Blue component (0..255)
   * @param alpha Alpha component (0..1)
   * @default 1
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-constructor Original Ext JS documentation of 'constructor'
   */
  public function Color(red:Number, green:Number, blue:Number, alpha:Number = 1) {
    super();
  }

  [ExtConfig]
  /**
   * The default factor to compute the lighter or darker color.
   * @default 0.2
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#cfg-lightnessFactor Original Ext JS documentation of 'lightnessFactor'
   */
  public native function get lightnessFactor():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set lightnessFactor(value:Number):void;

  /**
   * Returns a new color that is darker than this color in the HSL color space.
   * @param factor Darker factor (0..1).
   * @default 0.2
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-createDarker Original Ext JS documentation of 'createDarker'
   */
  public native function createDarker(factor:Number = 0.2):Color;

  /**
   * Returns a new color that is lighter than this color in the HSL color space.
   * @param factor Lighter factor (0..1).
   * @default 0.2
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-createLighter Original Ext JS documentation of 'createLighter'
   */
  public native function createLighter(factor:Number = 0.2):Color;

  /**
   * Darken this color in the HSL color space.
   * @param factor Darken factor (0..1).
   * @default 0.2
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-darken Original Ext JS documentation of 'darken'
   */
  public native function darken(factor:Number = 0.2):void;

  /**
   * The the brightness of a color as defined by W3C:
   * https://www.w3.org/TR/AERT#color-contrast
   * @return The brightness, between <code>0</code> and <code>100</code>.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-getBrightness Original Ext JS documentation of 'getBrightness'
   */
  public native function getBrightness():Number;

  /**
   * Returns the gray value (0 to 255) of the color.
   * <p>The gray value is calculated using the formula r<i>0.3 + g</i>0.59 + b&#42;0.11.</p>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-getGrayscale Original Ext JS documentation of 'getGrayscale'
   */
  public native function getGrayscale():Number;

  [ArrayElementType("Number")]
  /**
   * Get the equivalent HSL components of the color.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-getHSL Original Ext JS documentation of 'getHSL'
   */
  public native function getHSL():Array;

  [ArrayElementType("Number")]
  /**
   * Get the equivalent HSV components of the color.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-getHSV Original Ext JS documentation of 'getHSV'
   */
  public native function getHSV():Array;

  /**
   * Lighten this color in the HSL color space.
   * @param factor Lighten factor (0..1).
   * @default 0.2
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-lighten Original Ext JS documentation of 'lighten'
   */
  public native function lighten(factor:Number = 0.2):void;

  /**
   * Parse the string and set the current color.
   * <p>Supported formats:</p>
   * <ul>
   * <li>'#rrggbb'</li>
   * <li>'#rgb', 'rgb(r,g,b)'</li>
   * <li>'rgba(r,g,b,a)'</li>
   * <li>supported CSS color names (e.g., 'black', 'white', etc).</li>
   * </ul>
   * <p>If the string is not recognized, setFromString returns rgba(0,0,0,0).</p>
   * @param str Color as string.
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-setFromString Original Ext JS documentation of 'setFromString'
   */
  public native function setFromString(str:String):*;

  /**
   * Set current color based on the specified HSL values.
   * @param h Hue component [0..360)
   * @param s Saturation component [0..1]
   * @param l Lightness component [0..1]
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-setHSL Original Ext JS documentation of 'setHSL'
   */
  public native function setHSL(h:Number, s:Number, l:Number):Color;

  /**
   * Set current color based on the specified HSV values.
   * @param h Hue component [0..360)
   * @param s Saturation component [0..1]
   * @param v Value component [0..1]
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-setHSV Original Ext JS documentation of 'setHSV'
   */
  public native function setHSV(h:Number, s:Number, v:Number):Color;

  /**
   * Get this color in hexadecimal format.
   * @return The color in hexadecimal format.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-toHex Original Ext JS documentation of 'toHex'
   */
  public native function toHex(color:*):String;

  /**
   * toString() returns a color in hex format ('#rrggbb') if the alpha is 1. If the
   * alpha is less than one, toString() returns the color in RGBA format ('rgba(255,0,0,0.3)').
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Color.html#method-toString Original Ext JS documentation of 'toString'
   */
  public native function toString():String;
}
}