package ext.util {
import ext.Base;

[Native("Ext.util.DelimitedValue", require)]
/**
 * This base class contains utility methods for dealing with formats such as CSV (Comma
 * Separated Values) as specified in <a href="http://tools.ietf.org/html/rfc4180">RFC 4180</a>.
 * <p>The base class implements the mechanics and is governed by these config options:</p>
 * <ul>
 * <li><code>→delimiter</code></li>
 * <li><code>→lineBreak</code></li>
 * <li><code>→quote</code></li>
 * </ul>
 * <p>These options affect the <code>→encode()</code> and <code>→decode()</code> methods.
 * When <i>decoding</i>, however, <code>→lineBreak</code> is ignored and instead each line can
 * be separated by any standard line terminator character or character sequence:</p>
 * <ul>
 * <li><code>\u000a</code></li>
 * <li><code>\u000d</code></li>
 * <li><code>\u000d\u000a</code></li>
 * </ul>
 * <p>Strings which contain the →<code>delimiter</code> character are quoted using the
 * →<code>quote</code> character, and any internal →<code>quote</code> characters are doubled.</p>
 * <p><i>Important</i>
 * While the primary use case is to encode strings, other atomic data types can be encoded
 * as values within a line such as:</p>
 * <ul>
 * <li>Number</li>
 * <li>Boolean</li>
 * <li>Date (encoded as an <a href="http://www.iso.org/iso/home/standards/iso8601.htm">ISO 8601</a> date string.)</li>
 * <li>null (encoded as an empty string.)</li>
 * <li>undefined (encoded as an empty string.)</li>
 * </ul>
 * <p>Not that when <i>decoding</i>, all data is read as strings. This class does not convert
 * incoming data. To do that, use an →<code>ext.data.reader.ArrayReader</code>.</p>
 * <p>See <code>→ext.util.CSV</code> and <code>→ext.util.TsvDecoder</code> for pre-configured instances.</p>
 * @see #delimiter
 * @see #lineBreak
 * @see #quote
 * @see #encode()
 * @see #decode()
 * @see ext.data.reader.ArrayReader
 * @see ext.util.#CSV ext.util.CSV
 * @see ext.util.TsvDecoder
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.DelimitedValue.html Original Ext JS documentation of 'Ext.util.DelimitedValue'
 */
public class DelimitedValue extends Base {
  /**
   * @param config @inheritDoc
   */
  public function DelimitedValue(config:DelimitedValue = null) {
    super();
  }

  [ExtConfig]
  /**
   * The →<code>ext.DateUtil.format()</code> to use for dates
   * @default 'C'
   * @see ext.SDateUtil#format() ext.DateUtil.format()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.DelimitedValue.html#cfg-dateFormat Original Ext JS documentation of 'dateFormat'
   */
  public native function get dateFormat():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set dateFormat(value:String):void;

  [ExtConfig]
  /**
   * The string used to separate the values in a row. Common values for this config
   * are comma (",") and tab ("\t"). See <code>→ext.util.CSV</code> and <code>→ext.util.TsvDecoder</code>
   * for pre-configured instances of these formats.
   * @default '\t'
   * @see ext.util.#CSV ext.util.CSV
   * @see ext.util.TsvDecoder
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.DelimitedValue.html#cfg-delimiter Original Ext JS documentation of 'delimiter'
   */
  public native function get delimiter():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set delimiter(value:String):void;

  [ExtConfig]
  /**
   * The string used by <code>→encode()</code> to separate each row. The <code>→decode()</code>
   * method accepts all forms of line break.
   * @default '\n'
   * @see #encode()
   * @see #decode()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.DelimitedValue.html#cfg-lineBreak Original Ext JS documentation of 'lineBreak'
   */
  public native function get lineBreak():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set lineBreak(value:String):void;

  [ExtConfig]
  /**
   * The character to use as to quote values that contain the special <code>→delimiter</code>
   * or <code>→lineBreak</code> characters.
   * @default '"'
   * @see #delimiter
   * @see #lineBreak
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.DelimitedValue.html#cfg-quote Original Ext JS documentation of 'quote'
   */
  public native function get quote():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set quote(value:String):void;

  /**
   * Decodes a string of encoded values into an array of rows. Each row is an array of
   * strings.
   * <p>Note that this function does not convert the string values in each column into
   * other data types. To do that, use an →<code>ext.data.reader.ArrayReader</code>.</p>
   * <p>For example:</p>
   * <pre>
   * Ext.util.CSV.decode('"foo ""bar"", bletch",Normal String,2010-01-01T21:45:32.004Z\u000a3.141592653589793,1,false');
   * </pre>
   * <p>produces the following array of string arrays:</p>
   * <pre>
   * [
   *     ['foo "bar", bletch','Normal String', '2010-01-01T21:45:32.004Z'],
   *     ['3.141592653589793', '1', 'false']
   * ]
   * </pre>
   * @param input The string to parse.
   * @param delimiter The column delimiter to use if the default value
   * of →<code>delimiter</code> is not desired.
   * @param quoteChar The character used to quote fields. Defaults to
   * double quote ("). Pass <code>null</code> to suppress field quoting.
   * @return An array of rows where each row is an array of Strings.
   * @see ext.data.reader.ArrayReader
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.DelimitedValue.html#method-decode Original Ext JS documentation of 'decode'
   * @see #delimiter
   */
  public native function decode(input:String, delimiter:String = null, quoteChar:String = null):*;

  /**
   * Converts a two-dimensional array into an encoded string.
   * <p>For example:</p>
   * <pre>
   * Ext.util.CSV.encode([
   *     ['foo "bar", bletch', 'Normal String', new Date()],
   *     [Math.PI, 1, false]
   * ]);
   * </pre>
   * <p>The above produces the following string:</p>
   * <pre>
   * '"foo ""bar"", bletch",Normal String,2010-01-01T21:45:32.004Z\u000a3.141592653589793,1,false'
   * </pre>
   * @param input An array of row data arrays.
   * @param delimiter The column delimiter to use if the default value
   * of →<code>delimiter</code> is not desired.
   * @param quoteChar The character used to quote fields. Defaults to
   * double quote ("). Pass <code>null</code> to suppress field quoting.
   * @return A string in which data items are separated by →<code>delimiter</code>
   * characters, and rows are separated by →<code>lineBreak</code> characters.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.DelimitedValue.html#method-encode Original Ext JS documentation of 'encode'
   * @see #delimiter
   * @see #lineBreak
   */
  public native function encode(input:*, delimiter:String = null, quoteChar:String = null):String;
}
}