package ext.util {

[Mixin("ext.util.Positionable")]
/**
 * <b>NOTE: This is a private utility class for internal use by the framework. Don't rely on its existence.</b>
 * <p>This mixin provides a common interface for objects that can be positioned, e.g.
 * <i>Components</i> (→<code>ext.Component</code>) and <i>Elements</i> (→<code>ext.dom.Element</code>)</p>
 * @see ext.Component
 * @see ext.dom.Element
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html Original Ext JS documentation of 'Ext.util.Positionable'
 * @see ext.util.Positionable
 */
public interface IPositionable {
  /**
   * Aligns the element with another element relative to the specified anchor points. If
   * the other element is the document it aligns it to the viewport. The position
   * parameter is optional, and can be specified in any one of the following formats:
   * <ul>
   * <li><b>Blank</b>: Defaults to aligning the element's top-left corner to the target's
   * bottom-left corner ("tl-bl").</li>
   * <li><b>Two anchors</b>: If two values from the table below are passed separated by a dash,
   * the first value is used as the element's anchor point, and the second value is
   * used as the target's anchor point.</li>
   * <li><b>Two edge/offset descriptors:</b> An edge/offset descriptor is an edge initial
   * (<code>t</code>/<code>r</code>/<code>b</code>/<code>l</code>) followed by a percentage along that side. This describes a
   * point to align with a similar point in the target. So <code>'t0-b0'</code> would be
   * the same as <code>'tl-bl'</code>, <code>'l0-r50'</code> would place the top left corner of this item
   * halfway down the right edge of the target item. This allows more flexibility
   * and also describes which two edges are considered adjacent when positioning a tip pointer.</li>
   * </ul>
   * <p>Following are all of the supported predefined anchor positions:</p>
   * <pre>
   *  Value  Description
   *  -----  -----------------------------
   *  tl     The top left corner
   *  t      The center of the top edge
   *  tr     The top right corner
   *  l      The center of the left edge
   *  c      The center
   *  r      The center of the right edge
   *  bl     The bottom left corner
   *  b      The center of the bottom edge
   *  br     The bottom right corner
   * </pre>
   * <p>You can put a '?' at the end of the alignment string to constrain the positioned element
   * to the →<code>ext.container.Viewport</code>. The element will attempt to align as specified, but
   * the position will be adjusted to constrain to the viewport if necessary. Note that
   * the element being aligned might be swapped to align to a different position than that
   * specified in order to enforce the viewport constraints.</p>
   * <p>Example Usage:</p>
   * <pre>
   * // align el to other-el using the default positioning
   * // ("tl-bl", non-constrained)
   * el.alignTo("other-el");
   *
   * // align the top left corner of el with the top right corner of other-el
   * // (constrained to viewport)
   * el.alignTo("other-el", "tl-tr?");
   *
   * // align the bottom right corner of el with the center left edge of other-el
   * el.alignTo("other-el", "br-l?");
   *
   * // align the center of el with the bottom left corner of other-el and
   * // adjust the x position by -6 pixels (and the y position by 0)
   * el.alignTo("other-el", "c-bl", [-6, 0]);
   *
   * // align the 25% point on the bottom edge of this el
   * // with the 75% point on the top edge of other-el.
   * el.alignTo("other-el", 'b25-t75');
   * </pre>
   * @param anchorToEl The Positionable,
   * HTMLElement, or id of the element to align to.
   * @param alignment The position to align to
   * @default "tl-bl?"
   * @param offsets Offset the positioning by [x, y]
   * @param animate true for the default animation or a standard
   * Element animation config object
   * @see ext.container.Viewport
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-alignTo Original Ext JS documentation of 'alignTo'
   */
  function alignTo(anchorToEl:*, alignment:String = "tl-bl?", offsets:Array = null, animate:* = undefined):void;

  /**
   * Anchors an element to another element and realigns it when the window is resized.
   * <p><strong>Defined in override Ext.overrides.util.Positionable.</strong></p>
   * @param anchorToEl The Positionable,
   * HTMLElement, or id of the element to align to.
   * @param alignment The position to align to
   * @default "tl-bl?"
   * @param offsets Offset the positioning by [x, y]
   * @param animate true for the default animation or a standard
   * Element animation config object
   * @param monitorScroll True to monitor body scroll and
   * reposition. If this parameter is a number, it is used as the buffer delay in
   * milliseconds.
   * @default 50
   * @param callback The function to call after the animation finishes
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-anchorTo Original Ext JS documentation of 'anchorTo'
   */
  function anchorTo(anchorToEl:*, alignment:String = "tl-bl?", offsets:Array = null, animate:* = undefined, monitorScroll:* = 50, callback:Function = null):void;

  [ArrayElementType("Number")] /**
   * Gets the x,y coordinates to align this element with another element. See
   * →<code>alignTo()</code> for more info on the supported position values.
   * @param alignToEl The Positionable,
   * HTMLElement, or id of the element to align to.
   * @param position The position to align to
   * @default "tl-bl?"
   * @param offsets Offset the positioning by [x, y]
   * @return [x, y]
   * @see ext.util.Positionable#alignTo()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getAlignToXY Original Ext JS documentation of 'getAlignToXY'
   */
  function getAlignToXY(alignToEl:*, position:String = "tl-bl?", offsets:Array = null):Array;

  [ArrayElementType("Number")] /**
   * Gets the x,y coordinates specified by the anchor position on the element.
   * @param anchor The specified anchor position.
   * See →<code>alignTo()</code> for details on supported anchor positions.
   * @default 'tl'
   * @param local True to get the local (element top/left-relative) anchor
   * position instead of page coordinates
   * @param size An object containing the size to use for calculating anchor
   * position {width: (target width), height: (target height)} (defaults to the
   * element's current size)
   * @return [x, y] An array containing the element's x and y coordinates
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getAnchorXY Original Ext JS documentation of 'getAnchorXY'
   * @see ext.util.Positionable#alignTo()
   */
  function getAnchorXY(anchor:String = 'tl', local:Boolean = false, size:Object = null):Array;

  /**
   * Return an object defining the area of this Element which can be passed to
   * →<code>setBox()</code> to set another Element's size/location to match this element.
   * @param contentBox If true a box for the content of the element is
   * returned.
   * @param local If true the element's left and top relative to its
   * <code>offsetParent</code> are returned instead of page x/y.
   * @return An object in the format
   * <ul>
   * <li><code>x:Number</code> (optional) —
   * The element's X position.
   * </li>
   * <li><code>y:Number</code> (optional) —
   * The element's Y position.
   * </li>
   * <li><code>width:Number</code> (optional) —
   * The element's width.
   * </li>
   * <li><code>height:Number</code> (optional) —
   * The element's height.
   * </li>
   * <li><code>bottom:Number</code> (optional) —
   * The element's lower bound.
   * </li>
   * <li><code>right:Number</code> (optional) —
   * The element's rightmost bound.
   * <p>The returned object may also be addressed as an Array where index 0 contains the X
   * position and index 1 contains the Y position. The result may also be used for
   * →<code>setXY()</code></p>
   * </li>
   * </ul>
   * @see ext.util.Positionable#setBox()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getBox Original Ext JS documentation of 'getBox'
   * @see ext.util.Positionable#setXY()
   */
  function getBox(contentBox:Boolean = false, local:Boolean = false):Object;

  /**
   * Returns a region object that defines the client area of this element.
   * <p>That is, the area <i>within</i> any scrollbars.</p>
   * @return A Region containing "top, left, bottom, right" properties.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getClientRegion Original Ext JS documentation of 'getClientRegion'
   */
  function getClientRegion():Region;

  /**
   * Returns the content region of this element for purposes of constraining or clipping floating
   * children. That is the region within the borders and scrollbars, but not within the padding.
   * @return A Region containing "top, left, bottom, right" properties.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getConstrainRegion Original Ext JS documentation of 'getConstrainRegion'
   */
  function getConstrainRegion():Region;

  /**
   * Returns the <code>[X, Y]</code> vector by which this Positionable's element must be translated to make
   * a best attempt to constrain within the passed constraint. Returns <code>false</code> if the element
   * does not need to be moved.
   * <p>Priority is given to constraining the top and left within the constraint.</p>
   * <p>The constraint may either be an existing element into which the element is to be
   * constrained, or a →<code>ext.util.Region</code> into which this element is to be
   * constrained.</p>
   * <p>By default, any extra shadow around the element is <b>not</b> included in the constrain
   * calculations - the edges of the element are used as the element bounds. To constrain
   * the shadow within the constrain region, set the <code>constrainShadow</code> property on this element
   * to <code>true</code>.</p>
   * @param constrainTo The
   * Positionable, HTMLElement, element id, or Region into which the element is to be
   * constrained.
   * @param proposedPosition A proposed <code>[X, Y]</code> position to test for validity
   * and to produce a vector for instead of using the element's current position
   * @param proposedSize A proposed <code>[width, height]</code> size to constrain
   * instead of using the element's current size
   * @return <b>If</b> the element <i>needs</i> to be translated, an <code>[X, Y]</code>
   * vector by which this element must be translated. Otherwise, <code>false</code>.
   * @see ext.util.Region
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getConstrainVector Original Ext JS documentation of 'getConstrainVector'
   */
  function getConstrainVector(constrainTo:* = undefined, proposedPosition:Array = null, proposedSize:Array = null):*;

  /**
   * Returns the x coordinate of this element reletive to its <code>offsetParent</code>.
   * @return The local x coordinate
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getLocalX Original Ext JS documentation of 'getLocalX'
   */
  function getLocalX():Number;

  [ArrayElementType("Number")] /**
   * Returns the x and y coordinates of this element relative to its <code>offsetParent</code>.
   * @return The local XY position of the element
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getLocalXY Original Ext JS documentation of 'getLocalXY'
   */
  function getLocalXY():Array;

  /**
   * Returns the y coordinate of this element reletive to its <code>offsetParent</code>.
   * @return The local y coordinate
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getLocalY Original Ext JS documentation of 'getLocalY'
   */
  function getLocalY():Number;

  [ArrayElementType("Number")] /**
   * Returns the offsets of this element from the passed element. The element must both
   * be part of the DOM tree and not have display:none to have page coordinates.
   * @param offsetsTo The Positionable,
   * HTMLElement, or element id to get get the offsets from.
   * @return The XY page offsets (e.g. <code>[100, -200]</code>)
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getOffsetsTo Original Ext JS documentation of 'getOffsetsTo'
   */
  function getOffsetsTo(offsetsTo:*):Array;

  /**
   * Returns a region object that defines the area of this element.
   * @param contentBox If true a box for the content of the element is
   * returned.
   * @param local If true the element's left and top relative to its
   * <code>offsetParent</code> are returned instead of page x/y.
   * @return A Region containing "top, left, bottom, right" properties.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getRegion Original Ext JS documentation of 'getRegion'
   */
  function getRegion(contentBox:Boolean = false, local:Boolean = false):Region;

  /**
   * Returns the <b>content</b> region of this element. That is the region within the borders
   * and padding.
   * @return A Region containing "top, left, bottom, right" member data.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getViewRegion Original Ext JS documentation of 'getViewRegion'
   */
  function getViewRegion():Region;

  /**
   * Gets the current X position of the DOM element based on page coordinates.
   * @return The X position of the element
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getX Original Ext JS documentation of 'getX'
   */
  function getX():Number;

  [ArrayElementType("Number")] /**
   * Gets the current position of the DOM element based on page coordinates.
   * @return The XY position of the element
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getXY Original Ext JS documentation of 'getXY'
   */
  function getXY():Array;

  /**
   * Gets the current Y position of the DOM element based on page coordinates.
   * @return The Y position of the element
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-getY Original Ext JS documentation of 'getY'
   */
  function getY():Number;

  /**
   * Move the element relative to its current position.
   * @param direction Possible values are:
   * <ul>
   * <li><code>"l"</code> (or <code>"left"</code>)</li>
   * <li><code>"r"</code> (or <code>"right"</code>)</li>
   * <li><code>"t"</code> (or <code>"top"</code>, or <code>"up"</code>)</li>
   * <li><code>"b"</code> (or <code>"bottom"</code>, or <code>"down"</code>)</li>
   * </ul>
   * @param distance How far to move the element in pixels
   * @param animate true for the default animation or a standard
   * Element animation config object
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-move Original Ext JS documentation of 'move'
   */
  function move(direction:String, distance:Number, animate:* = undefined):void;

  /**
   * Remove any anchor to this element. See →<code>anchorTo()</code>.
   * @see ext.util.Positionable#anchorTo()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-removeAnchor Original Ext JS documentation of 'removeAnchor'
   */
  function removeAnchor():void;

  /**
   * Sets the element's box.
   * @param box The box to fill {x, y, width, height}
   * @param animate true for the default animation or a standard
   * Element animation config object
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-setBox Original Ext JS documentation of 'setBox'
   */
  function setBox(box:Object, animate:Boolean = undefined):void;

  /**
   * Sets the local x coordinate of this element using CSS style. When used on an
   * absolute positioned element this method is symmetrical with →<code>getLocalX()</code>, but
   * may not be symmetrical when used on a relatively positioned element.
   * @param x The x coordinate. A value of <code>null</code> sets the left style to 'auto'.
   * @see ext.util.Positionable#getLocalX()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-setLocalX Original Ext JS documentation of 'setLocalX'
   */
  function setLocalX(x:Number):void;

  /**
   * Sets the local x and y coordinates of this element using CSS style. When used on an
   * absolute positioned element this method is symmetrical with →<code>getLocalXY()</code>, but
   * may not be symmetrical when used on a relatively positioned element.
   * @param x The x coordinate or an array containing [x, y]. A value of
   * <code>null</code> sets the left style to 'auto'
   * @param y The y coordinate, required if x is not an array. A value of
   * <code>null</code> sets the top style to 'auto'
   * @see ext.util.Positionable#getLocalXY()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-setLocalXY Original Ext JS documentation of 'setLocalXY'
   */
  function setLocalXY(x:*, y:Number = NaN):void;

  /**
   * Sets the local y coordinate of this element using CSS style. When used on an
   * absolute positioned element this method is symmetrical with →<code>getLocalY()</code>, but
   * may not be symmetrical when used on a relatively positioned element.
   * @param y The y coordinate. A value of <code>null</code> sets the top style to 'auto'.
   * @see ext.util.Positionable#getLocalY()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-setLocalY Original Ext JS documentation of 'setLocalY'
   */
  function setLocalY(y:Number):void;

  /**
   * Sets the X position of the DOM element based on page coordinates.
   * @param x The X position
   * @param animate True for the default animation, or a standard
   * Element animation config object
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-setX Original Ext JS documentation of 'setX'
   */
  function setX(x:Number, animate:* = undefined):void;

  /**
   * Sets the position of the DOM element in page coordinates.
   * @param pos Contains X &amp; Y [x, y] values for new position (coordinates
   * are page-based)
   * @param animate True for the default animation, or a standard
   * Element animation config object
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-setXY Original Ext JS documentation of 'setXY'
   */
  function setXY(pos:Array, animate:* = undefined):void;

  /**
   * Sets the Y position of the DOM element based on page coordinates.
   * @param y The Y position
   * @param animate True for the default animation, or a standard
   * Element animation config object
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-setY Original Ext JS documentation of 'setY'
   */
  function setY(y:Number, animate:* = undefined):void;

  /**
   * Translates the passed page coordinates into left/top css values for the element
   * @param x The page x or an array containing [x, y]
   * @param y The page y, required if x is not an array
   * @return An object with left and top properties. e.g.
   * {left: (value), top: (value)}
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Positionable.html#method-translatePoints Original Ext JS documentation of 'translatePoints'
   */
  function translatePoints(x:*, y:Number = NaN):Object;
}
}