package ext.util {

[Mixin("ext.util.Sortable")]
/**
 * A mixin which allows a data component to be sorted. This is used by e.g. →<code>ext.data.Store</code>
 * and →<code>ext.data.TreeStore</code>.
 * <p><b>NOTE</b>: This mixin is mainly for internal use and most users should not need to use it
 * directly. It is more likely you will want to use one of the component classes that import
 * this mixin, such as →<code>ext.data.Store</code> or →<code>ext.data.TreeStore</code>.</p>
 * @see ext.data.Store
 * @see ext.data.TreeStore
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Sortable.html Original Ext JS documentation of 'Ext.util.Sortable'
 * @see ext.util.Sortable
 */
public interface ISortable {
  [ExtConfig] /**
   * The default sort direction to use if one is not specified.
   * @default "ASC"
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Sortable.html#cfg-defaultSortDirection Original Ext JS documentation of 'defaultSortDirection'
   */
  function get defaultSortDirection():String;

  [ExtConfig] function set defaultSortDirection(value:String):void;

  /**
   * <code>true</code> in this class to identify an object as an instantiated Sortable, or subclass thereof.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Sortable.html#property-isSortable Original Ext JS documentation of 'isSortable'
   */
  function get isSortable():Boolean;

  function set isSortable(value:Boolean):void;

  [ExtConfig] /**
   * The maximum number of sorters which may be applied to this Sortable when using the "multi"
   * insertion position when adding sorters.
   * <p>New sorters added using the "multi" insertion position are inserted at the top of the
   * sorters list becoming the new primary sort key.</p>
   * <p>If the sorters collection has grown to longer then <b><code>→multiSortLimit</code></b>, then it is trimmed.</p>
   * @default 3
   * @see #multiSortLimit
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Sortable.html#cfg-multiSortLimit Original Ext JS documentation of 'multiSortLimit'
   */
  function get multiSortLimit():Number;

  [ExtConfig] function set multiSortLimit(value:Number):void;

  [ExtConfig] /**
   * The property in each item that contains the data to sort.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Sortable.html#cfg-sortRoot Original Ext JS documentation of 'sortRoot'
   */
  function get sortRoot():String;

  [ExtConfig] function set sortRoot(value:String):void;

  [ExtConfig]
  [Bindable] /**
   * The initial set of <i>Sorters</i> (→<code>ext.util.Sorter</code>).
   * <pre>
   * sorters: [{
   *     property: 'age',
   *     direction: 'DESC'
   * }, {
   *     property: 'firstName',
   *     direction: 'ASC'
   * }]
   * </pre>
   * @default null
   * @see ext.util.Sorter
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Sortable.html#cfg-sorters Original Ext JS documentation of 'sorters'
   * @see #getSorters()
   * @see #setSorters()
   */
  function get sorters():Array;

  [ExtConfig]
  [Bindable]
  [ArrayElementType("ext.util.Sorter")] function set sorters(value:Array):void;

  /**
   * Returns a comparator function which compares two items and returns -1, 0, or 1 depending
   * on the currently defined set of →<code>sorters</code>.
   * <p>If there are no →<code>sorters</code> defined, it returns a function which returns <code>0</code> meaning
   * that no sorting will occur.</p>
   * @see ext.util.Sortable#sorters
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Sortable.html#method-generateComparator Original Ext JS documentation of 'generateComparator'
   */
  function generateComparator():void;

  [ArrayElementType("ext.util.Sorter")] /**
   * Returns the value of <code>sorters</code>.
   * @see #sorters
   */
  function getSorters():Array;

  /**
   * Sets the value of <code>sorters</code>.
   * @param sorters The new value.
   * @see #sorters
   */
  function setSorters(sorters:Array):void;

  [ArrayElementType("ext.util.Sorter")] /**
   * Updates the sorters collection and triggers sorting of this Sortable. Example usage:
   * <pre>
   * //sort by a single field
   * myStore.sort('myField', 'DESC');
   *
   * //sorting by multiple fields
   * myStore.sort([{
   *     property : 'age',
   *     direction: 'ASC'
   * }, {
   *     property : 'name',
   *     direction: 'DESC'
   * }]);
   * </pre>
   * <p>Classes which use this mixin must implement a <b><code>soSort</code></b> method which accepts a comparator
   * function computed from the full sorter set which performs the sort
   * in an implementation-specific way.</p>
   * <p>When passing a single string argument to sort, Store maintains a ASC/DESC toggler per field,
   * so this code:</p>
   * <pre>
   * store.sort('myField');
   * store.sort('myField');
   * </pre>
   * <p>Is equivalent to this code, because Store handles the toggling automatically:</p>
   * <pre>
   * store.sort('myField', 'ASC');
   * store.sort('myField', 'DESC');
   * </pre>
   * @param sorters Either a string name of one of the fields
   * in this Store's configured →<code>ext.data.Model</code>, or an array of sorter
   * configurations.
   * @param direction The overall direction to sort the data by.
   * @default "ASC"
   * @param insertionPosition Where to put the new sorter in the collection
   * of sorters. This may take the following values:
   * <ul>
   * <li>
   * <p><code>replace</code>: This means that the new sorter(s) becomes the sole sorter set for this Sortable.
   * This is the most useful call mode to programatically sort by multiple fields.</p>
   * </li>
   * <li>
   * <p><code>prepend</code>: This means that the new sorters are inserted as the primary sorters, unchanged,
   * and the sorter list length must be controlled by the developer.</p>
   * </li>
   * <li>
   * <p><code>multi</code>: This is mainly useful for implementing intuitive "Sort by this" user interfaces
   * such as the →<code>ext.grid.GridPanel</code>'s column sorting UI. This mode is only
   * supported when passing a property name and a direction. This means that the new sorter
   * becomes the primary sorter. If the sorter was <b>already</b> the primary sorter, the direction
   * of sort is toggled if no direction parameter is specified. The number of sorters maintained
   * is limited by the →<code>multiSortLimit</code> configuration.</p>
   * </li>
   * <li>
   * <p><code>append</code> : This means that the new sorter becomes the last sorter.</p>
   * </li>
   * </ul>
   * @default "replace"
   * @param doSort True to sort using a generated sorter function that combines all
   * of the Sorters passed
   * @return The new sorters.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Sortable.html#method-sort Original Ext JS documentation of 'sort'
   * @see ext.data.Model
   * @see ext.grid.GridPanel
   * @see ext.util.Sortable#multiSortLimit
   */
  function sort(sorters:* = undefined, direction:String = "ASC", insertionPosition:String = "replace", doSort:Boolean = false):Array;
}
}