package ext.util {
import ext.Base;
import ext.event.Event;

[Native("Ext.util.KeyMap", require)]
/**
 * Handles mapping key events to handling functions for an element or a Component.
 * One KeyMap can be used for multiple actions.
 * <p>A KeyMap must be configured with a →<code>target</code> as an event source which may be
 * an Element or a Component.</p>
 * <p>If the target is an element, then the <code>keydown</code> event will trigger the invocation
 * of →<code>binding</code>s.</p>
 * <p>It is possible to configure the KeyMap with a custom →<code>eventName</code> to listen for.
 * This may be useful when the →<code>target</code> is a Component.</p>
 * <p>The KeyMap's event handling requires that the first parameter passed is a key event.
 * So if the Component's event signature is different, specify a →<code>processEvent</code>
 * configuration which accepts the event's parameters and returns a key event.</p>
 * <p>Functions specified in →<code>binding</code>s are called with this signature:
 * <code>(String key, Ext.event.Event e)</code> (if the match is a multi-key combination
 * the callback will still be called only once). A KeyMap can also handle a string
 * representation of keys. By default KeyMap starts enabled.</p>
 * <p>Usage:</p>
 * <pre>
 *  // map one key by key code
 *  var map = new Ext.util.KeyMap({
 *      target: "my-element",
 *      key: 13, // or Ext.event.Event.ENTER
 *      handler: myHandler,
 *      scope: myObject
 *  });
 *
 *  // map multiple keys to one action by string
 *  var map = new Ext.util.KeyMap({
 *      target: "my-element",
 *      key: "a\r\n\t",
 *      handler: myHandler,
 *      scope: myObject
 *  });
 *
 *  // map multiple keys to multiple actions by strings and array of codes
 *  var map = new Ext.util.KeyMap({
 *      target: "my-element",
 *      binding: [{
 *          key: [10, 13],
 *          handler: function() {
 *              alert("Return was pressed");
 *          }
 *      }, {
 *          key: "abc",
 *          handler: function() {
 *              alert('a, b or c was pressed');
 *          }
 *      }, {
 *          key: "\t",
 *          ctrl: true,
 *          shift: true,
 *          handler: function() {
 *              alert('Control + shift + tab was pressed.');
 *          }
 *      }]
 *  });
 * </pre>
 * <p>KeyMaps can also bind to Components and process key-based events fired by Components.</p>
 * <p>To bind to a Component, include the Component event name to listen for, and a <code>→processEvent</code>
 * implementation which returns the key event for further processing by the KeyMap:</p>
 * <pre>
 *  var map = new Ext.util.KeyMap({
 *      target: myGridView,
 *      eventName: 'itemkeydown',
 *      processEvent: function(view, record, node, index, event) {
 *          // Load the event with the extra information needed by the mappings
 *          event.view = view;
 *          event.store = view.getStore();
 *          event.record = record;
 *          event.index = index;
 *          return event;
 *      },
 *      binding: {
 *          key: Ext.event.Event.DELETE,
 *          handler: function(keyCode, e) {
 *              e.store.remove(e.record);
 *
 *              // Attempt to select the record that's now in its place
 *              e.view.getSelectionModel().select(e.index);
 *          }
 *      }
 *  });
 * </pre>
 * @see #target
 * @see #binding
 * @see #eventName
 * @see #processEvent
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html Original Ext JS documentation of 'Ext.util.KeyMap'
 */
public class KeyMap extends Base {
  /**
   * @param config @inheritDoc
   */
  public function KeyMap(config:KeyMap = null) {
    super();
  }

  [ExtConfig]
  /**
   * Either a single object describing a handling function for s specified key (or set of keys),
   * or an array of such objects.
   * <ul>
   * <li><code>key</code> (optional) —
   * A single keycode or an array of keycodes to handle, or
   * a RegExp which specifies characters to handle, eg <code>/[a-z]/</code>.
   * </li>
   * <li><code>shift:Boolean</code> (optional) —
   * <code>true</code> to handle key only when shift is pressed, <code>false</code>
   * to handle the key only when shift is not pressed (defaults to undefined)
   * </li>
   * <li><code>ctrl:Boolean</code> (optional) —
   * <code>true</code> to handle key only when ctrl is pressed, <code>false</code>
   * to handle the key only when ctrl is not pressed (defaults to undefined)
   * </li>
   * <li><code>alt:Boolean</code> (optional) —
   * <code>true</code> to handle key only when alt is pressed, <code>false</code>
   * to handle the key only when alt is not pressed (defaults to undefined)
   * </li>
   * <li><code>handler:Function</code> (optional) —
   * The function to call when KeyMap finds the expected
   * key combination
   * </li>
   * <li><code>fn:Function</code> (optional) —
   * Alias of handler (for backwards-compatibility)
   * </li>
   * <li><code>scope:Object</code> (optional) —
   * The scope (<code>this</code> context) in which the handler function
   * is executed.
   * </li>
   * <li><code>defaultEventAction:String</code> (optional) —
   * A default action to apply to the event
   * <i>when the handler returns <code>true</code></i>. Possible values are: stopEvent, stopPropagation,
   * preventDefault. If no value is set no action is performed.
   * </li>
   * </ul>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#cfg-binding Original Ext JS documentation of 'binding'
   */
  public native function get binding():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set binding(value:*):void;

  [ExtConfig]
  /**
   * The event to listen for to pick up key events.
   * @default 'keydown'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#cfg-eventName Original Ext JS documentation of 'eventName'
   */
  public native function get eventName():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set eventName(value:String):void;

  [ExtConfig]
  /**
   * Configure this as <code>true</code> if there are any input fields within the →<code>target</code>,
   * and this KeyNav should not process events from input fields
   * (<code>&lt;input&gt;</code>,<code>&lt;textarea&gt; and elements with</code>contentEditable="true"`)
   * @default false
   * @see #target
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#cfg-ignoreInputFields Original Ext JS documentation of 'ignoreInputFields'
   */
  public native function get ignoreInputFields():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set ignoreInputFields(value:Boolean):void;

  /**
   * The last key event that this KeyMap handled.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#property-lastKeyEvent Original Ext JS documentation of 'lastKeyEvent'
   */
  public native function get lastKeyEvent():Event;

  /**
   * @private
   */
  public native function set lastKeyEvent(value:Event):void;

  [ExtConfig]
  /**
   * The priority to set on this KeyMap's listener. Listeners with a higher priority
   * are fired before those with lower priority.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#cfg-priority Original Ext JS documentation of 'priority'
   */
  public native function get priority():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set priority(value:Number):void;

  [ExtConfig]
  /**
   * An optional event processor function which accepts the argument list provided by the
   * <i>configured event</i> (→<code>eventName</code>) of the →<code>target</code>, and returns a keyEvent
   * for processing by the KeyMap.
   * <p>This may be useful when the →<code>target</code> is a Component with a complex event signature,
   * where the event is not the first parameter. Extra information from the event arguments
   * may be injected into the event for use by the handler functions before returning it.</p>
   * <p>If <code>null</code> is returned the KeyMap stops processing the event.</p>
   * @default Ext.identityFn
   * @see #eventName
   * @see #target
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#cfg-processEvent Original Ext JS documentation of 'processEvent'
   */
  public native function get processEvent():Function;

  [ExtConfig]
  /**
   * @private
   */
  public native function set processEvent(value:Function):void;

  [ExtConfig]
  /**
   * The scope (<code>this</code> context) in which the →<code>processEvent</code> method is executed.
   * @default this
   * @see #processEvent
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#cfg-processEventScope Original Ext JS documentation of 'processEventScope'
   */
  public native function get processEventScope():Object;

  [ExtConfig]
  /**
   * @private
   */
  public native function set processEventScope(value:Object):void;

  [ExtConfig]
  /**
   * The object on which to listen for the event specified by the →<code>eventName</code>
   * config option.
   * @see #eventName
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#cfg-target Original Ext JS documentation of 'target'
   */
  public native function get target():*;

  [ExtConfig]
  /**
   * @private
   */
  public native function set target(value:*):void;

  /**
   * Add a new binding to this KeyMap.
   * <p>Usage:</p>
   * <pre>
   *  // Create a KeyMap
   *  var map = new Ext.util.KeyMap({
   *      target: Ext.getDoc(),
   *      key: Ext.event.Event.ENTER,
   *      handler: handleKey
   *  });
   *
   *  // Add a new binding to the existing KeyMap later
   *  map.addBinding({
   *      key: 'abc',
   *      shift: true,
   *      handler: handleKey
   *  });
   * </pre>
   * @param binding A single KeyMap config or an array of configs.
   * The following config object properties are supported:
   * <ul>
   * <li><code>key</code> (optional) —
   * A single keycode or an array of keycodes to handle,
   * or a RegExp which specifies characters to handle, eg <code>/[a-z]/</code>.
   * </li>
   * <li><code>shift:Boolean</code> (optional) —
   * <code>true</code> to handle key only when shift is pressed,
   * <code>false</code> to handle the key only when shift is not pressed (defaults to undefined).
   * </li>
   * <li><code>ctrl:Boolean</code> (optional) —
   * <code>true</code> to handle key only when ctrl is pressed,
   * <code>false</code> to handle the key only when ctrl is not pressed (defaults to undefined).
   * </li>
   * <li><code>alt:Boolean</code> (optional) —
   * <code>true</code> to handle key only when alt is pressed,
   * <code>false</code> to handle the key only when alt is not pressed (defaults to undefined).
   * </li>
   * <li><code>handler:Function</code> (optional) —
   * The function to call when KeyMap finds the
   * expected key combination.
   * </li>
   * <li><code>fn:Function</code> (optional) —
   * Alias of handler (for backwards-compatibility).
   * </li>
   * <li><code>scope:Object</code> (optional) —
   * The scope (<code>this</code> context) in which the handler function
   * is executed.
   * </li>
   * <li><code>defaultEventAction:String</code> (optional) —
   * A default action to apply to the event
   * <i>when the handler returns <code>true</code></i>. Possible values are: stopEvent, stopPropagation,
   * preventDefault. If no value is set no action is performed.
   * </li>
   * </ul>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#method-addBinding Original Ext JS documentation of 'addBinding'
   */
  public native function addBinding(binding:*):void;

  /**
   * Destroys the KeyMap instance and removes all handlers.
   * @param params true to also remove the →<code>target</code>
   * @see #target
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#method-destroy Original Ext JS documentation of 'destroy'
   */
  override public native function destroy(...params):void;

  /**
   * Disable this KeyMap
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#method-disable Original Ext JS documentation of 'disable'
   */
  public native function disable():void;

  /**
   * Enables this KeyMap
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#method-enable Original Ext JS documentation of 'enable'
   */
  public native function enable():void;

  /**
   * Returns true if this KeyMap is enabled
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#method-isEnabled Original Ext JS documentation of 'isEnabled'
   */
  public native function isEnabled():Boolean;

  /**
   * Shorthand for adding a single key listener.
   * @param key Either the numeric key code, array of key codes
   * or an object with the following options: <code>{key: (number or array), shift: (true/false), ctrl: (true/false), alt: (true/false)}</code>
   * @param fn The function to call
   * @param scope The scope (<code>this</code> reference) in which the function is executed.
   * Defaults to the browser window.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#method-on Original Ext JS documentation of 'on'
   */
  public native function on(key:*, fn:Function, scope:Object = null):void;

  /**
   * Remove a binding from this KeyMap.
   * @param binding See <i>for options</i> (→<code>addBinding()</code>)
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#method-removeBinding Original Ext JS documentation of 'removeBinding'
   * @see #addBinding()
   */
  public native function removeBinding(binding:Object):void;

  /**
   * Convenience function for setting disabled/enabled by boolean.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#method-setDisabled Original Ext JS documentation of 'setDisabled'
   */
  public native function setDisabled(disabled:Boolean):void;

  /**
   * Shorthand for removing a single key listener.
   * @param key Either the numeric key code, array of key codes
   * or an object with the following options: <code>{key: (number or array), shift: (true/false), ctrl: (true/false), alt: (true/false)}</code>
   * @param fn The function to call
   * @param scope The scope (<code>this</code> reference) in which the function is executed.
   * Defaults to the browser window.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.KeyMap.html#method-un Original Ext JS documentation of 'un'
   */
  public native function un(key:*, fn:Function, scope:Object = null):void;
}
}