package ext.util {
import ext.Base;

[Native("Ext.util.LocalStorage", require)]
/**
 * This class provides a common API to LocalStorage with backwards compatibility for IE.
 * <p>The primary aspects of this API match the HTML5 standard API except that this class
 * provides a scoping mechanism to isolate property values by instance. This scope is
 * determined from the <code>→id</code> property. Further, this class does not expose the number of
 * keys in the store as a <code>length</code> property as this cannot be maintained reliably without
 * undue cost. Instead there is a <code>→getKeys()</code> method that returns the cached array of keys
 * which is lazily populated on first call.</p>
 * <p>For example:</p>
 * <pre>
 *  var store = new Ext.util.LocalStorage({
 *          id: 'foo'
 *      });
 *
 *  store.setItem('bar', 'stuff');
 *
 *  // Equivalent to:
 *  window.localStorage.setItem('foo-bar', 'stuff');
 * </pre>
 * <p>In all cases, the <code>→id</code> property is only used by the underlying storage and should not
 * be needed in item access calls or appear when enumerating keys.</p>
 * <p>To continue with the previous example:</p>
 * <pre>
 *  var keys = store.getKeys();
 *  console.log(keys.length);   // logs 1
 *  console.log(store.key(0));  // logs "bar"
 * </pre>
 * <p><b><i>Sharing Instances</i></b></p>
 * <p>The management of the underlying storage can be broken if multiple instances of this
 * class are created with the same <code>→id</code> simultaneously. To avoid creating multiple instances
 * with the same <code>→id</code>, use the <code>→get()</code> method and it will lazily create and share a single
 * instance. When you are done with the shared instance, call <code>→release()</code>.</p>
 * <pre>
 *  var storage = Ext.util.LocalStorage.get('id');
 *
 *  ...
 *
 *  storage.release(); // do not call `→<code>destroy()</code>` as others may be using this object
 * </pre>
 * <p><b>IMPORTANT:</b> Do not mix direction instantiation and <code>→get()</code> with the same <code>→id</code>.</p>
 * <p><b><i>Legacy IE</i></b></p>
 * <p>Older IE browsers (specifically IE7 and below) do not support <code>localStorage</code> so this
 * class provides equivalent support using the IE proprietary persistence mechanism: the
 * <a href="http://msdn.microsoft.com/en-us/library/ms531424(VS.85).aspx"><code>userData</code> behavior</a>. In
 * this mode, the <code>→id</code> serves as name passed to the <code>load</code> and <code>→save()</code> methods and as the
 * suffix on the DOM element added to the <code>head</code>.</p>
 * <p>In this mode, writes to the underlying storage are buffered and delayed for performance
 * reasons. This can be managed using the <code>flushDelay</code> config or by directly calling the
 * <code>→save()</code> method.</p>
 * @see #id
 * @see #getKeys()
 * @see #get()
 * @see #release()
 * @see #destroy()
 * @see #save()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html Original Ext JS documentation of 'Ext.util.LocalStorage'
 */
public class LocalStorage extends Base {
  /**
   * This will be <code>true</code> if some form of local storage is supported or <code>false</code> if not.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#static-property-supported Original Ext JS documentation of 'supported'
   */
  public static native function get supported():Boolean;

  /**
   * Returns a shared instance of the desired local store given its <code>→id</code>. When you
   * are finished with the returned object call the <code>→release()</code> method:
   * <pre>
   *  var store = Ext.util.LocalStorage.get('foo');
   *
   *  // .. use store
   *
   *  store.release();
   * </pre>
   * <p><b>NOTE:</b> Do not mix this call with direct instantiation of the same <code>→id</code>.</p>
   * @param id The <code>→id</code> of the desired instance or a config object
   * with an <code>→id</code> property at a minimum.
   * @return The desired instance, created if needed.
   * @see #id
   * @see #release()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#static-method-get Original Ext JS documentation of 'get'
   */
  public static native function get(id:*):LocalStorage;

  /**
   * @param config @inheritDoc
   */
  public function LocalStorage(config:LocalStorage = null) {
    super();
  }

  /**
   * This property is set to <code>true</code> when the instance's <code>→destroy()</code> method is called.
   * @default false
   * @see #destroy()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#property-destroyed Original Ext JS documentation of 'destroyed'
   */
  override public native function get destroyed():Boolean;

  [ExtConfig]
  /**
   * The unique identifier for this store. This config is required to scope this storage
   * distinctly from others. Ultimately, this is used to set a prefix on all keys.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#cfg-id Original Ext JS documentation of 'id'
   */
  public native function get id():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set id(value:String):void;

  [ExtConfig]
  /**
   * Determines if the keys collection is continuously maintained by this object. By
   * default the keys array is lazily fetched from the underlying store and when keys
   * are removed, the array is discarded. This heuristic tends to be safer than doing
   * the linear removal and array rippling to remove keys from the array on each call
   * to <code>→removeItem()</code>. If the cost of scanning <code>localStorage</code> for keys is high enough
   * and if the keys are frequently needed, then this flag can be set to <code>false</code> to
   * instruct this class to maintain the keys array once it has been determined.
   * @default true
   * @see #removeItem()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#cfg-lazyKeys Original Ext JS documentation of 'lazyKeys'
   */
  public native function get lazyKeys():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set lazyKeys(value:Boolean):void;

  [ExtConfig]
  /**
   * The prefix to apply to all <code>localStorage</code> keys manages by this instance. This does
   * not apply to the legacy IE mechanism but only to the HTML5 <code>localStorage</code> keys. If
   * not provided, the <code>→id</code> property initializes this value with <code>"id-"</code>.
   * @default ''
   * @see #id
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#cfg-prefix Original Ext JS documentation of 'prefix'
   */
  public native function get prefix():String;

  [ExtConfig]
  /**
   * @private
   */
  public native function set prefix(value:String):void;

  [ExtConfig]
  /**
   * Specify this as <code>true</code> to use <code>sessionStorage</code> instead of the default <code>localStoreage</code>.
   * This option is not supported in legacy IE browsers (IE 6 and 7) and is ignored.
   * @default false
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#cfg-session Original Ext JS documentation of 'session'
   */
  public native function get session():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set session(value:Boolean):void;

  /**
   * Removes all of the keys of this storage.
   * <b>NOTE:</b> This method conforms to the standard HTML5 Storage interface.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#method-clear Original Ext JS documentation of 'clear'
   */
  public native function clear():void;

  /**
   * Destroys this instance and for legacy IE, ensures data is flushed to persistent
   * storage. This method should not be called directly on instances returned by the
   * <code>→get()</code> method. Call <code>→release()</code> instead for such instances.
   * <p><i>NOTE:</i> For non-legacy IE browsers, there is no harm in failing to call this
   * method. In legacy IE, however, failing to call this method can result in memory
   * leaks.</p>
   * @see #get()
   * @see #release()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#method-destroy Original Ext JS documentation of 'destroy'
   */
  override public native function destroy(...params):void;

  /**
   * Returns the value associated with the given <code>→key()</code>.
   * <b>NOTE:</b> This method conforms to the standard HTML5 Storage interface.
   * @param key The key.
   * @return The value associated with the given <code>→key()</code>.
   * @see #key()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#method-getItem Original Ext JS documentation of 'getItem'
   */
  public native function getItem(key:String):String;

  [ArrayElementType("String")]
  /**
   * Returns the keys for this storage.
   * @return The keys for this storage. This array should be considered as
   * readonly.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#method-getKeys Original Ext JS documentation of 'getKeys'
   */
  public native function getKeys():Array;

  /**
   * Returns the specified key given its <code>index</code>. These keys have the scoping prefix
   * removed so they match what was passed to <code>→setItem()</code>.
   * <b>NOTE:</b> This method conforms to the standard HTML5 Storage interface.
   * @param index The index of the desired key.
   * @return The key.
   * @see #setItem()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#method-key Original Ext JS documentation of 'key'
   */
  public native function key(index:Number):String;

  /**
   * Call this method when finished with an instance returned by <code>→get()</code> instead of calling
   * <code>→destroy()</code>. When the last shared use of this instance calls <code>→release()</code>, the <code>→destroy()</code>
   * method is called automatically.
   * <p><i>NOTE:</i> Failing to call this method will result in memory leaks.</p>
   * @see #get()
   * @see #destroy()
   * @see #release()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#method-release Original Ext JS documentation of 'release'
   */
  public native function release():void;

  /**
   * Removes the value associated with the given <code>→key()</code>.
   * <b>NOTE:</b> This method conforms to the standard HTML5 Storage interface.
   * @param key The key.
   * @see #key()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#method-removeItem Original Ext JS documentation of 'removeItem'
   */
  public native function removeItem(key:String):void;

  /**
   * Sets the value associated with the given <code>→key()</code>.
   * <b>NOTE:</b> This method conforms to the standard HTML5 Storage interface.
   * @param key The key.
   * @param value The new associated value for <code>→key()</code>.
   * @see #key()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.LocalStorage.html#method-setItem Original Ext JS documentation of 'setItem'
   */
  public native function setItem(key:String, value:String):void;
}
}