package ext.util {
import ext.Base;

[Native]
/**
 * This class is a centralized place for formatting functions. It includes
 * functions to format various different types of data, such as text, dates and numeric values.
 * <p><b><i>Localization</i></b></p>
 * <p>This class contains several options for localization. These can be set once the library
 * has loaded, all calls to the functions from that point will use the locale settings
 * that were specified.</p>
 * <p>Options include:</p>
 * <ul>
 * <li>thousandSeparator</li>
 * <li>decimalSeparator</li>
 * <li>currencyPrecision</li>
 * <li>currencySign</li>
 * <li>currencyAtEnd</li>
 * </ul>
 * <p>This class also uses the default date format defined here: →<code>ext.DateUtil.defaultFormat</code>.</p>
 * <p><b><i>Using with renderers</i></b></p>
 * <p>There are two helper functions that return a new function that can be used in conjunction with
 * grid renderers:</p>
 * <pre>
 * columns: [{
 *     dataIndex: 'date',
 *     renderer: Ext.util.Format.dateRenderer('Y-m-d')
 * }, {
 *     dataIndex: 'time',
 *     renderer: Ext.util.Format.numberRenderer('0.000')
 * }]
 * </pre>
 * <p>Functions that only take a single argument can also be passed directly:</p>
 * <pre>
 * columns: [{
 *     dataIndex: 'cost',
 *     renderer: Ext.util.Format.usMoney
 * }, {
 *     dataIndex: 'productCode',
 *     renderer: Ext.util.Format.uppercase
 * }]
 * </pre>
 * <p><b><i>Using with XTemplates</i></b></p>
 * <p>XTemplates can also directly use Ext.util.Format functions:</p>
 * <pre>
 * new Ext.XTemplate([
 *     'Date: {startDate:date("Y-m-d")}',
 *     'Cost: {cost:usMoney}'
 * ]);
 * </pre>
 * <p>Type of singleton Format.</p>
 * @see ext.SDateUtil#defaultFormat ext.DateUtil.defaultFormat
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html Original Ext JS documentation of 'Ext.util.Format'
 * @see ext.util.#Format ext.util.Format
 */
public class SFormat extends Base {
  /**
   * This may be set to <code>true</code> to make the →<code>currency()</code> function
   * append the currency sign to the formatted value.
   * <p>This may be overridden in a locale file.</p>
   * @default false
   * @see SFormat#currency()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#property-currencyAtEnd Original Ext JS documentation of 'currencyAtEnd'
   */
  public native function get currencyAtEnd():Boolean;

  /**
   * @private
   */
  public native function set currencyAtEnd(value:Boolean):void;

  /**
   * The number of decimal places that the →<code>currency()</code> function displays.
   * <p>This may be overridden in a locale file.</p>
   * @default 2
   * @see SFormat#currency()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#property-currencyPrecision Original Ext JS documentation of 'currencyPrecision'
   */
  public native function get currencyPrecision():Number;

  /**
   * @private
   */
  public native function set currencyPrecision(value:Number):void;

  /**
   * The currency sign that the →<code>currency()</code> function displays.
   * <p>This may be overridden in a locale file.</p>
   * @default '$'
   * @see SFormat#currency()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#property-currencySign Original Ext JS documentation of 'currencySign'
   */
  public native function get currencySign():String;

  /**
   * @private
   */
  public native function set currencySign(value:String):void;

  /**
   * True to add a space between the currency and the value.
   * <p>This may be overridden in a locale file.</p>
   * @default ''
   * @since 6.2.0
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#property-currencySpacer Original Ext JS documentation of 'currencySpacer'
   */
  public native function get currencySpacer():String;

  /**
   * @private
   */
  public native function set currencySpacer(value:String):void;

  /**
   * The character that the →<code>number()</code> function uses as a decimal point.
   * <p>This may be overridden in a locale file.</p>
   * @default '.'
   * @see SFormat#number()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#property-decimalSeparator Original Ext JS documentation of 'decimalSeparator'
   */
  public native function get decimalSeparator():String;

  /**
   * @private
   */
  public native function set decimalSeparator(value:String):void;

  /**
   * The global default date format.
   * @default 'm/d/Y'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#property-defaultDateFormat Original Ext JS documentation of 'defaultDateFormat'
   */
  public native function get defaultDateFormat():*;

  /**
   * @private
   */
  public native function set defaultDateFormat(value:*):void;

  /**
   * The percent sign that the →<code>percent()</code> function displays.
   * <p>This may be overridden in a locale file.</p>
   * @default '%'
   * @see SFormat#percent()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#property-percentSign Original Ext JS documentation of 'percentSign'
   */
  public native function get percentSign():String;

  /**
   * @private
   */
  public native function set percentSign(value:String):void;

  /**
   * The character that the →<code>number()</code> function uses as a thousand separator.
   * <p>This may be overridden in a locale file.</p>
   * @default ','
   * @see SFormat#number()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#property-thousandSeparator Original Ext JS documentation of 'thousandSeparator'
   */
  public native function get thousandSeparator():String;

  /**
   * @private
   */
  public native function set thousandSeparator(value:String):void;

  /**
   * Formats an object of name value properties as HTML element attribute values
   * suitable for using when creating textual markup.
   * @param attributes An object containing the HTML attributes as properties
   * e.g.: <code>{height:40, vAlign:'top'}</code>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-attributes Original Ext JS documentation of 'attributes'
   */
  public native function attributes(attributes:Object):void;

  /**
   * Capitalize the first letter of the given string.
   * Alias for →<code>ext.StringUtil.capitalize()</code>.
   * @see ext.SStringUtil#capitalize() ext.StringUtil.capitalize()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-capitalize Original Ext JS documentation of 'capitalize'
   */
  public native function capitalize(string:String):String;

  /**
   * Format a number as a currency.
   * @param value The numeric value to format
   * @param currencySign The currency sign to use (defaults to
   * →<code>currencySign</code>)
   * @param decimals The number of decimals to use for the currency
   * (defaults to →<code>currencyPrecision</code>)
   * @param end True if the currency sign should be at the end of the string
   * (defaults to →<code>currencyAtEnd</code>)
   * @param currencySpacer True to add a space between the currency and value
   * @return The formatted currency string
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-currency Original Ext JS documentation of 'currency'
   * @see SFormat#currencySign
   * @see SFormat#currencyPrecision
   * @see SFormat#currencyAtEnd
   */
  public native function currency(value:*, currencySign:String = null, decimals:Number = NaN, end:Boolean = false, currencySpacer:String = null):String;

  /**
   * Formats the passed date using the specified format pattern.
   * Note that this uses the native Javascript Date.parse() method and is therefore subject
   * to its idiosyncrasies. Most formats assume the local timezone unless specified.
   * One notable exception is 'YYYY-MM-DD' (note the dashes) which is typically interpreted
   * in UTC and can cause date shifting.
   * @param value The value to format. Strings must conform to the format
   * expected by the JavaScript Date object's
   * <a href="http://developer.mozilla.org/en-US/docs/JavaScript/Reference/Global_Objects/Date/parse">parse() method</a>.
   * @param format Any valid date format string. Defaults to
   * →<code>ext.DateUtil.defaultFormat</code>.
   * @return The formatted date string.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-date Original Ext JS documentation of 'date'
   * @see ext.SDateUtil#defaultFormat ext.DateUtil.defaultFormat
   */
  public native function date(value:*, format:String = null):String;

  /**
   * Returns a date rendering function that can be reused to apply a date format multiple
   * times efficiently.
   * @param format Any valid date format string. Defaults to
   * →<code>ext.DateUtil.defaultFormat</code>.
   * @return The date formatting function
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-dateRenderer Original Ext JS documentation of 'dateRenderer'
   * @see ext.SDateUtil#defaultFormat ext.DateUtil.defaultFormat
   */
  public native function dateRenderer(format:String):Function;

  /**
   * Checks a reference and converts it to the default value if it's empty.
   * @param value Reference to check
   * @param defaultValue The value to insert of it's undefined.
   * @default ""
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-defaultValue Original Ext JS documentation of 'defaultValue'
   */
  public native function defaultValue(value:Object, defaultValue:String = ""):String;

  /**
   * Truncate a string and add an ellipsis ('...') to the end if it exceeds
   * the specified length.
   * Alias for →<code>ext.StringUtil.ellipsis()</code>.
   * @param value The string to truncate.
   * @param length The maximum length to allow before truncating.
   * @param word <code>true</code> to try to find a common word break.
   * @default false
   * @return The converted text.
   * @see ext.SStringUtil#ellipsis() ext.StringUtil.ellipsis()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-ellipsis Original Ext JS documentation of 'ellipsis'
   */
  public native function ellipsis(value:String, length:Number, word:Boolean = false):String;

  /**
   * Escapes the passed string for ' and .
   * Alias for →<code>ext.StringUtil.escape()</code>.
   * @param string The string to escape.
   * @return The escaped string.
   * @see ext.SStringUtil#escape() ext.StringUtil.escape()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-escape Original Ext JS documentation of 'escape'
   */
  public native function escape(string:String):String;

  /**
   * Escapes the passed string for use in a regular expression.
   * Alias for →<code>ext.StringUtil.escapeRegex()</code>.
   * @param string The string to escape.
   * @return The escaped string.
   * @see ext.SStringUtil#escapeRegex() ext.StringUtil.escapeRegex()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-escapeRegex Original Ext JS documentation of 'escapeRegex'
   */
  public native function escapeRegex(string:String):String;

  /**
   * Simple format for a file size (xxx bytes, xxx KB, xxx MB).
   * @param size The numeric value to format
   * @return The formatted file size
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-fileSize Original Ext JS documentation of 'fileSize'
   */
  public native function fileSize(size:*):String;

  /**
   * Allows you to define a tokenized string and pass an arbitrary number of arguments to replace
   * the tokens. Each token must be unique, and must increment in the format {0}, {1}, etc.
   * Example usage:
   * <pre>
   * var cls = 'my-class',
   *     text = 'Some text';
   * var s = Ext.String.format('&lt;div class="{0}"&gt;{1}&lt;/div&gt;', cls, text);
   * // s now contains the string: '&lt;div class="my-class"&gt;Some text&lt;/div&gt;'
   * </pre>
   * <p>Alias for →<code>ext.StringUtil.format()</code>.</p>
   * @param format The tokenized string to be formatted.
   * @param values The values to replace tokens <code>{0}</code>, <code>{1}</code>, etc in order.
   * @return The formatted string.
   * @see ext.SStringUtil#format() ext.StringUtil.format()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-format Original Ext JS documentation of 'format'
   */
  public native function format(format:String, ...values):String;

  /**
   * Returns the given number as a base 16 string at least <code>digits</code> in length. If
   * the number is fewer digits, 0's are prepended as necessary. If <code>digits</code> is
   * negative, the absolute value is the <i>exact</i> number of digits to return. In this
   * case, if then number has more digits, only the least significant digits are
   * returned.
   * <pre>
   *  expect(Ext.util.Format.hex(0x12e4, 2)).toBe('12e4');
   *  expect(Ext.util.Format.hex(0x12e4, -2)).toBe('e4');
   *  expect(Ext.util.Format.hex(0x0e, 2)).toBe('0e');
   * </pre>
   * @param value The number to format in hex.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-hex Original Ext JS documentation of 'hex'
   */
  public native function hex(value:Number, digits:Number):String;

  /**
   * Convert certain characters (&amp;, &lt;, &gt;, ', and ") from their HTML character equivalents.
   * Alias for →<code>ext.StringUtil.htmlDecode()</code>.
   * @param value The string to decode.
   * @return The decoded text.
   * @see ext.SStringUtil#htmlDecode() ext.StringUtil.htmlDecode()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-htmlDecode Original Ext JS documentation of 'htmlDecode'
   */
  public native function htmlDecode(value:String):String;

  /**
   * Convert certain characters (&amp;, &lt;, &gt;, ', and ") to their HTML character equivalents
   * for literal display in web pages.
   * Alias for →<code>ext.StringUtil.htmlEncode()</code>.
   * @param value The string to encode.
   * @return The encoded text.
   * @see ext.SStringUtil#htmlEncode() ext.StringUtil.htmlEncode()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-htmlEncode Original Ext JS documentation of 'htmlEncode'
   */
  public native function htmlEncode(value:String):String;

  /**
   * Pads the left side of a string with a specified character. This is especially useful
   * for normalizing number and date strings. Example usage:
   * <pre>
   * var s = Ext.String.leftPad('123', 5, '0');
   * // s now contains the string: '00123'
   * </pre>
   * <p>Alias for →<code>ext.StringUtil.leftPad()</code>.</p>
   * @param string The original string.
   * @param size The total length of the output string.
   * @param character The character with which to pad the original
   * string.
   * @default ' '
   * @return The padded string.
   * @see ext.SStringUtil#leftPad() ext.StringUtil.leftPad()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-leftPad Original Ext JS documentation of 'leftPad'
   */
  public native function leftPad(string:String, size:Number, character:String = ' '):String;

  /**
   * Compares <code>value</code> against <code>threshold</code> and returns...
   * <ul>
   * <li>if <code>value</code> &lt; <code>threshold</code> then it returns <code>below</code></li>
   * <li>if <code>value</code> &gt; <code>threshold</code> then it returns <code>above</code></li>
   * <li>if <code>value</code> = <code>threshold</code> then it returns <code>equal</code> or <code>above</code> when <code>equal</code> is missing</li>
   * </ul>
   * <p>The usefulness of this formatter method is in templates. For example:</p>
   * <pre>
   *  {foo:lessThanElse(0, 'negative', 'positive')}
   *
   *  {bar:lessThanElse(200, 'lessThan200', 'greaterThan200', 'equalTo200')}
   * </pre>
   * @param value Value that will be checked
   * @param threshold Value to compare against
   * @param below Value to return when <code>value</code> &lt; <code>threshold</code>
   * @param above Value to return when <code>value</code> &gt; <code>threshold</code>.
   * If <code>value</code> = <code>threshold</code> and <code>equal</code> is missing then <code>above</code> is returned.
   * @param equal Value to return when <code>value</code> = <code>threshold</code>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-lessThanElse Original Ext JS documentation of 'lessThanElse'
   */
  public native function lessThanElse(value:Number, threshold:Number, below:*, above:*, equal:*):*;

  /**
   * Converts a string to all lower case letters.
   * @param value The text to convert
   * @return The converted text
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-lowercase Original Ext JS documentation of 'lowercase'
   */
  public native function lowercase(value:String):String;

  /**
   * It does simple math for use in a template, for example...
   * <pre>
   * var tpl = new Ext.Template('{value} &#42; 10 = {value:math("&#42; 10")}');
   * </pre>
   * @return A function that operates on the passed value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-math Original Ext JS documentation of 'math'
   */
  public native function math():Function;

  /**
   * Returns a non-breaking space ("NBSP") for any "blank" value.
   * @since 6.2.0
   * @param value
   * @param strict Pass <code>false</code> to convert all falsey values to an
   * NBSP. By default, only '', <code>null</code> and <code>undefined</code> will be converted.
   * @default true
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-nbsp Original Ext JS documentation of 'nbsp'
   */
  public native function nbsp(value:*, strict:Boolean = true):*;

  /**
   * Converts newline characters to the HTML tag <code>&lt;br/&gt;</code>
   * @param v The string value to format.
   * @return The string with embedded <code>&lt;br/&gt;</code> tags in place of newlines.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-nl2br Original Ext JS documentation of 'nl2br'
   */
  public native function nl2br(v:String):String;

  /**
   * Formats the passed number according to the passed format string.
   * <p>The number of digits after the decimal separator character specifies the number of
   * decimal places in the resulting string. The <i>local-specific</i> decimal character is
   * used in the result.</p>
   * <p>The <i>presence</i> of a thousand separator character in the format string specifies that
   * the <i>locale-specific</i> thousand separator (if any) is inserted separating thousand groups.</p>
   * <p>By default, "," is expected as the thousand separator, and "." is expected as the decimal
   * separator.</p>
   * <p>Locale-specific characters are always used in the formatted output when inserting
   * thousand and decimal separators. These can be set using the →<code>thousandSeparator</code>
   * and →<code>decimalSeparator</code> options.</p>
   * <p>The format string must specify separator characters according to US/UK conventions
   * ("," as the thousand separator, and "." as the decimal separator)</p>
   * <p>To allow specification of format strings according to local conventions for separator
   * characters, add the string <code>/i</code> to the end of the format string. This format depends
   * on the →<code>thousandSeparator</code> and →<code>decimalSeparator</code> options. For example,
   * if using European style separators, then the format string can be specified
   * as <code>'0.000,00'</code>. This would be equivalent to using <code>'0,000.00'</code> when using US style
   * formatting.</p>
   * <p>Examples (123456.789):</p>
   * <ul>
   * <li><code>0</code> - (123457) show only digits, no precision</li>
   * <li><code>0.00</code> - (123456.79) show only digits, 2 precision</li>
   * <li><code>0.0000</code> - (123456.7890) show only digits, 4 precision</li>
   * <li><code>0,000</code> - (123,457) show comma and digits, no precision</li>
   * <li><code>0,000.00</code> - (123,456.79) show comma and digits, 2 precision</li>
   * <li><code>0,0.00</code> - (123,456.79) shortcut method, show comma and digits, 2 precision</li>
   * <li><code>0.####</code> - (123,456.789) Allow maximum 4 decimal places, but do not right pad
   * with zeroes</li>
   * <li><code>0.00##</code> - (123456.789) Show at least 2 decimal places, maximum 4, but do not
   * right pad with zeroes</li>
   * </ul>
   * @param v The number to format.
   * @param formatString The way you would like to format this text.
   * @return The formatted number.
   * @see SFormat#thousandSeparator
   * @see SFormat#decimalSeparator
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-number Original Ext JS documentation of 'number'
   */
  public native function number(v:Number, formatString:String):String;

  /**
   * Returns a number rendering function that can be reused to apply a number format multiple
   * times efficiently.
   * @param format Any valid number format string for →<code>number()</code>
   * @return The number formatting function
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-numberRenderer Original Ext JS documentation of 'numberRenderer'
   * @see SFormat#number()
   */
  public native function numberRenderer(format:String):Function;

  /**
   * Returns this result...
   * <pre>
   *  value || orValue
   * </pre>
   * <p>The usefulness of this formatter method is in templates. For example:</p>
   * <pre>
   *  {foo:or("bar")}
   * </pre>
   * @param value The "if" value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-or Original Ext JS documentation of 'or'
   */
  public native function or(value:Boolean, orValue:*):void;

  /**
   * Parses a number or string representing margin sizes into an object.
   * Supports CSS-style margin declarations (e.g. 10, "10", "10 10", "10 10 10" and
   * "10 10 10 10" are all valid options and would return the same result).
   * @param box The encoded margins
   * @return An object with margin sizes for top, right, bottom and left
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-parseBox Original Ext JS documentation of 'parseBox'
   */
  public native function parseBox(box:*):Object;

  /**
   * Formats the passed number as a percentage according to the passed format string.
   * The number should be between 0 and 1 to represent 0% to 100%.
   * @param value The percentage to format.
   * @param formatString See →<code>number()</code> for details.
   * @default "0"
   * @return The formatted percentage.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-percent Original Ext JS documentation of 'percent'
   * @see SFormat#number()
   */
  public native function percent(value:Number, formatString:String = "0"):String;

  /**
   * If <code>value</code> is a number, returns the argument from that index. For example
   * <pre>
   *  var s = Ext.util.Format.pick(2, 'zero', 'one', 'two');
   *  // s === 'two'
   * </pre>
   * <p>Otherwise, <code>value</code> is treated in a truthy/falsey manner like so:</p>
   * <pre>
   *  var s = Ext.util.Format.pick(null, 'first', 'second');
   *  // s === 'first'
   *
   *  s = Ext.util.Format.pick({}, 'first', 'second');
   *  // s === 'second'
   * </pre>
   * <p>The usefulness of this formatter method is in templates. For example:</p>
   * <pre>
   *  {foo:pick("F","T")}
   *
   *  {bar:pick("first","second","third")}
   * </pre>
   * @param value The "if" value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-pick Original Ext JS documentation of 'pick'
   */
  public native function pick(value:Boolean, firstValue:*, secondValue:*):void;

  /**
   * Selectively return the plural form of a word based on a numeric value.
   * <p>For example, the following template would result in "1 Comment". If the
   * value of <code>count</code> was 0 or greater than 1, the result would be "x Comments".</p>
   * <pre>
   * var tpl = new Ext.XTemplate('{count:plural("Comment")}');
   *
   * tpl.apply({
   *     count: 1
   * }); // returns "1 Comment"
   * </pre>
   * <p>Examples using the static <code>→plural()</code> method call:</p>
   * <pre>
   * Ext.util.Format.plural(2, 'Comment');
   * // returns "2 Comments"
   *
   * Ext.util.Format.plural(4, 'person', 'people');
   * // returns "4 people"
   * </pre>
   * @param value The value to compare against
   * @param singular The singular form of the word
   * @param plural The plural form of the word (defaults to the
   * singular form with an "s" appended)
   * @return output The pluralized output of the passed singular form
   * @see SFormat#plural()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-plural Original Ext JS documentation of 'plural'
   */
  public native function plural(value:Number, singular:String, plural:String = null):String;

  /**
   * Resolves the specified resource <code>url</code> with an optional <code>prefix</code>. This resolution
   * is based on →<code>ext.Ext.resolveResource()</code>. The prefix is intended to be used for
   * a package or resource pool identifier.
   * @param url The resource url to resolve
   * @param prefix A prefix/identifier to include in the resolution.
   * @see ext.SExt#resolveResource() ext.Ext.resolveResource()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-resource Original Ext JS documentation of 'resource'
   */
  public native function resource(url:String, prefix:String = null):String;

  /**
   * Rounds the passed number to the required decimal precision.
   * @param value The numeric value to round.
   * @param precision The number of decimal places to which to round the
   * first parameter's value. If <code>undefined</code> the <code>value</code> is passed to <code>Math.round</code>
   * otherwise the value is returned unmodified.
   * @return The rounded value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-round Original Ext JS documentation of 'round'
   */
  public native function round(value:*, precision:Number = NaN):Number;

  /**
   * Checks if <code>value</code> is a positive or negative number and returns the proper param.
   * <p>The usefulness of this formatter method is in templates. For example:</p>
   * <pre>
   *  {foo:sign("clsNegative","clsPositive")}
   * </pre>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-sign Original Ext JS documentation of 'sign'
   */
  public native function sign(value:Number, negative:*, positive:*, zero:*):*;

  /**
   * Strips all script tags.
   * @param value The text from which to strip script tags
   * @return The stripped text
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-stripScripts Original Ext JS documentation of 'stripScripts'
   */
  public native function stripScripts(value:Object):String;

  /**
   * Strips all HTML tags.
   * @param value The text from which to strip tags
   * @return The stripped text
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-stripTags Original Ext JS documentation of 'stripTags'
   */
  public native function stripTags(value:Object):String;

  /**
   * Returns a substring from within an original string.
   * @param value The original text
   * @param start The start index of the substring
   * @param length The length of the substring
   * @return The substring
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-substr Original Ext JS documentation of 'substr'
   */
  public native function substr(value:String, start:Number, length:Number):String;

  /**
   * Utility function that allows you to easily switch a string between two alternating
   * values. The passed value is compared to the current string, and if they are equal,
   * the other value that was passed in is returned. If they are already different,
   * the first value passed in is returned. Note that this method returns the new value
   * but does not change the current string.
   * <pre>
   * // alternate sort directions
   * sort = Ext.String.toggle(sort, 'ASC', 'DESC');
   *
   * // instead of conditional logic:
   * sort = (sort === 'ASC' ? 'DESC' : 'ASC');
   * </pre>
   * <p>Alias for →<code>ext.StringUtil.toggle()</code>.</p>
   * @param string The current string.
   * @param value The value to compare to the current string.
   * @param other The new value to use if the string already equals the first value
   * passed in.
   * @return The new value.
   * @see ext.SStringUtil#toggle() ext.StringUtil.toggle()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-toggle Original Ext JS documentation of 'toggle'
   */
  public native function toggle(string:String, value:String, other:String):String;

  /**
   * Trims whitespace from either end of a string, leaving spaces within the string intact.
   * Example:
   * <pre>
   * var s = '  foo bar  ';
   * alert('-' + s + '-');                   //alerts "- foo bar -"
   * alert('-' + Ext.String.trim(s) + '-');  //alerts "-foo bar-"
   * </pre>
   * <p>Alias for →<code>ext.StringUtil.trim()</code>.</p>
   * @param string The string to trim.
   * @return The trimmed string.
   * @see ext.SStringUtil#trim() ext.StringUtil.trim()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-trim Original Ext JS documentation of 'trim'
   */
  public native function trim(string:String):String;

  /**
   * Uncapitalize the first letter of a given string.
   * Alias for →<code>ext.StringUtil.uncapitalize()</code>.
   * @see ext.SStringUtil#uncapitalize() ext.StringUtil.uncapitalize()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-uncapitalize Original Ext JS documentation of 'uncapitalize'
   */
  public native function uncapitalize(string:String):String;

  /**
   * Checks a reference and converts it to empty string if it is undefined.
   * @param value Reference to check
   * @return Empty string if converted, otherwise the original value
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-undef Original Ext JS documentation of 'undef'
   */
  public native function undef(value:Object):Object;

  /**
   * Converts a string to all upper case letters.
   * @param value The text to convert
   * @return The converted text
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-uppercase Original Ext JS documentation of 'uppercase'
   */
  public native function uppercase(value:String):String;

  /**
   * Formats the given value using <code>encodeURI</code>.
   * @since 6.2.0
   * @param value The value to encode.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-uri Original Ext JS documentation of 'uri'
   */
  public native function uri(value:String):String;

  /**
   * Formats the given value using <code>encodeURIComponent</code>.
   * @since 6.2.0
   * @param value The value to encode.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-uriCmp Original Ext JS documentation of 'uriCmp'
   */
  public native function uriCmp(value:String):String;

  /**
   * Format a number as US currency.
   * @param value The numeric value to format
   * @return The formatted currency string
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-usMoney Original Ext JS documentation of 'usMoney'
   */
  public native function usMoney(value:*):String;

  /**
   * Returns the word at the given <code>index</code>. Spaces and punctuation are considered
   * as word separators by default. For example:
   * <pre>
   *  console.log(Ext.util.Format.word('Hello, my name is Bob.', 2);
   *  // == 'name'
   * </pre>
   * @param value The sentence to break into words.
   * @param index The 0-based word index.
   * @param sep The pattern by which to separate words.
   * @default "[\W\s]+"
   * @return The requested word or empty string.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Format.html#method-word Original Ext JS documentation of 'word'
   */
  public native function word(value:String, index:Number, sep:* = "[\\W\\s]+"):String;
}
}