package ext.util {
import ext.Base;

[Native]
/**
 * General purpose inflector class that <i>pluralizes</i> (→<code>pluralize()</code>),
 * <i>singularizes</i> (→<code>singularize()</code>) and <i>ordinalizes</i> (→<code>ordinalize()</code>) words. Sample usage:
 * <pre>
 * //turning singular words into plurals
 * Ext.util.Inflector.pluralize('word'); //'words'
 * Ext.util.Inflector.pluralize('person'); //'people'
 * Ext.util.Inflector.pluralize('sheep'); //'sheep'
 *
 * //turning plurals into singulars
 * Ext.util.Inflector.singularize('words'); //'word'
 * Ext.util.Inflector.singularize('people'); //'person'
 * Ext.util.Inflector.singularize('sheep'); //'sheep'
 *
 * //ordinalizing numbers
 * Ext.util.Inflector.ordinalize(11); //"11th"
 * Ext.util.Inflector.ordinalize(21); //"21st"
 * Ext.util.Inflector.ordinalize(1043); //"1043rd"
 * </pre>
 * <p><b>Customization</b></p>
 * <p>The Inflector comes with a default set of US English pluralization rules. These can be augmented
 * with additional rules if the default rules do not meet your application's requirements,
 * or swapped out entirely for other languages. Here is how we might add a rule that pluralizes
 * "ox" to "oxen":</p>
 * <pre>
 * Ext.util.Inflector.plural(/^(ox)$/i, "$1en");
 * </pre>
 * <p>Each rule consists of two items - a regular expression that matches one or more rules,
 * and a replacement string. In this case, the regular expression will only match the string "ox",
 * and will replace that match with "oxen". Here's how we could add the inverse rule:</p>
 * <pre>
 * Ext.util.Inflector.singular(/^(ox)en$/i, "$1");
 * </pre>
 * <p>Note that the ox/oxen rules are present by default.</p>
 * <p>Type of singleton Inflector.</p>
 * @see SInflector#pluralize()
 * @see SInflector#singularize()
 * @see SInflector#ordinalize()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Inflector.html Original Ext JS documentation of 'Ext.util.Inflector'
 * @see ext.util.#Inflector ext.util.Inflector
 */
public class SInflector extends Base {
  /**
   * Returns the correct →<code>ext.data.Model</code> name for a given string. Mostly used
   * internally by the data
   * package
   * @param word The word to classify
   * @return The classified version of the word
   * @see ext.data.Model
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Inflector.html#method-classify Original Ext JS documentation of 'classify'
   */
  public native function classify(word:String):String;

  /**
   * Removes all registered pluralization rules
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Inflector.html#method-clearPlurals Original Ext JS documentation of 'clearPlurals'
   */
  public native function clearPlurals():void;

  /**
   * Removes all registered singularization rules
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Inflector.html#method-clearSingulars Original Ext JS documentation of 'clearSingulars'
   */
  public native function clearSingulars():void;

  /**
   * Returns true if the given word is transnumeral (the word is its own singular and
   * plural form - e.g. sheep, fish)
   * @param word The word to test
   * @return True if the word is transnumeral
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Inflector.html#method-isTransnumeral Original Ext JS documentation of 'isTransnumeral'
   */
  public native function isTransnumeral(word:String):Boolean;

  /**
   * Ordinalizes a given number by adding a prefix such as 'st', 'nd', 'rd' or 'th' based on
   * the last digit of the number. 21 -&gt; 21st, 22 -&gt; 22nd, 23 -&gt; 23rd, 24 -&gt; 24th etc
   * @param number The number to ordinalize
   * @return The ordinalized number
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Inflector.html#method-ordinalize Original Ext JS documentation of 'ordinalize'
   */
  public native function ordinalize(number:Number):String;

  /**
   * Adds a new pluralization rule to the Inflector. See the intro docs for more information
   * @param matcher The matcher regex
   * @param replacer The replacement string, which can reference matches from the matcher
   * argument
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Inflector.html#method-plural Original Ext JS documentation of 'plural'
   */
  public native function plural(matcher:RegExp, replacer:String):void;

  /**
   * Returns the pluralized form of a word (e.g. Ext.util.Inflector.pluralize('word')
   * returns 'words')
   * @param word The word to pluralize
   * @return The pluralized form of the word
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Inflector.html#method-pluralize Original Ext JS documentation of 'pluralize'
   */
  public native function pluralize(word:String):String;

  /**
   * Adds a new singularization rule to the Inflector. See the intro docs for more information
   * @param matcher The matcher regex
   * @param replacer The replacement string, which can reference matches from the matcher
   * argument
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Inflector.html#method-singular Original Ext JS documentation of 'singular'
   */
  public native function singular(matcher:RegExp, replacer:String):void;

  /**
   * Returns the singularized form of a word (e.g. Ext.util.Inflector.singularize('words')
   * returns 'word')
   * @param word The word to singularize
   * @return The singularized form of the word
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.Inflector.html#method-singularize Original Ext JS documentation of 'singularize'
   */
  public native function singularize(word:String):String;
}
}