package ext.util {
import ext.Base;

[Native("Ext.util.TaskRunner", require)]
/**
 * Provides the ability to execute one or more arbitrary tasks in an asynchronous manner.
 * <p>Generally, you can use the singleton →<code>ext.util.TaskManager</code>. Or you can create
 * separate TaskRunner instances to start and stop unique tasks independent of one
 * another.</p>
 * <p>Example usage:</p>
 * <p><b>Example:</b></p>
 * <pre>
 * var runner = new Ext.util.TaskRunner(),
 *     clock, updateClock, task;
 *
 * clock = Ext.getBody().appendChild({
 *     id: 'clock'
 * });
 *
 * // Start a simple clock task that updates a div once per second
 * updateClock = function() {
 *     clock.setHtml(Ext.Date.format(new Date(), 'g:i:s A'));
 * };
 *
 * task = runner.start({
 *     run: updateClock,
 *     interval: 1000
 * });
 * </pre>
 * <p>The equivalent using TaskManager:</p>
 * <p><b>Example:</b></p>
 * <pre>
 * var clock, updateClock, task;
 *
 * clock = Ext.getBody().appendChild({
 *     id: 'clock'
 * });
 *
 * // Start a simple clock task that updates a div once per second
 * updateClock = function() {
 *     clock.setHtml(Ext.Date.format(new Date(), 'g:i:s A'));
 * };
 *
 * var task = Ext.TaskManager.start({
 *     run: updateClock,
 *     interval: 1000
 * });
 * </pre>
 * <p>To end a running task:</p>
 * <pre>
 *  task.destroy();
 * </pre>
 * <p>If a task needs to be started and stopped repeated over time, you can create a
 * →<code>ext.util.TaskRunnerTask</code> instance.</p>
 * <pre>
 * var runner = new Ext.util.TaskRunner(),
 *     task;
 *
 * task = runner.newTask({
 *     run: function() {
 *         // useful code
 *     },
 *     interval: 1000
 * });
 *
 * task.start();
 *
 * // ...
 *
 * task.stop();
 *
 * // ...
 *
 * task.start();
 * </pre>
 * <p>A re-usable, single-run task can be managed similar to the above:</p>
 * <pre>
 * var runner = new Ext.util.TaskRunner(),
 *     task;
 *
 * task = runner.newTask({
 *     run: function() {
 *         // useful code
 *     },
 *     interval: 1000,
 *     repeat: 1
 * });
 *
 * task.start();
 *
 * // ...
 *
 * task.stop();
 *
 * // ...
 *
 * task.start();
 * </pre>
 * <p>See the →<code>start()</code> method for details about how to configure a Task.</p>
 * <p>Also see →<code>ext.util.DelayedTask</code>.</p>
 * @see ext.util.#TaskManager ext.util.TaskManager
 * @see ext.util.TaskRunnerTask
 * @see #start()
 * @see ext.util.DelayedTask
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TaskRunner.html Original Ext JS documentation of 'Ext.util.TaskRunner'
 */
public class TaskRunner extends Base {
  /**
   * @param interval The minimum precision in milliseconds supported by
   * this TaskRunner instance. Alternatively, a config object to apply to the new instance.
   * @default 10
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TaskRunner.html#method-constructor Original Ext JS documentation of 'constructor'
   */
  public function TaskRunner(interval:Number = 10) {
    super();
  }

  [ExtConfig]
  /**
   * This may be configured <code>false</code> to inhibit firing of the <i>idle event</i> (→<code>ext.GlobalEvents.event:onIdle</code>) after task invocation. By default the tasks
   * run in a given tick determine whether <code>idle</code> events fire.
   * @default null
   * @see ext.SGlobalEvents#event:onIdle ext.GlobalEvents.event:onIdle
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TaskRunner.html#cfg-fireIdleEvent Original Ext JS documentation of 'fireIdleEvent'
   */
  public native function get fireIdleEvent():Boolean;

  [ExtConfig]
  /**
   * @private
   */
  public native function set fireIdleEvent(value:Boolean):void;

  [ExtConfig]
  /**
   * How often to run the task in milliseconds. Defaults to every 10ms.
   * @default 10
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TaskRunner.html#cfg-interval Original Ext JS documentation of 'interval'
   */
  public native function get interval():Number;

  [ExtConfig]
  /**
   * @private
   */
  public native function set interval(value:Number):void;

  /**
   * Destroys this instance, stopping all tasks that are currently running.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TaskRunner.html#method-destroy Original Ext JS documentation of 'destroy'
   */
  override public native function destroy(...params):void;

  /**
   * Creates a new →<code>ext.util.TaskRunnerTask</code> instance. These instances can
   * be easily started and stopped.
   * @param config The config object. For details on the supported properties,
   * see →<code>start()</code>.
   * @return Ext.util.TaskRunner.Task instance, which can be useful for method chaining.
   * @see ext.util.TaskRunnerTask
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TaskRunner.html#method-newTask Original Ext JS documentation of 'newTask'
   * @see #start()
   */
  public native function newTask(config:Object):TaskRunnerTask;

  /**
   * Starts a new task.
   * <p>Before each invocation, Ext injects the property <code>taskRunCount</code> into the task object
   * so that calculations based on the repeat count can be performed.</p>
   * <p>The returned task will contain a <code>→destroy()</code> method that can be used to destroy the
   * task and cancel further calls. This is equivalent to the →<code>stop()</code> method.</p>
   * @param task A config object that supports the following properties:
   * <ul>
   * <li><code>run:Function</code> (optional) —
   * The function to execute each time the task is invoked. The
   * function will be called at each interval and passed the <code>args</code> argument if specified,
   * and the current invocation count if not.
   * <p>If a particular scope (<code>this</code> reference) is required, be sure to specify it using
   * the <code>scope</code> argument.</p>
   * <ul>
   * <li><code>return:Boolean</code> (optional) —
   * <code>false</code> from this function to terminate the task.
   * </li>
   * </ul>
   * </li>
   * <li><code>onError:Function</code> (optional) —
   * The function to execute in case of unhandled
   * error on task.run.
   * </li>
   * <li><code>interval:Number</code> (optional) —
   * The frequency in milliseconds with which the task
   * should be invoked.
   * </li>
   * <li><code>args:Array</code> (optional) —
   * An array of arguments to be passed to the function
   * specified by <code>run</code>. If not specified, the current invocation count is passed.
   * </li>
   * <li><code>addCountToArgs:Boolean</code> (default = <code>false</code>) —
   * True to add the current invocation count as
   * one of the arguments of args.
   * Note: This only takes effect when args is specified.
   * </li>
   * <li><code>scope:Object</code> (optional) —
   * The scope (<code>this</code> reference) in which to execute the
   * <code>run</code> function. Defaults to the task config object.
   * </li>
   * <li><code>duration:Number</code> (optional) —
   * The length of time in milliseconds to invoke the task
   * before stopping automatically (defaults to indefinite).
   * </li>
   * <li><code>repeat:Number</code> (optional) —
   * The number of times to invoke the task before stopping
   * automatically (defaults to indefinite).
   * </li>
   * <li><code>fireIdleEvent:Number</code> (default = <code>true</code>) —
   * If all tasks in a TaskRunner's execution
   * sweep are configured with <code>fireIdleEvent: false</code>, then the
   * <i>idle event</i> (→<code>ext.GlobalEvents.event:onIdle</code>) is not fired when the TaskRunner's execution
   * sweep finishes.
   * </li>
   * <li><code>fireOnStart:Boolean</code> (default = <code>false</code>) —
   * True to run the task immediately instead of
   * waiting for the <i>interval's</i> initial pass to call the <i>run</i> function.
   * </li>
   * </ul>
   * @see #destroy()
   * @see #stop()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TaskRunner.html#method-start Original Ext JS documentation of 'start'
   * @see ext.SGlobalEvents#event:onIdle ext.GlobalEvents.event:onIdle
   */
  public native function start(task:Object):Object;

  /**
   * Stops an existing running task.
   * @param task The task to stop.
   * @param andRemove Pass <code>true</code> to also remove the task from the queue.
   * @return The task
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TaskRunner.html#method-stop Original Ext JS documentation of 'stop'
   */
  public native function stop(task:Object, andRemove:Boolean):Object;

  /**
   * Stops all tasks that are currently running.
   * @param andRemove Pass <code>true</code> to also remove the tasks from the queue.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TaskRunner.html#method-stopAll Original Ext JS documentation of 'stopAll'
   */
  public native function stopAll(andRemove:Boolean):void;
}
}