package ext.util {
import ext.Base;

[Native("Ext.util.TextMetrics", require)]
/**
 * Provides precise pixel measurements for blocks of text so that you can determine
 * the exact pixel height and width of a block of text.
 * <p><b>Note:</b> The TextMetrics tool should only be utilized to measure plain text. Attempting to
 * measure text that includes HTML may return inaccurate results.</p>
 * <p>This measurement works by copying the relevant font-related CSS styles from the element
 * param to the TextMetrics' cached measurement element. This returns the dimensions of the cached
 * element wrapping the text. By default, the wrapping element is auto-sized.
 * You must provide a <b>fixed width</b> if the passed text is multi-lined.</p>
 * <p>When multiple measurements are being done with the same element styling, you should
 * create a single, reusable TextMetrics instance. This is more efficient than using the
 * static →<code>measure()</code> method. The element styles are copied to the cached
 * TextMetrics element once during instantiation versus repeated copying using
 * <i>measure()</i>.</p>
 * <p>The following example demonstrates the recommended use of TextMetrics where the custom
 * textfield class sets up a reusable TextMetrics instance used to measure the label
 * width. This example assumes that all instances of <i>mytextfield</i> have the same
 * →<code>ext.form.Labelable.labelClsExtra</code> and
 * →<code>ext.form.Labelable.labelStyle</code> configs.</p>
 * <pre>
 * Ext.define('MyApp.view.MyTextField', {
 *     extend: 'Ext.form.field.Text',
 *     xtype: 'mytextfield',
 *
 *     initComponent: function () {
 *         var me = this,
 *             tm = me.getTextMetrics();
 *
 *         me.labelWidth = tm.getWidth(me.fieldLabel + me.labelSeparator);
 *         me.callParent();
 *     },
 *
 *     getTextMetrics: function () {
 *         var me = this,
 *             // Using me.self allows labelCls etc. to vary by derived
 *             // class, but not by instance.
 *             cls = me.self,
 *             tm = cls.measurer,
 *             el;
 *
 *         if (!tm) {
 *             el = Ext.getBody().createChild();
 *             el.addCls(me.labelCls + ' ' + me.labelClsExtra).
 *                 applyStyles(me.labelStyle);
 *
 *             cls.measurer = tm = new Ext.util.TextMetrics(el);
 *         }
 *
 *         return tm;
 *     }
 * });
 *
 * Ext.create('Ext.form.Panel', {
 *     title: 'Contact Info',
 *     width: 600,
 *     bodyPadding: 10,
 *     renderTo: Ext.getBody(),
 *     items: [{
 *         xtype: 'mytextfield',
 *         fieldLabel: 'Name',
 *         labelStyle: 'font-size: 10px;'
 *     }, {
 *         xtype: 'mytextfield',
 *         fieldLabel: 'Email Address',
 *         labelStyle: 'font-size: 10px;'
 *     }]
 * });
 * </pre>
 * <p>While less efficient than the preceding example, this example allows each instance of
 * <i>mytextfield2</i> to have unique labelClsExtra and labelStyle configs. Each custom
 * textfield instance uses the static TextMetrics measure method which will copy the
 * label styles repeatedly, thus being less efficient but more versatile.</p>
 * <pre>
 * Ext.define('MyApp.view.MyTextField2', {
 *     extend: 'Ext.form.field.Text',
 *     xtype: 'mytextfield2',
 *
 *     initComponent: function () {
 *         var me = this,
 *             el = me.getMeasurementEl(),
 *             tm = Ext.util.TextMetrics;
 *
 *         me.labelWidth = tm.measure(el, me.fieldLabel + me.labelSeparator).width;
 *         me.callParent();
 *     },
 *
 *     getMeasurementEl: function () {
 *         var me = this,
 *             cls = MyApp.view.MyTextField2,
 *             el = cls.measureEl;
 *
 *         if (!el) {
 *             cls.measureEl = el = Ext.getBody().createChild();
 *         }
 *
 *         el.dom.removeAttribute('style');
 *         el.removeCls(el.dom.className).
 *             addCls(me.labelCls + ' ' + me.labelClsExtra).
 *             applyStyles(me.labelStyle);
 *
 *         return el;
 *     }
 * });
 *
 * Ext.create('Ext.form.Panel', {
 *     title: 'Contact Info',
 *     width: 600,
 *     bodyPadding: 10,
 *     renderTo: Ext.getBody(),
 *     items: [{
 *         xtype: 'mytextfield2',
 *         fieldLabel: 'Name',
 *         labelStyle: 'font-size: 14px;font-weight: bold;',
 *         labelClsExtra: 'nameLabel'
 *     }, {
 *         xtype: 'mytextfield2',
 *         fieldLabel: 'Email Address',
 *         labelStyle: 'font-size: 10px;',
 *         labelClsExtra: 'emailLabel'
 *     }]
 * });
 * </pre>
 * @see #measure()
 * @see ext.form.Labelable#labelClsExtra
 * @see ext.form.Labelable#labelStyle
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TextMetrics.html Original Ext JS documentation of 'Ext.util.TextMetrics'
 */
public class TextMetrics extends Base {
  /**
   * Destroy the TextMetrics instance created by →<code>measure()</code>.
   * @see #measure()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TextMetrics.html#static-method-destroy Original Ext JS documentation of 'destroy'
   */
  public static native function destroy():void;

  /**
   * Measures the size of the specified text
   * @param el The element, dom node or id from which to copy existing
   * CSS styles that can affect the size of the rendered text
   * @param text The text to measure
   * @param fixedWidth If the text will be multiline, you have to set a fixed width
   * in order to accurately measure the text height
   * @return An object containing the text's size
   * <code>{width: (width), height: (height)}</code>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TextMetrics.html#static-method-measure Original Ext JS documentation of 'measure'
   */
  public static native function measure(el:*, text:String, fixedWidth:Number = NaN):Object;

  /**
   * Creates new TextMetrics.
   * @param bindTo The element or its ID to bind to.
   * @param fixedWidth A fixed width to apply to the measuring element.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TextMetrics.html#method-constructor Original Ext JS documentation of 'constructor'
   */
  public function TextMetrics(bindTo:*, fixedWidth:Number = NaN) {
    super();
  }

  /**
   * Binds this TextMetrics instance to a new element
   * @param el The element or its ID.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TextMetrics.html#method-bind Original Ext JS documentation of 'bind'
   */
  public native function bind(el:*):void;

  /**
   * Destroy this instance
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TextMetrics.html#method-destroy Original Ext JS documentation of 'destroy'
   */
  override public native function destroy(...params):void;

  /**
   * Returns the measured height of the specified text
   * @param text The text to measure
   * @return height The height in pixels
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TextMetrics.html#method-getHeight Original Ext JS documentation of 'getHeight'
   */
  public native function getHeight(text:String):Number;

  /**
   * Returns the size of the specified text based on the internal element's style and width
   * properties
   * @param text The text to measure
   * @return An object containing the text's size <code>{width: (width), height: (height)}</code>
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TextMetrics.html#method-getSize Original Ext JS documentation of 'getSize'
   */
  public native function getSize(text:String):Object;

  /**
   * Returns the measured width of the specified text
   * @param text The text to measure
   * @return width The width in pixels
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TextMetrics.html#method-getWidth Original Ext JS documentation of 'getWidth'
   */
  public native function getWidth(text:String):Number;

  /**
   * Sets a fixed width on the internal measurement element. If the text will be multiline,
   * you have to set a fixed width in order to accurately measure the text height.
   * @param width The width to set on the element
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.util.TextMetrics.html#method-setFixedWidth Original Ext JS documentation of 'setFixedWidth'
   */
  public native function setFixedWidth(width:Number):void;
}
}