package ext.window {
[Native("Ext.window.MessageBox", require)]
/**
 * Utility class for generating different styles of message boxes. The singleton instance,
 * Ext.MessageBox alias <code>→ext.MessageBox</code> can also be used.
 * <p>Note that a MessageBox is asynchronous. Unlike a regular JavaScript <code>→alert()</code> (which will halt
 * browser execution), showing a MessageBox will not cause the code to stop. For this reason,
 * if you have code that should only run <i>after</i> some user feedback from the MessageBox,
 * you must use a callback function (see the <code>function</code> parameter for →<code>show()</code>
 * for more details).</p>
 * <p>Basic alert</p>
 * <p><b>Example:</b></p>
 * <pre>
 * Ext.Msg.alert('Status', 'Changes saved successfully.');
 * </pre>
 * <p>Prompt for user data and process the result using a callback</p>
 * <p><b>Example:</b></p>
 * <pre>
 * Ext.Msg.prompt('Name', 'Please enter your name:', function(btn, text){
 *     if (btn == 'ok'){
 *         // process text value and close...
 *     }
 * });
 * </pre>
 * <p>Show a dialog using config options</p>
 * <p><b>Example:</b></p>
 * <pre>
 * Ext.Msg.show({
 *     title:'Save Changes?',
 *     message: 'You are closing a tab that has unsaved changes. Do you want to save changes?',
 *     buttons: Ext.Msg.YESNOCANCEL,
 *     icon: Ext.Msg.QUESTION,
 *     fn: function(btn) {
 *         if (btn === 'yes') {
 *             console.log('Yes pressed');
 *         } else if (btn === 'no') {
 *             console.log('No pressed');
 *         } else {
 *             console.log('Cancel pressed');
 *         }
 *     }
 * });
 * </pre>
 * <p>Showing Ext.Msg while it's already shown will cause the visible instance to be
 * overwritten with the newly passed config. While this may be the desired outcome, you
 * can also create a new MessageBox that can exist alongside the Ext.Msg
 * singleton instance.</p>
 * <p><b>Example:</b></p>
 * <pre>
 * var myMsg = Ext.create('Ext.window.MessageBox', {
 *     // set closeAction to 'destroy' if this instance is not
 *     // intended to be reused by the application
 *     closeAction: 'destroy'
 * }).show({
 *     title: 'Custom MessageBox Instance',
 *     message: 'I can exist along with Ext.Msg'
 * });
 *
 * Ext.Msg.alert('Overlapping', 'Ext.Msg instance');
 * </pre>
 * <p><b><i>Static Values</i></b></p>
 * <p>The following values can be utilized by Ext.window.MessageBox and Ext.Msg:</p>
 * <pre>
 * OK    : {text: 'OK',     itemId: 'ok'}
 * YES   : {text: 'Yes',    itemId: 'yes'}
 * NO    : {text: 'No',     itemId: 'no'}
 * CANCEL: {text: 'Cancel', itemId: 'cancel'}
 *
 * INFO    : Ext.baseCSSPrefix + 'msgbox-info'
 * WARNING : Ext.baseCSSPrefix + 'msgbox-warning'
 * QUESTION: Ext.baseCSSPrefix + 'msgbox-question'
 * ERROR   : Ext.baseCSSPrefix + 'msgbox-error'
 *
 * OKCANCEL: [
 *     {text: 'Cancel', itemId: 'cancel'},
 *     {text: 'OK',     itemId: 'ok'}
 * ]
 *
 * YESNOCANCEL: [
 *     {text: 'Cancel', itemId: 'cancel'},
 *     {text: 'No',     itemId: 'no'},
 *     {text: 'Yes',    itemId: 'yes'}
 * ]
 *
 * YESNO: [
 *     {text: 'No',  itemId: 'no'},
 *     {text: 'Yes', itemId: 'yes'}
 * ]
 * </pre>
 * @see ext.#MessageBox ext.MessageBox
 * @see #alert()
 * @see #show()
 * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html Original Ext JS documentation of 'Ext.window.MessageBox'
 */
public class MessageBoxWindow extends Window {
  /**
   * Button config that displays a single Cancel button
   * @default 8
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-CANCEL Original Ext JS documentation of 'CANCEL'
   */
  public static native function get CANCEL():*;

  /**
   * The CSS class that provides the ERROR icon image
   * @default Ext.baseCSSPrefix + 'message-box-error'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-ERROR Original Ext JS documentation of 'ERROR'
   */
  public static native function get ERROR():*;

  /**
   * The CSS class that provides the INFO icon image
   * @default Ext.baseCSSPrefix + 'message-box-info'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-INFO Original Ext JS documentation of 'INFO'
   */
  public static native function get INFO():*;

  /**
   * Button config that displays a single No button
   * @default 4
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-NO Original Ext JS documentation of 'NO'
   */
  public static native function get NO():*;

  /**
   * Button config that displays a single OK button
   * @default 1
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-OK Original Ext JS documentation of 'OK'
   */
  public static native function get OK():*;

  /**
   * Button config that displays OK and Cancel buttons
   * @default 9
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-OKCANCEL Original Ext JS documentation of 'OKCANCEL'
   */
  public static native function get OKCANCEL():*;

  /**
   * The CSS class that provides the QUESTION icon image
   * @default Ext.baseCSSPrefix + 'message-box-question'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-QUESTION Original Ext JS documentation of 'QUESTION'
   */
  public static native function get QUESTION():*;

  /**
   * The CSS class that provides the WARNING icon image
   * @default Ext.baseCSSPrefix + 'message-box-warning'
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-WARNING Original Ext JS documentation of 'WARNING'
   */
  public static native function get WARNING():*;

  /**
   * Button config that displays a single Yes button
   * @default 2
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-YES Original Ext JS documentation of 'YES'
   */
  public static native function get YES():*;

  /**
   * Button config that displays Yes and No buttons
   * @default 6
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-YESNO Original Ext JS documentation of 'YESNO'
   */
  public static native function get YESNO():*;

  /**
   * Button config that displays Yes, No and Cancel buttons
   * @default 14
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-YESNOCANCEL Original Ext JS documentation of 'YESNOCANCEL'
   */
  public static native function get YESNOCANCEL():*;

  /**
   * @param config @inheritDoc
   */
  public function MessageBoxWindow(config:MessageBoxWindow = null) {
    super();
  }

  /**
   * An object containing the default button text strings that can be overriden for localized
   * language support. Supported properties are: ok, cancel, yes and no.
   * Generally you should include a locale-specific resource file for handling language support
   * across the framework.
   * Customize the default text like so:
   * <pre>
   * Ext.window.MessageBox.buttonText.yes = "oui"; //french
   * </pre>
   * @default {
   *     ok: 'OK',
   *     yes: 'Yes',
   *     no: 'No',
   *     cancel: 'Cancel'
   * }
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-buttonText Original Ext JS documentation of 'buttonText'
   */
  public native function get buttonText():*;

  /**
   * @private
   */
  public native function set buttonText(value:*):void;

  /**
   * The default height in pixels of the message box's multiline textarea if displayed.
   * @default 75
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-defaultTextHeight Original Ext JS documentation of 'defaultTextHeight'
   */
  public native function get defaultTextHeight():*;

  /**
   * @private
   */
  public native function set defaultTextHeight(value:*):void;

  /**
   * The minimum width in pixels of the message box if it is a progress-style dialog.
   * This is useful for setting a different minimum width than text-only dialogs may need.
   * @default 250
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-minProgressWidth Original Ext JS documentation of 'minProgressWidth'
   */
  public native function get minProgressWidth():*;

  /**
   * @private
   */
  public native function set minProgressWidth(value:*):void;

  /**
   * The minimum width in pixels of the message box if it is a prompt dialog. This is useful
   * for setting a different minimum width than text-only dialogs may need.
   * @default 250
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#property-minPromptWidth Original Ext JS documentation of 'minPromptWidth'
   */
  public native function get minPromptWidth():*;

  /**
   * @private
   */
  public native function set minPromptWidth(value:*):void;

  /**
   * Displays a standard read-only message box with an OK button (comparable to the basic
   * JavaScript alert prompt). If a callback function is passed it will be called after the user
   * clicks the button, and the id of the button that was clicked will be passed as the only
   * parameter to the callback (could also be the top-right close button, which will always
   * report as "cancel").
   * @param title The title bar text
   * @param message The message box body text
   * @param fn The callback function invoked after the message box is closed.
   * See →<code>show()</code> method for details.
   * @param scope (Default window) The scope (<code>this</code> reference) in which the callback
   * is executed.
   * @default window
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#method-alert Original Ext JS documentation of 'alert'
   * @see #show()
   */
  public native function alert(title:String, message:String, fn:Function = null, scope:Object = null):MessageBoxWindow;

  /**
   * Displays a confirmation message box with Yes and No buttons (comparable to JavaScript's
   * confirm). If a callback function is passed it will be called after the user clicks either
   * button, and the id of the button that was clicked will be passed as the only parameter
   * to the callback (could also be the top-right close button, which will always report
   * as "cancel").
   * @param title The title bar text
   * @param message The message box body text
   * @param fn The callback function invoked after the message box is closed.
   * See →<code>show()</code> method for details.
   * @param scope (Default window) The scope (<code>this</code> reference) in which the callback
   * is executed.
   * @default window
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#method-confirm Original Ext JS documentation of 'confirm'
   * @see #show()
   */
  public native function confirm(title:String, message:String, fn:Function = null, scope:Object = null):MessageBoxWindow;

  /**
   * Displays a message box with a progress bar.
   * <p>You are responsible for updating the progress bar as needed via
   * →<code>ext.window.MessageBoxWindow.updateProgress()</code> and closing the message box when the process
   * is complete.</p>
   * @param title The title bar text
   * @param message The message box body text
   * @param progressText The text to display inside the progress bar
   * @default ''
   * @return this
   * @see ext.window.MessageBoxWindow#updateProgress()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#method-progress Original Ext JS documentation of 'progress'
   */
  public native function progress(title:String, message:String, progressText:String = ''):MessageBoxWindow;

  /**
   * Displays a message box with OK and Cancel buttons prompting the user to enter some text
   * (comparable to JavaScript's prompt). The prompt can be a single-line or multi-line textbox.
   * If a callback function is passed it will be called after the user clicks either button,
   * and the id of the button that was clicked (could also be the top-right close button,
   * which will always report as "cancel") and the text that was entered will be passed
   * as the two parameters to the callback.
   * @param title The title bar text
   * @param message The message box body text
   * @param fn The callback function invoked after the message box is closed.
   * See →<code>show()</code> method for details.
   * @param scope (Default window) The scope (<code>this</code> reference) in which the callback
   * is executed.
   * @default window
   * @param multiline True to create a multiline textbox
   * using the defaultTextHeight property, or the height in pixels to create the textbox.
   * @default false
   * @param value Default value of the text input element
   * @default ''
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#method-prompt Original Ext JS documentation of 'prompt'
   * @see #show()
   */
  public native function prompt(title:String, message:String, fn:Function = null, scope:Object = null, multiline:* = false, value:String = ''):MessageBoxWindow;

  /**
   * Adds the specified icon to the dialog. By default, the class 'x-messagebox-icon'
   * is applied for default styling, and the class passed in is expected to supply
   * the background image url. Pass in empty string ('') to clear any existing icon.
   * This method must be called before the MessageBox is shown. The following built-in icon
   * classes are supported, but you can also pass in a custom class name:
   * <pre>
   * Ext.window.MessageBox.INFO
   * Ext.window.MessageBox.WARNING
   * Ext.window.MessageBox.QUESTION
   * Ext.window.MessageBox.ERROR
   * </pre>
   * @param icon The new value.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#method-setIcon Original Ext JS documentation of 'setIcon'
   */
  override public native function setIcon(icon:String):void;

  /**
   * Displays a new message box, or re-initializes an existing message box, based on the config
   * options passed in. All display functions (e.g. prompt, alert, etc.) on MessageBox
   * call this function internally, although those calls are basic shortcuts and do not support
   * all of the config options allowed here.
   * <p>Example usage:</p>
   * <pre>
   * Ext.Msg.show({
   *     title: 'Address',
   *     message: 'Please enter your address:',
   *     width: 300,
   *     buttons: Ext.Msg.OKCANCEL,
   *     multiline: true,
   *     fn: saveAddress,
   *     animateTarget: 'addAddressBtn',
   *     icon: Ext.window.MessageBox.INFO
   * });
   * </pre>
   * @since 6.2.0
   * @param animateTarget <b>only valid for →<code>floating</code>
   * Components such as →<code>ext.window.Window</code>s or →<code>ext.tip.ToolTip</code>s,
   * or regular Components which have been configured with <code>floating: true</code>.</b> The target
   * from which the Component should animate from while opening.
   * @default null
   * @param callback A callback function to call after the Component is displayed.
   * Only necessary if animation was specified.
   * @param scope The scope (<code>this</code> reference) in which the callback is executed.
   * Defaults to this Component.
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#method-show Original Ext JS documentation of 'show'
   * @see #floating
   * @see ext.window.Window
   * @see ext.tip.ToolTip
   */
  [Return("this")]
  override public native function show(animateTarget:* = null, callback:Function = null, scope:Object = null):void;

  /**
   * Updates a progress-style message box's text and progress bar. Only relevant on message boxes
   * initiated via →<code>ext.window.MessageBoxWindow.progress()</code> or →<code>ext.window.MessageBoxWindow.wait()</code>,
   * or by calling →<code>ext.window.MessageBoxWindow.show()</code> with progress: true.
   * @param value Any number between 0 and 1 (e.g., .5)
   * @default 0
   * @param progressText The progress text to display inside the progress bar.
   * @default ''
   * @param message The message box's body text is replaced with the specified string
   * (defaults to undefined so that any existing body text will not get overwritten by default
   * unless a new value is passed in)
   * @return this
   * @see ext.window.MessageBoxWindow#progress()
   * @see ext.window.MessageBoxWindow#wait()
   * @see ext.window.MessageBoxWindow#show()
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#method-updateProgress Original Ext JS documentation of 'updateProgress'
   */
  public native function updateProgress(value:Number = 0, progressText:String = '', message:String = null):MessageBoxWindow;

  /**
   * Displays a message box with an infinitely auto-updating progress bar. This can be used
   * to block user interaction while waiting for a long-running process to complete that does not
   * have defined intervals. You are responsible for closing the message box when the process
   * is complete.
   * @param message The message box body text
   * @param title The title bar text
   * @param config A →<code>ext.ProgressBar.wait()</code> config object
   * @return this
   * @see https://docs.sencha.com/extjs/7.2.0/classic/Ext.window.MessageBox.html#method-wait Original Ext JS documentation of 'wait'
   * @see ext.ProgressBar#wait()
   */
  public native function wait(message:String, title:String = null, config:Object = null):MessageBoxWindow;
}
}