package flexunit.framework {

/**
 * The Base Class for test cases. A Test case defines the fixture in which to run multiple tests.
 * @description
 * Each test runs in its own fixture so there can be no side effects among test runs.
 * Here is an example:
 * <pre>
 * import samples.com.iterationtwo.Money;
 * import flexunit.framework.TestCase;
 *
 * class samples.test.com.iterationtwo.TestMoney extends TestCase
 * {
 *     ...
 * }
 * </pre>
 *
 * For each test, implement a method which interacts
 * with the fixture. Verify the expected results using the assertions
 * residing in the base class <code>Assert</code>.
 * <pre>
 *  public function testAddMoney()
 *  {
 *     var dollars1 : Number = 3;
 *     var cents1 : Number = 50;
 *     var money1 : Money = new Money( dollars1, cents1 );
 *
 *     var dollars2 : Number = 3;
 *     var cents2 : Number = 20;
 *     var money2 : Money = new Money( dollars2, cents2 );
 *
 *     var money3 : Money = money1.addMoney( money2 )
 *
 *     assertNotNull( "money was null", money3 );
 *     assertNotUndefined( "money was undefined", money3 );
 *
 *     assertEquals( "Dollars should be 6", 6, money3.dollars );
 *     assertEquals( "Cents should be 70", 70, money3.cents );
 *  }
 * </pre>
 * You can also specify <code>setUp()</code> and <code>tearDown()</code> methods
 * to initialize and clean-up tests.
 * <pre>
 * import samples.com.iterationtwo.Money;
 * import flexunit.framework.TestCase;
 *
 * class samples.test.com.iterationtwo.TestMoneyWithSetUpTearDown extends TestCase
 * {
 *    public function setUp() : void
 *    {
 *       money = new Money( 3, 50 );
 *    }
 *
 *    public function tearDown() : void
 *    {
 *       money = null;
 *    }
 *
 *    private var money : Money;
 * }
 * </pre>
 * The tests to be run can be collected into a <code>TestSuite</code> and run using a a <code>TestRunner</code>.
 *
 * The AS3 version has been extended to allow asynchronous tests.  This means that an individual test method
 * will not be considered complete until any subsequent asynchronous calls that have been specified have completed.
 * Adding an asynchronous callback is easy, simply call the addAsync method passing in the callback method and the
 * time in which it is expected to complete.  You can also pass any data you want carried through the call.  addAsync
 * will return a function capable of handling generic events which you can then set as the event listener on an object.
 * E.g.,
 * <pre>
 *   public function testAsync() : void
 *   {
 *      var myDispatcher : ValueDispatcher = new ValueDispatcher();
 *      var helper : Function = addAsync(eventHandler, 200, "foo");
 *      myDispatcher.addEventListener("value", helper);
 *      myDispatcher.dispatchValue("foo", 100); //the dispatcher should send a value in 100 ms
 *   }
 *
 *   //the expected is the extra data that was passed in addAsync above
 *   public function eventHandler(event : ValueEvent, expected : String)
 *   {
 *      var actual : String = event.value;
 *      assertEquals(expected, actual);
 *   }
 * </pre>
 *
 *
 * @see flexunit.framework.Assert
 * @see flexunit.framework.TestSuite
 * @see flexunit.framework.TestResult
 */
public class TestCase extends flexunit.framework.Assert implements flexunit.framework.Test {
  /**
   * The TestCase constructor. If you provide a contstructor in a <code>TestCase</code> subclass,
   * you should ensure that this constructor is called.
   * @param methodName The name of the test method to be called in the test run.
   */
  public function TestCase(methodName:String = null) {
    super();
  }

  /**
   * Creates a new <code>TestResult</code> and runs the tests, populating that <code>TestResult</code>
   * with the results.
   */
  public native function run():flexunit.framework.TestResult;

  /**
   * Runs the tests, populating the <code>result</code> parameter.
   * @param result The TestResult instance to be populated
   */
  public native function runWithResult(result:flexunit.framework.TestResult):void;

  /**
   * Runs <code>setUp()</code>
   */
  public native function runStart():void;

  /**
   * Runs the normal test method or the next asynchronous method
   */
  public native function runMiddle():void;

  /**
   * Runs <code>tearDown()</code>
   */
  public native function runFinish():void;

  /**
   * Empty implementation of <code>setUp()</code>. Can be overridden in test class.
   */
  public native function setUp():void;

  /**
   * Empty implementation of <code>tearDown()</code>. Can be overridden in test class.
   */
  public native function tearDown():void;

  /**
   * The number of test cases in this test class.
   * @return A Number representation the count of test cases in this test class. Always returns 1 for <code>TestCase</code>
   */
  public native function countTestCases():Number;

  /**
   * A string representation of the test case
   * @return A string representation of the test class name and the test method name.
   */
  public native function toString():String;

  /**
   * Returns the the fully qualified class name
   * @return The fully qualified class name
   */
  public native function get className():String;

  public native function get assertionsMade():Number;

  /**
   * Add an asynchronous check point to the test.
   * This method will return an event handler function.
   *
   * @param func the Function to execute when things have been handled
   * @param timeout if the function isn't called within this time the test is considered a failure
   * @param passThroughData data that will be passed to your function (only if non-null) as the 2nd argument
   * @param failFunc a Function that will be called if the asynchronous function fails to execute, useful if perhaps the failure to
   *     execute was intentional or if you want a specific failure message
   * @return the Function that can be used as an event listener
   */
  public native function addAsync(func:Function, timeout:int, passThroughData:Object = null, failFunc:Function = null):Function;

  /**
   * Returns true if there are any asynchronous methods remaining to be called
   */
  public native function hasAsync():Boolean;

  /**
   * Called by the TestResult to kick off wait for the next asynchronous method
   */
  public native function startAsync():void;

  /**
   * The AsyncTestHelper will call this when it's ready for to start the next async.  It's possible that
   * it will need to get access to it even before async has been started if the call didn't actually end
   * up being asynchronous.
   */
  public native function getNextAsync():Object;

  /**
   * Called by the TestResult to pass along so that it can be passed for async
   */
  public native function setTestResult(result:flexunit.framework.TestResult):void;

  public native function set assertionsMade(value:Number):void;

  public native function getTestMethodNames():Array;

  /**
   * The method name of the individual test to be run
   */
  public var methodName:String;
}
}