package net.jqwik.api.configurators;

import org.apiguardian.api.*;

import net.jqwik.api.*;
import net.jqwik.api.providers.*;

import static org.apiguardian.api.API.Status.*;

/**
 * Implementors can modify any arbitrary before it's being used for value generation.
 * Most implementations use {@linkplain ArbitraryConfiguratorBase} to derive from
 *
 * <p>
 * Implementations must be registered in <code>/META-INF/services/net.jqwik.api.configurators.ArbitraryConfigurator</code>
 * so that they will be automatically considered for arbitrary configuration.
 * <p>
 */
@API(status = MAINTAINED, since = "1.0")
public interface ArbitraryConfigurator extends Comparable<ArbitraryConfigurator> {

	/**
	 * Configure a given {@code arbitrary} and return the configured instance.
	 *
	 * @param arbitrary The arbitrary instance to be configured
	 * @param targetType The type of the object to be generated by the arbitrary
	 * @return the newly configured arbitrary instance
	 */
	<T> Arbitrary<T> configure(Arbitrary<T> arbitrary, TypeUsage targetType);

	/**
	 * Determines the order in which a configurator will be applied in regards to other configurators.
	 * Default value is {@code 100}. Use lower values to enforce earlier application and
	 * higher values for later application.
	 *
	 * @return the order
	 */
	default int order() {
		return 100;
	}

	@Override
	default int compareTo(ArbitraryConfigurator other) {
		return Integer.compare(this.order(), other.order());
	}
}
