/**
 * The MIT License (MIT)
 *
 * Copyright (c) 2017 Paul Campbell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies
 * or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package net.kemitix.quality.api;

import lombok.Getter;
import lombok.NonNull;
import org.apache.maven.model.Dependency;
import org.apache.maven.model.Plugin;
import org.codehaus.plexus.util.xml.Xpp3Dom;

import java.util.HashMap;
import java.util.Map;

/**
 * Plugin and Goal with constructor to create the Plugin.
 *
 * @author Paul Campbell (pcampbell@kemitix.net)
 */
public abstract class AbstractPluginGoal implements PluginGoal {

    private static final String CONFIGURATION = "configuration";

    @Getter
    private final Plugin plugin = new Plugin();

    @Getter
    private final Xpp3Dom configuration = new Xpp3Dom(CONFIGURATION);

    /**
     * Constructor to assign configuration to plugin.
     */
    protected AbstractPluginGoal() {
        plugin.setConfiguration(configuration);
    }

    /**
     * Set the Maven Co-ordinates for the plugin.
     *
     * @param groupId    The groupId
     * @param artifactId The artifactId
     * @param version    The version
     */
    @Override
    public void groupArtifactVersion(
            @NonNull final String groupId, @NonNull final String artifactId, @NonNull final String version
                                    ) {
        plugin.setGroupId(groupId);
        plugin.setArtifactId(artifactId);
        plugin.setVersion(version);
    }

    /**
     * Add name and boolean value element to the root config.
     *
     * @param name  The name of the element
     * @param value The value of the element
     */
    protected final void addRootElement(final String name, final boolean value) {
        addRootElement(name, booleanToString(value));
    }

    /**
     * Add name and string value element to the root config.
     *
     * @param name  The name of the element
     * @param value The value of the element
     */
    protected final void addRootElement(final String name, final String value) {
        addChildElement(getConfiguration(), name, value);
    }

    /**
     * Add container element to the root config.
     *
     * @param name The name of the container
     *
     * @return the container
     */
    protected final Xpp3Dom addRootContainer(final String name) {
        return addChildContainer(getConfiguration(), name);
    }

    /**
     * Add container element to the parent config.
     *
     * @param parent The parent of the container
     * @param name   The name of the container
     *
     * @return the container
     */
    protected final Xpp3Dom addChildContainer(final Xpp3Dom parent, final String name) {
        final Xpp3Dom childContainer = new Xpp3Dom(name);
        parent.addChild(childContainer);
        return childContainer;
    }

    /**
     * Add name and string value element to the parent config.
     *
     * @param parent The parent of the element
     * @param name   The name of the element
     * @param value  The value of the element
     */
    protected final void addChildElement(final Xpp3Dom parent, final String name, final String value) {
        final Xpp3Dom basePackageElement = new Xpp3Dom(name);
        basePackageElement.setValue(value);
        parent.addChild(basePackageElement);
    }

    private static String booleanToString(final boolean value) {
        final Map<Boolean, String> booleanStringMap = new HashMap<>(2);
        booleanStringMap.put(false, "false");
        booleanStringMap.put(true, "true");
        return booleanStringMap.get(value);
    }

    /**
     * Add a dependency for the plugin.
     *
     * @param groupId    The groupId
     * @param artifactId The artifactId
     * @param version    The version
     */
    protected final void addDependency(
            final String groupId, final String artifactId, final String version
                                      ) {
        final Dependency dependency = new Dependency();
        dependency.setGroupId(groupId);
        dependency.setArtifactId(artifactId);
        dependency.setVersion(version);
        getPlugin().addDependency(dependency);
    }
}
