/**
 * The MIT License (MIT)
 *
 * Copyright (c) 2017 Paul Campbell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies
 * or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package net.kemitix.quality;

import lombok.Getter;
import lombok.Setter;
import net.kemitix.quality.api.PluginGoal;
import net.kemitix.quality.goals.checkstyle.CheckstyleConfiguration;
import net.kemitix.quality.goals.digraph.DigraphConfiguration;
import net.kemitix.quality.goals.enforcer.EnforcerConfiguration;
import net.kemitix.quality.goals.highwheel.HighwheelConfiguration;
import net.kemitix.quality.goals.huntbugs.HuntbugsConfiguration;
import net.kemitix.quality.goals.jacoco.JacocoConfiguration;
import net.kemitix.quality.goals.pitest.PitestConfiguration;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Paul Campbell (pcampbell@kemitix.net)
 */
public abstract class AbstractCheckMojo extends AbstractMojo
        implements DigraphConfiguration, CheckstyleConfiguration, EnforcerConfiguration, HuntbugsConfiguration,
        JacocoConfiguration, HighwheelConfiguration, PitestConfiguration {

    @Setter
    @Component(role = QualityRunner.class)
    private QualityRunner qualityRunner;

    // Digraph properties

    @Getter
    @Parameter(property = "digraphVersion", defaultValue = "0.9.0", readonly = true)
    private String digraphVersion;

    @Getter
    @Parameter(property = "digraphBasePackage", required = true, readonly = true)
    private String digraphBasePackage;

    @Getter
    @Parameter(property = "digraphIncludeTests", defaultValue = "false", readonly = true)
    private boolean digraphIncludeTests;

    @Getter
    @Parameter(property = "digraphDebug", defaultValue = "false", readonly = true)
    private boolean digraphDebug;

    // checkstyle properties

    @Getter
    @Parameter(property = "checkstylePluginVersion", defaultValue = "2.17", readonly = true)
    private String checkstylePluginVersion;

    @Getter
    @Parameter(property = "checkstyleVersion", defaultValue = "8.0", readonly = true)
    private String checkstyleVersion;

    @Getter
    @Parameter(property = "sevntuVersion", defaultValue = "1.24.1", readonly = true)
    private String sevntuVersion;

    @Getter
    @Parameter(property = "rulesetVersion", defaultValue = "3.3.0", readonly = true)
    private String rulesetVersion;

    @Getter
    @Parameter(property = "rulesetLevel", defaultValue = "5-complexity", readonly = true)
    private String rulesetLevel;

    // enforcer properties

    @Getter
    @Parameter(property = "enforcerVersion", defaultValue = "1.4.1", readonly = true)
    private String enforcerVersion;

    @Getter
    @Parameter(property = "requiredMavenVersion", defaultValue = "3.3.9", readonly = true)
    private String requiredMavenVersion;

    @Getter
    @Parameter(property = "requiredJavaVersion", defaultValue = "1.8", readonly = true)
    private String requiredJavaVersion;

    // huntbugs

    @Getter
    @Parameter(property = "huntbugsVersion", defaultValue = "0.0.11", readonly = true)
    private String huntbugsVersion;

    // jacoco

    @Getter
    @Parameter(property = "jacocoVersion", defaultValue = "0.7.9", readonly = true)
    private String jacocoVersion;

    @Getter
    @Parameter(property = "jacocoCoverage", defaultValue = "1", readonly = true)
    private String jacocoCoverage;

    @Getter
    @Parameter(property = "jacocoMissedClasses", defaultValue = "0", readonly = true)
    private String jacocoMissedClasses;

    // highwheel

    @Getter
    @Parameter(property = "highwheelVersion", defaultValue = "1.2", readonly = true)
    private String highwheelVersion;

    // pitest

    @Getter
    @Parameter(property = "pitestVersion", defaultValue = "1.2.2", readonly = true)
    private String pitestVersion;

    @Getter
    @Parameter(property = "pitestMutation", defaultValue = "100", readonly = true)
    private String pitestMutation;

    @Getter
    @Parameter(property = "pitestCoverage", defaultValue = "100", readonly = true)
    private String pitestCoverage;

    // disabled elements

    @Getter
    @Parameter(property = "qualityDisabled", readonly = true)
    private final List<String> qualityDisabled = new ArrayList<>();

    // execute

    /**
     * The Lifecycle Phase to invoke the Mojo in.
     *
     * @return the lifecycle phase
     */
    protected abstract PluginGoal.LifecyclePhase getLifecyclePhase();

    @Override
    public final void execute() throws MojoExecutionException, MojoFailureException {
        qualityRunner.setCheckMojo(this);
        qualityRunner.setLifecyclePhase(getLifecyclePhase());
        qualityRunner.run();
    }
}
