/**
 * The MIT License (MIT)
 *
 * Copyright (c) 2017 Paul Campbell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies
 * or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package net.kemitix.quality;

import net.kemitix.quality.api.PluginGoal;
import net.kemitix.quality.mojoexecutor.MojoExecutorWrapper;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.model.Plugin;
import org.apache.maven.plugin.BuildPluginManager;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.codehaus.plexus.util.xml.Xpp3Dom;

import javax.inject.Inject;
import javax.inject.Named;

/**
 * Execute goal on a maven plugin.
 *
 * @author Paul Campbell (pcampbell@kemitix.net).
 */
@Named
final class PluginExecutorImpl implements PluginExecutor {

    private final MojoExecutorWrapper mojoExecutorWrapper;

    private final MavenSession mavenSession;

    private final BuildPluginManager pluginManager;

    /**
     * Constructor.
     *
     * @param mojoExecutorWrapper The MojoExecutor wrapper
     * @param mavenSession        The Maven Session
     * @param pluginManager       The Build Plugin Manager
     */
    @Inject
    PluginExecutorImpl(
            final MojoExecutorWrapper mojoExecutorWrapper, final MavenSession mavenSession,
            final BuildPluginManager pluginManager
                      ) {
        this.mojoExecutorWrapper = mojoExecutorWrapper;
        this.mavenSession = mavenSession;
        this.pluginManager = pluginManager;
    }

    @Override
    public void execute(final PluginGoal pluginGoal, final Log log) {
        pluginGoal.getGoals()
                  .forEach(goal -> executeGoal(pluginGoal, log, goal));
    }

    private void executeGoal(final PluginGoal pluginGoal, final Log log, final String goal) {
        final Plugin plugin = pluginGoal.getPlugin();
        log.info(String.format("-- %s:%s:%s --", plugin.getArtifactId(), plugin.getVersion(), goal));
        pluginGoal.onExecute(goal);
        try {
            mojoExecutorWrapper.executeMojo(
                    plugin, goal, (Xpp3Dom) plugin.getConfiguration(), mavenSession, pluginManager);
            pluginGoal.onSuccess(goal);
        } catch (final MojoExecutionException e) {
            pluginGoal.onError(goal, e);
            throw new QualityPluginException(goal, e);
        } finally {
            pluginGoal.onComplete(goal);
        }
    }

}
