/**
 * The MIT License (MIT)
 *
 * Copyright (c) 2017 Paul Campbell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software
 * and associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies
 * or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package net.kemitix.quality;

import lombok.Setter;
import net.kemitix.quality.api.PluginGoal;

import javax.inject.Inject;
import javax.inject.Named;
import java.util.Comparator;
import java.util.List;
import java.util.function.Predicate;

/**
 * Implementation of {@link QualityRunner}.
 *
 * @author Paul Campbell (pcampbell@kemitix.net)
 */
@Named
class QualityRunnerImpl implements QualityRunner {

    private final PluginExecutor pluginExecutor;

    private final List<PluginGoal> pluginGoals;

    @Setter
    private AbstractCheckMojo checkMojo;

    @Setter
    private PluginGoal.LifecyclePhase lifecyclePhase;

    /**
     * Constructor.
     *
     * @param pluginExecutor the PluginExecutor
     * @param pluginGoals    The PluginGoals
     */
    @Inject
    QualityRunnerImpl(
            final PluginExecutor pluginExecutor,
            final List<PluginGoal> pluginGoals
                     ) {
        this.pluginExecutor = pluginExecutor;
        this.pluginGoals = pluginGoals;
    }

    @Override
    public void run() {
        pluginGoals.stream()
                   .filter(matchesLifecyclePhase())
                   .filter(isNotDisabled())
                   .sorted(Comparator.comparing(PluginGoal::getWeight))
                   .forEach(this::executePluginGoal);
    }

    private Predicate<PluginGoal> matchesLifecyclePhase() {
        return pluginGoal -> pluginGoal.matches(lifecyclePhase);
    }

    private Predicate<PluginGoal> isNotDisabled() {
        return isDisabled().negate();
    }

    private Predicate<PluginGoal> isDisabled() {
        return pluginGoal -> checkMojo.getQualityDisabled()
                                      .contains(pluginGoal.getName());
    }

    private void executePluginGoal(final PluginGoal pluginGoal) {
        pluginGoal.configuration(checkMojo);
        pluginExecutor.execute(pluginGoal, checkMojo.getLog());
    }

}
