/*
 * Decompiled with CFR 0.152.
 */
package com.rabbitmq.client.impl;

import com.rabbitmq.client.impl.CredentialsProvider;
import com.rabbitmq.client.impl.CredentialsRefreshService;
import java.time.Duration;
import java.util.Iterator;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultCredentialsRefreshService
implements CredentialsRefreshService {
    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultCredentialsRefreshService.class);
    private final ScheduledExecutorService scheduler;
    private final ConcurrentMap<CredentialsProvider, CredentialsProviderState> credentialsProviderStates = new ConcurrentHashMap<CredentialsProvider, CredentialsProviderState>();
    private final boolean privateScheduler;
    private final Function<Duration, Duration> refreshDelayStrategy;
    private final Function<Duration, Boolean> approachingExpirationStrategy;

    public DefaultCredentialsRefreshService(ScheduledExecutorService scheduler, Function<Duration, Duration> refreshDelayStrategy, Function<Duration, Boolean> approachingExpirationStrategy) {
        if (refreshDelayStrategy == null) {
            throw new IllegalArgumentException("Refresh delay strategy can not be null");
        }
        this.refreshDelayStrategy = refreshDelayStrategy;
        Function<Duration, Boolean> function = this.approachingExpirationStrategy = approachingExpirationStrategy == null ? duration -> false : approachingExpirationStrategy;
        if (scheduler == null) {
            this.scheduler = Executors.newScheduledThreadPool(1);
            this.privateScheduler = true;
        } else {
            this.scheduler = scheduler;
            this.privateScheduler = false;
        }
    }

    public static Function<Duration, Duration> ratioRefreshDelayStrategy(double ratio) {
        return new RatioRefreshDelayStrategy(ratio);
    }

    public static Function<Duration, Duration> fixedDelayBeforeExpirationRefreshDelayStrategy(Duration duration) {
        return new FixedDelayBeforeExpirationRefreshDelayStrategy(duration);
    }

    public static Function<Duration, Boolean> fixedTimeApproachingExpirationStrategy(Duration limitBeforeExpiration) {
        return new FixedTimeApproachingExpirationStrategy(limitBeforeExpiration.toMillis());
    }

    private static Runnable refresh(ScheduledExecutorService scheduler, CredentialsProviderState credentialsProviderState, Function<Duration, Duration> refreshDelayStrategy) {
        return () -> {
            LOGGER.debug("Refreshing token");
            credentialsProviderState.refresh();
            Duration timeBeforeExpiration = credentialsProviderState.credentialsProvider.getTimeBeforeExpiration();
            Duration newDelay = (Duration)refreshDelayStrategy.apply(timeBeforeExpiration);
            LOGGER.debug("Scheduling refresh in {} seconds", (Object)newDelay.getSeconds());
            ScheduledFuture<?> scheduledFuture = scheduler.schedule(DefaultCredentialsRefreshService.refresh(scheduler, credentialsProviderState, refreshDelayStrategy), newDelay.getSeconds(), TimeUnit.SECONDS);
            credentialsProviderState.refreshTask.set(scheduledFuture);
        };
    }

    @Override
    public String register(CredentialsProvider credentialsProvider, Callable<Boolean> refreshAction) {
        String registrationId = UUID.randomUUID().toString();
        LOGGER.debug("New registration {}", (Object)registrationId);
        Registration registration = new Registration(registrationId, refreshAction);
        CredentialsProviderState credentialsProviderState = this.credentialsProviderStates.computeIfAbsent(credentialsProvider, credentialsProviderKey -> new CredentialsProviderState((CredentialsProvider)credentialsProviderKey));
        credentialsProviderState.add(registration);
        credentialsProviderState.maybeSetRefreshTask(() -> {
            Duration delay = this.refreshDelayStrategy.apply(credentialsProvider.getTimeBeforeExpiration());
            LOGGER.debug("Scheduling refresh in {} seconds", (Object)delay.getSeconds());
            return this.scheduler.schedule(DefaultCredentialsRefreshService.refresh(this.scheduler, credentialsProviderState, this.refreshDelayStrategy), delay.getSeconds(), TimeUnit.SECONDS);
        });
        return registrationId;
    }

    @Override
    public void unregister(CredentialsProvider credentialsProvider, String registrationId) {
        CredentialsProviderState credentialsProviderState = (CredentialsProviderState)this.credentialsProviderStates.get(credentialsProvider);
        if (credentialsProviderState != null) {
            credentialsProviderState.unregister(registrationId);
        }
    }

    @Override
    public boolean isApproachingExpiration(Duration timeBeforeExpiration) {
        return this.approachingExpirationStrategy.apply(timeBeforeExpiration);
    }

    public void close() {
        if (this.privateScheduler) {
            this.scheduler.shutdownNow();
        }
    }

    public static class DefaultCredentialsRefreshServiceBuilder {
        private ScheduledExecutorService scheduler;
        private Function<Duration, Duration> refreshDelayStrategy = DefaultCredentialsRefreshService.ratioRefreshDelayStrategy(0.8);
        private Function<Duration, Boolean> approachingExpirationStrategy = ttl -> false;

        public DefaultCredentialsRefreshServiceBuilder scheduler(ScheduledThreadPoolExecutor scheduler) {
            this.scheduler = scheduler;
            return this;
        }

        public DefaultCredentialsRefreshServiceBuilder refreshDelayStrategy(Function<Duration, Duration> refreshDelayStrategy) {
            this.refreshDelayStrategy = refreshDelayStrategy;
            return this;
        }

        public DefaultCredentialsRefreshServiceBuilder approachingExpirationStrategy(Function<Duration, Boolean> approachingExpirationStrategy) {
            this.approachingExpirationStrategy = approachingExpirationStrategy;
            return this;
        }

        public DefaultCredentialsRefreshService build() {
            return new DefaultCredentialsRefreshService(this.scheduler, this.refreshDelayStrategy, this.approachingExpirationStrategy);
        }
    }

    static class CredentialsProviderState {
        private final CredentialsProvider credentialsProvider;
        private final Map<String, Registration> registrations = new ConcurrentHashMap<String, Registration>();
        private final AtomicReference<ScheduledFuture<?>> refreshTask = new AtomicReference();
        private final AtomicBoolean refreshTaskSet = new AtomicBoolean(false);

        CredentialsProviderState(CredentialsProvider credentialsProvider) {
            this.credentialsProvider = credentialsProvider;
        }

        void add(Registration registration) {
            this.registrations.put(registration.id, registration);
        }

        void maybeSetRefreshTask(Supplier<ScheduledFuture<?>> scheduledFutureSupplier) {
            if (this.refreshTaskSet.compareAndSet(false, true)) {
                this.refreshTask.set(scheduledFutureSupplier.get());
            }
        }

        void refresh() {
            if (Thread.currentThread().isInterrupted()) {
                return;
            }
            boolean refreshSucceeded = false;
            for (int attemptCount = 0; attemptCount < 3; ++attemptCount) {
                LOGGER.debug("Refreshing token for credentials provider {}", (Object)this.credentialsProvider);
                try {
                    this.credentialsProvider.refresh();
                    LOGGER.debug("Token refreshed for credentials provider {}", (Object)this.credentialsProvider);
                    refreshSucceeded = true;
                    break;
                }
                catch (Exception e) {
                    LOGGER.warn("Error while trying to refresh token: {}", (Object)e.getMessage());
                    try {
                        Thread.sleep(1000L);
                        continue;
                    }
                    catch (InterruptedException e2) {
                        Thread.currentThread().interrupt();
                        return;
                    }
                }
            }
            if (!refreshSucceeded) {
                LOGGER.warn("Token refresh failed after retry, aborting callbacks");
                return;
            }
            Iterator<Registration> iterator = this.registrations.values().iterator();
            while (iterator.hasNext() && !Thread.currentThread().isInterrupted()) {
                Registration registration = iterator.next();
                try {
                    boolean refreshed = (Boolean)registration.refreshAction.call();
                    if (!refreshed) {
                        LOGGER.debug("Registration did not refresh token");
                        iterator.remove();
                    }
                    registration.errorHistory.set(0);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                catch (Exception e) {
                    LOGGER.warn("Error while trying to refresh a connection token", e);
                    registration.errorHistory.incrementAndGet();
                    if (registration.errorHistory.get() < 5) continue;
                    this.registrations.remove(registration.id);
                }
            }
        }

        void unregister(String registrationId) {
            this.registrations.remove(registrationId);
        }
    }

    static class Registration {
        private final Callable<Boolean> refreshAction;
        private final AtomicInteger errorHistory = new AtomicInteger(0);
        private final String id;

        Registration(String id, Callable<Boolean> refreshAction) {
            this.refreshAction = refreshAction;
            this.id = id;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Registration that = (Registration)o;
            return this.id.equals(that.id);
        }

        public int hashCode() {
            return this.id.hashCode();
        }
    }

    private static class RatioRefreshDelayStrategy
    implements Function<Duration, Duration> {
        private final double ratio;

        private RatioRefreshDelayStrategy(double ratio) {
            if (ratio < 0.0 || ratio > 1.0) {
                throw new IllegalArgumentException("Ratio should be > 0 and <= 1: " + ratio);
            }
            this.ratio = ratio;
        }

        @Override
        public Duration apply(Duration duration) {
            return Duration.ofSeconds((long)((double)duration.getSeconds() * this.ratio));
        }
    }

    private static class FixedDelayBeforeExpirationRefreshDelayStrategy
    implements Function<Duration, Duration> {
        private final Duration delay;

        private FixedDelayBeforeExpirationRefreshDelayStrategy(Duration delay) {
            this.delay = delay;
        }

        @Override
        public Duration apply(Duration timeBeforeExpiration) {
            Duration refreshTimeBeforeExpiration = timeBeforeExpiration.minus(this.delay);
            if (refreshTimeBeforeExpiration.isNegative()) {
                return timeBeforeExpiration;
            }
            return refreshTimeBeforeExpiration;
        }
    }

    private static class FixedTimeApproachingExpirationStrategy
    implements Function<Duration, Boolean> {
        private final long limitBeforeExpiration;

        private FixedTimeApproachingExpirationStrategy(long limitBeforeExpiration) {
            this.limitBeforeExpiration = limitBeforeExpiration;
        }

        @Override
        public Boolean apply(Duration timeBeforeExpiration) {
            return timeBeforeExpiration.toMillis() <= this.limitBeforeExpiration;
        }
    }
}

