/*
 * Decompiled with CFR 0.152.
 */
package net.kuujo.catalog.client.session;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import net.kuujo.catalog.client.Command;
import net.kuujo.catalog.client.Query;
import net.kuujo.catalog.client.error.RaftError;
import net.kuujo.catalog.client.error.UnknownSessionException;
import net.kuujo.catalog.client.request.CommandRequest;
import net.kuujo.catalog.client.request.KeepAliveRequest;
import net.kuujo.catalog.client.request.PublishRequest;
import net.kuujo.catalog.client.request.QueryRequest;
import net.kuujo.catalog.client.request.RegisterRequest;
import net.kuujo.catalog.client.request.Request;
import net.kuujo.catalog.client.request.SessionRequest;
import net.kuujo.catalog.client.response.PublishResponse;
import net.kuujo.catalog.client.response.Response;
import net.kuujo.catalog.client.response.SessionResponse;
import net.kuujo.catalog.client.session.Session;
import net.kuujo.catalyst.serializer.Serializer;
import net.kuujo.catalyst.transport.Address;
import net.kuujo.catalyst.transport.Client;
import net.kuujo.catalyst.transport.Connection;
import net.kuujo.catalyst.transport.Transport;
import net.kuujo.catalyst.util.Assert;
import net.kuujo.catalyst.util.Listener;
import net.kuujo.catalyst.util.Listeners;
import net.kuujo.catalyst.util.Managed;
import net.kuujo.catalyst.util.concurrent.Context;
import net.kuujo.catalyst.util.concurrent.Futures;
import net.kuujo.catalyst.util.concurrent.Scheduled;
import net.kuujo.catalyst.util.concurrent.SingleThreadContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClientSession
implements Session,
Managed<Session> {
    private static final Logger LOGGER = LoggerFactory.getLogger(ClientSession.class);
    private static final double KEEP_ALIVE_RATIO = 0.4;
    private final Random random = new Random();
    private final Client client;
    private Set<Address> members;
    private final Context context;
    private List<Address> connectMembers;
    private Connection connection;
    private volatile State state = State.CLOSED;
    private volatile long id;
    private long timeout;
    private long failureTime;
    private CompletableFuture<Connection> connectFuture;
    private Scheduled retryFuture;
    private final List<Runnable> retries = new ArrayList<Runnable>();
    private Scheduled keepAliveFuture;
    private final Listeners<Session> openListeners = new Listeners();
    private final Listeners<Object> receiveListeners = new Listeners();
    private final Listeners<Session> closeListeners = new Listeners();
    private long requestSequence;
    private long responseSequence;
    private long eventVersion;
    private long eventSequence;
    private long version;

    public ClientSession(Transport transport, Collection<Address> members, Serializer serializer) {
        UUID id = UUID.randomUUID();
        this.client = ((Transport)Assert.notNull((Object)transport, (String)"transport")).client(id);
        this.members = new HashSet<Address>((Collection)Assert.notNull(members, (String)"members"));
        this.context = new SingleThreadContext("catalog-client-" + id.toString(), ((Serializer)Assert.notNull((Object)serializer, (String)"serializer")).clone());
        this.connectMembers = new ArrayList<Address>(members);
    }

    public long id() {
        return this.id;
    }

    public Context context() {
        return this.context;
    }

    private void setMembers(Collection<Address> members) {
        this.members = new HashSet<Address>(members);
        this.connectMembers = new ArrayList<Address>(this.members);
    }

    private void setTimeout(long timeout) {
        this.timeout = timeout;
    }

    public <T> CompletableFuture<T> submit(Command<T> command) {
        if (!this.isOpen()) {
            return Futures.exceptionalFuture((Throwable)new IllegalStateException("session not open"));
        }
        CompletableFuture future = new CompletableFuture();
        this.context.executor().execute(() -> {
            CommandRequest request = ((CommandRequest.Builder)CommandRequest.builder().withSession(this.id)).withSequence(++this.requestSequence).withCommand(command).build();
            this.submit(request, future);
        });
        return future;
    }

    private <T> CompletableFuture<T> submit(CommandRequest request, CompletableFuture<T> future) {
        if (!this.isOpen()) {
            future.completeExceptionally(new IllegalStateException("session not open"));
            return future;
        }
        request.acquire();
        this.request(request).whenComplete((response, error) -> {
            if (error == null) {
                if (response.status() == Response.Status.OK) {
                    this.responseSequence = Math.max(this.responseSequence, request.sequence());
                    this.version = Math.max(this.version, response.version());
                    future.complete(response.result());
                    this.resetMembers();
                } else {
                    future.completeExceptionally((Throwable)response.error().createException());
                }
                response.release();
            } else {
                future.completeExceptionally((Throwable)error);
            }
            request.release();
        });
        return future;
    }

    public <T> CompletableFuture<T> submit(Query<T> query) {
        if (!this.isOpen()) {
            return Futures.exceptionalFuture((Throwable)new IllegalStateException("session not open"));
        }
        CompletableFuture future = new CompletableFuture();
        this.context.executor().execute(() -> {
            QueryRequest request = ((QueryRequest.Builder)QueryRequest.builder().withSession(this.id)).withVersion(this.responseSequence).withQuery(query).build();
            this.submit(request, future);
        });
        return future;
    }

    private <T> CompletableFuture<T> submit(QueryRequest request, CompletableFuture<T> future) {
        if (!this.isOpen()) {
            future.completeExceptionally(new IllegalStateException("session not open"));
            return future;
        }
        request.acquire();
        this.request(request).whenComplete((response, error) -> {
            if (error == null) {
                if (response.status() == Response.Status.OK) {
                    this.version = Math.max(this.version, response.version());
                    future.complete(response.result());
                    this.resetMembers();
                } else {
                    future.completeExceptionally((Throwable)response.error().createException());
                }
                response.release();
            } else {
                future.completeExceptionally((Throwable)error);
            }
            request.release();
        });
        return future;
    }

    private <T extends SessionRequest<T>, U extends SessionResponse<U>> CompletableFuture<U> request(T request) {
        if (!this.isOpen()) {
            return Futures.exceptionalFutureAsync((Throwable)new IllegalStateException("session not open"), (Executor)this.context.executor());
        }
        return this.request(request, new CompletableFuture(), true, true);
    }

    private <T extends Request<T>, U extends Response<U>> CompletableFuture<U> request(T request, CompletableFuture<U> future, boolean checkOpen, boolean recordFailures) {
        this.context.checkThread();
        if (checkOpen && !this.isOpen()) {
            future.completeExceptionally(new IllegalStateException("session expired"));
            return future;
        }
        if (this.connection != null) {
            return this.request(request, this.connection, future, checkOpen, true);
        }
        if (this.connectMembers.isEmpty()) {
            if (!checkOpen) {
                LOGGER.warn("Failed to connect to cluster");
                future.completeExceptionally(new IllegalStateException("session not open"));
            } else if (this.retryFuture != null) {
                this.retries.add(() -> {
                    LOGGER.debug("Retrying: {}", (Object)request);
                    this.request(request, future, true, true);
                });
            } else if (this.failureTime > 0L && this.failureTime + this.timeout < System.currentTimeMillis()) {
                LOGGER.warn("Lost session");
                this.resetConnection().onExpire();
                future.completeExceptionally(new IllegalStateException("session expired"));
            } else {
                LOGGER.warn("Failed to communicate with cluster. Retrying");
                this.retryFuture = this.context.schedule(this::retryRequests, Duration.ofMillis(200L));
                this.retries.add(() -> this.resetMembers().request(request, future, true, true));
            }
            return future;
        }
        Address member = this.connectMembers.remove(this.random.nextInt(this.connectMembers.size()));
        if (this.connectFuture == null) {
            LOGGER.info("Connecting: {}", (Object)member.socketAddress());
            this.connectFuture = this.client.connect(member).whenComplete((connection, error) -> {
                this.connectFuture = null;
                if (!checkOpen || this.isOpen()) {
                    if (error == null) {
                        this.setupConnection((Connection)connection);
                        this.request(request, (Connection)connection, future, checkOpen, recordFailures);
                    } else {
                        LOGGER.info("Failed to connect: {}", (Object)member.socketAddress());
                        this.resetConnection().request(request, future, checkOpen, recordFailures);
                    }
                } else {
                    future.completeExceptionally(new IllegalStateException("session not open"));
                }
            });
        } else {
            this.connectFuture.whenComplete((connection, error) -> {
                if (!checkOpen || this.isOpen()) {
                    if (error == null) {
                        this.request(request, (Connection)connection, future, checkOpen, recordFailures);
                    } else {
                        this.request(request, future, checkOpen, recordFailures);
                    }
                } else {
                    future.completeExceptionally(new IllegalStateException("session not open"));
                }
            });
        }
        return future;
    }

    private <T extends Request<T>, U extends Response<U>> CompletableFuture<U> request(T request, Connection connection, CompletableFuture<U> future, boolean checkOpen, boolean recordFailures) {
        request.acquire();
        LOGGER.debug("Sending: {}", request);
        connection.send(request).whenComplete((response, error) -> {
            if (!checkOpen || this.isOpen()) {
                if (error == null) {
                    LOGGER.debug("Received: {}", response);
                    if (response.status() == Response.Status.ERROR) {
                        if (response.error() == RaftError.Type.NO_LEADER_ERROR) {
                            if (recordFailures) {
                                this.setFailureTime();
                            }
                            this.resetConnection().request(request, future, checkOpen, false);
                        } else if (response.error() == RaftError.Type.UNKNOWN_SESSION_ERROR) {
                            this.resetConnection().onExpire();
                            future.completeExceptionally(new IllegalStateException("session expired"));
                        } else if (response.error() == RaftError.Type.APPLICATION_ERROR || response.error() == RaftError.Type.INTERNAL_ERROR) {
                            this.resetFailureTime();
                            future.completeExceptionally((Throwable)response.error().createException());
                        } else {
                            this.resetFailureTime().resetConnection().request(request, future, checkOpen, false);
                        }
                    } else {
                        this.resetFailureTime();
                        future.complete(response);
                    }
                } else {
                    LOGGER.debug("Request timed out: {}", (Object)request);
                    this.resetConnection().request(request, future, checkOpen, recordFailures);
                }
            } else {
                future.completeExceptionally(new IllegalStateException("session not open"));
            }
        });
        return future;
    }

    private void retryRequests() {
        this.retryFuture = null;
        ArrayList<Runnable> retries = new ArrayList<Runnable>(this.retries);
        this.retries.clear();
        this.resetMembers();
        for (Runnable retry : retries) {
            retry.run();
        }
    }

    private ClientSession resetConnection() {
        this.connection = null;
        return this;
    }

    private ClientSession resetMembers() {
        if (this.connectMembers.isEmpty() || this.connectMembers.size() < this.members.size() - 1) {
            this.connectMembers = new ArrayList<Address>(this.members);
        }
        return this;
    }

    private ClientSession setFailureTime() {
        if (this.failureTime == 0L) {
            this.failureTime = System.currentTimeMillis();
        }
        return this;
    }

    private ClientSession resetFailureTime() {
        this.failureTime = 0L;
        return this;
    }

    private ClientSession setupConnection(Connection connection) {
        this.connection = connection;
        connection.closeListener(c -> {
            if (c.equals(this.connection)) {
                this.connection = null;
            }
        });
        connection.exceptionListener(c -> {
            if (c.equals(this.connection)) {
                this.connection = null;
            }
        });
        connection.handler(PublishRequest.class, this::handlePublish);
        return this;
    }

    private CompletableFuture<Void> register() {
        this.context.checkThread();
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        RegisterRequest request = RegisterRequest.builder().withConnection(this.client.id()).build();
        request.acquire();
        this.request(request, new CompletableFuture(), false, true).whenComplete((response, error) -> {
            if (error == null) {
                if (response.status() == Response.Status.OK) {
                    this.setMembers(response.members());
                    this.setTimeout(response.timeout());
                    this.onOpen(response.session());
                    future.complete(null);
                    this.resetMembers().keepAlive(Duration.ofMillis(Math.round((double)response.timeout() * 0.4)));
                } else {
                    future.completeExceptionally((Throwable)response.error().createException());
                }
                response.release();
            } else {
                future.completeExceptionally((Throwable)error);
            }
            request.release();
        });
        return future;
    }

    private void keepAlive(Duration interval) {
        this.keepAliveFuture = this.context.schedule(() -> {
            if (this.isOpen()) {
                this.context.checkThread();
                KeepAliveRequest request = ((KeepAliveRequest.Builder)KeepAliveRequest.builder().withSession(this.id)).withCommandSequence(this.responseSequence).withEventVersion(this.eventVersion).withEventSequence(this.eventSequence).build();
                request.acquire();
                this.request(request).whenComplete((response, error) -> {
                    if (error == null) {
                        if (response.status() == Response.Status.OK) {
                            this.setMembers(response.members());
                            this.resetMembers().keepAlive(interval);
                        } else if (this.isOpen()) {
                            this.keepAlive(interval);
                        }
                        response.release();
                    } else if (this.isOpen()) {
                        this.keepAlive(interval);
                    }
                    request.release();
                });
            }
        }, interval);
    }

    private void onOpen(long sessionId) {
        LOGGER.debug("Registered session: {}", (Object)sessionId);
        this.id = sessionId;
        this.state = State.OPEN;
        for (Consumer listener : this.openListeners) {
            listener.accept(this);
        }
    }

    public CompletableFuture<Session> open() {
        CompletableFuture<Session> future = new CompletableFuture<Session>();
        this.context.executor().execute(() -> this.register().whenComplete((result, error) -> {
            if (error == null) {
                this.state = State.OPEN;
                future.complete(this);
            } else {
                future.completeExceptionally((Throwable)error);
            }
        }));
        return future;
    }

    public boolean isOpen() {
        return this.state == State.OPEN;
    }

    public Listener<Session> onOpen(Consumer<Session> listener) {
        return this.openListeners.add((Consumer)Assert.notNull(listener, (String)"listener"));
    }

    public CompletableFuture<Void> publish(Object event) {
        Assert.notNull((Object)event, (String)"event");
        return CompletableFuture.runAsync(() -> {
            for (Consumer listener : this.receiveListeners) {
                listener.accept(event);
            }
        }, this.context.executor());
    }

    private CompletableFuture<PublishResponse> handlePublish(PublishRequest request) {
        if (request.session() != this.id) {
            return Futures.exceptionalFuture((Throwable)new UnknownSessionException("incorrect session ID", new Object[0]));
        }
        if (request.previousVersion() != this.eventVersion || request.previousSequence() != this.eventSequence) {
            return CompletableFuture.completedFuture(((PublishResponse.Builder)PublishResponse.builder().withStatus(Response.Status.ERROR)).withVersion(this.eventVersion).withSequence(this.eventSequence).build());
        }
        this.eventVersion = request.eventVersion();
        this.eventSequence = request.eventSequence();
        for (Consumer listener : this.receiveListeners) {
            listener.accept(request.message());
        }
        request.release();
        return CompletableFuture.completedFuture(((PublishResponse.Builder)PublishResponse.builder().withStatus(Response.Status.OK)).withVersion(this.eventVersion).withSequence(this.eventSequence).build());
    }

    public Listener<?> onEvent(Consumer listener) {
        return this.receiveListeners.add((Consumer)Assert.notNull((Object)listener, (String)"listener"));
    }

    public CompletableFuture<Void> close() {
        return CompletableFuture.runAsync(() -> {
            if (this.keepAliveFuture != null) {
                this.keepAliveFuture.cancel();
            }
            if (this.retryFuture != null) {
                this.retryFuture.cancel();
            }
            this.onClose();
        }, this.context.executor());
    }

    private void onClose() {
        if (this.isOpen()) {
            LOGGER.debug("Closed session: {}", (Object)this.id);
            this.id = 0L;
            this.state = State.CLOSED;
            if (this.connection != null) {
                this.connection.close();
            }
            this.client.close();
            this.context.close();
            this.closeListeners.forEach(l -> l.accept((Object)this));
        }
    }

    public boolean isClosed() {
        return this.state == State.CLOSED || this.state == State.EXPIRED;
    }

    public Listener<Session> onClose(Consumer<Session> listener) {
        return this.closeListeners.add((Consumer)Assert.notNull(listener, (String)"listener"));
    }

    private void onExpire() {
        if (this.isOpen()) {
            LOGGER.debug("Expired session: {}", (Object)this.id);
            this.id = 0L;
            this.state = State.EXPIRED;
            this.closeListeners.forEach(l -> l.accept((Object)this));
        }
    }

    public boolean isExpired() {
        return this.state == State.EXPIRED;
    }

    private static enum State {
        OPEN,
        CLOSED,
        EXPIRED;

    }
}

