/*
 * This file is part of adventure-platform-fabric, licensed under the MIT License.
 *
 * Copyright (c) 2022 KyoriPowered
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package net.kyori.adventure.platform.fabric.impl;

import io.netty.buffer.Unpooled;
import java.util.HashSet;
import java.util.Set;
import net.fabricmc.fabric.api.networking.v1.PacketSender;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.kyori.adventure.Adventure;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import org.jetbrains.annotations.NotNull;

/**
 * A packet sent client to server, to let the server know which optional argument types are available on the server.
 *
 * <p>This packet is sent by players on join, before the command tree is sent to the client.</p>
 *
 * @param known Known argument type ids
 */
public record ServerboundRegisteredArgumentTypesPacket(Set<class_2960> known) {
  public static final class_2960 ID = new class_2960(Adventure.NAMESPACE, "registered_args");

  public static void register() {
    ServerPlayNetworking.registerGlobalReceiver(ID, (server, player, handler, buffer, responder) -> {
      final ServerboundRegisteredArgumentTypesPacket pkt = ServerboundRegisteredArgumentTypesPacket.of(buffer);
      server.execute(() -> { // on main thread
        ServerArgumentTypes.knownArgumentTypes(player, pkt.known, responder);
      });
    });
  }

  public static ServerboundRegisteredArgumentTypesPacket of(final Set<class_2960> idents) {
    return new ServerboundRegisteredArgumentTypesPacket(Set.copyOf(idents));
  }

  public static ServerboundRegisteredArgumentTypesPacket of(final @NotNull class_2540 buf) {
    final int length = buf.method_10816();
    final Set<class_2960> items = new HashSet<>();
    for (int i = 0; i < length; ++i) {
      items.add(buf.method_10810());
    }
    return of(items);
  }

  private void toPacket(final class_2540 buffer) {
    buffer.method_10804(this.known.size());
    for (final class_2960 id : this.known) {
      buffer.method_10812(id);
    }
  }

  /**
   * Send the client's list of identifiers to the server.
   *
   * @param sender the sender to send the packet to
   */
  public void sendTo(final PacketSender sender) {
    final class_2540 buffer = new class_2540(Unpooled.buffer(this.known.size() * 8));
    this.toPacket(buffer);
    sender.sendPacket(ID, buffer);
  }
}
