/*
 * This file is part of adventure-platform-fabric, licensed under the MIT License.
 *
 * Copyright (c) 2020-2023 KyoriPowered
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package net.kyori.adventure.platform.fabric.impl.server;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import net.kyori.adventure.audience.MessageType;
import net.kyori.adventure.bossbar.BossBar;
import net.kyori.adventure.chat.ChatType;
import net.kyori.adventure.chat.SignedMessage;
import net.kyori.adventure.identity.Identity;
import net.kyori.adventure.inventory.Book;
import net.kyori.adventure.key.Key;
import net.kyori.adventure.platform.fabric.FabricAudiences;
import net.kyori.adventure.platform.fabric.impl.AdventureCommon;
import net.kyori.adventure.platform.fabric.impl.ControlledAudience;
import net.kyori.adventure.platform.fabric.impl.FabricAudiencesInternal;
import net.kyori.adventure.platform.fabric.impl.GameEnums;
import net.kyori.adventure.platform.fabric.impl.PointerProviderBridge;
import net.kyori.adventure.platform.fabric.impl.accessor.minecraft.network.ServerGamePacketListenerImplAccess;
import net.kyori.adventure.platform.fabric.impl.accessor.minecraft.world.level.LevelAccess;
import net.kyori.adventure.pointer.Pointers;
import net.kyori.adventure.resource.ResourcePackCallback;
import net.kyori.adventure.resource.ResourcePackInfo;
import net.kyori.adventure.resource.ResourcePackRequest;
import net.kyori.adventure.sound.Sound;
import net.kyori.adventure.sound.SoundStop;
import net.kyori.adventure.text.Component;
import net.kyori.adventure.text.serializer.gson.GsonComponentSerializer;
import net.kyori.adventure.text.serializer.plain.PlainTextComponentSerializer;
import net.kyori.adventure.title.Title;
import net.kyori.adventure.title.TitlePart;
import net.kyori.adventure.util.MonkeyBars;
import net.minecraft.class_1268;
import net.minecraft.class_1297;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1843;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2519;
import net.minecraft.class_2596;
import net.minecraft.class_2653;
import net.minecraft.class_2720;
import net.minecraft.class_2765;
import net.minecraft.class_2767;
import net.minecraft.class_2770;
import net.minecraft.class_2960;
import net.minecraft.class_3222;
import net.minecraft.class_3414;
import net.minecraft.class_3419;
import net.minecraft.class_5321;
import net.minecraft.class_5888;
import net.minecraft.class_5903;
import net.minecraft.class_5904;
import net.minecraft.class_5905;
import net.minecraft.class_6880;
import net.minecraft.class_7469;
import net.minecraft.class_7471;
import net.minecraft.class_7604;
import net.minecraft.class_7617;
import net.minecraft.class_7924;
import net.minecraft.class_8042;
import net.minecraft.class_8705;
import net.minecraft.class_9053;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import static java.util.Objects.requireNonNull;

public final class ServerPlayerAudience implements ControlledAudience {
  private final class_3222 player;
  private final FabricServerAudiencesImpl controller;

  public ServerPlayerAudience(final class_3222 player, final FabricServerAudiencesImpl controller) {
    this.player = player;
    this.controller = controller;
  }

  void sendPacket(final class_2596<? extends class_8705> packet) {
    this.player.field_13987.method_14364(packet);
  }

  @SuppressWarnings({"unchecked", "rawtypes"}) // bundle generics don't handle configuration phase
  void sendBundle(final List<? extends class_2596<? extends class_8705>> packet) {
    this.player.field_13987.method_14364(new class_8042((List) packet));
  }

  @Override
  public void sendMessage(final @NotNull Component message) {
    this.player.method_43496(this.controller.toNative(message));
  }

  @Override
  @Deprecated
  public void sendMessage(final Identity source, final Component text, final net.kyori.adventure.audience.MessageType type) {
    final boolean shouldSend = switch (this.player.method_14238()) {
      case field_7538 -> true;
      case field_7539 -> type == MessageType.SYSTEM;
      case field_7536 -> false;
    };

    if (shouldSend) {
      this.player.method_43496(this.controller.toNative(text));
    }
  }

  private net.minecraft.class_2556.class_7602 toMc(final ChatType.Bound adv) {
    return AdventureCommon.chatTypeToNative(adv, this.controller);
  }

  @Override
  public void sendMessage(final @NotNull Component message, final ChatType.@NotNull Bound boundChatType) {
    final class_7604 outgoing = new class_7604.class_7606(
        this.controller.toNative(message)
    );
    this.player.method_43505(outgoing, false, this.toMc(boundChatType));
  }

  @Override
  public void sendMessage(final @NotNull SignedMessage signedMessage, final ChatType.@NotNull Bound boundChatType) {
    if ((Object) signedMessage instanceof class_7471 pcm) {
      if (pcm.method_46293()) {
        this.player.method_43505(new class_7604.class_7606(pcm.method_46291()), false, this.toMc(boundChatType));
      } else {
        this.player.method_43505(new class_7604.class_7607(pcm), false, this.toMc(boundChatType));
      }
    } else {
      this.sendMessage(Objects.requireNonNullElse(signedMessage.unsignedContent(), Component.text(signedMessage.message())), boundChatType);
    }
  }

  @Override
  public void deleteMessage(final SignedMessage.@NotNull Signature signature) {
    this.sendPacket(new class_7617(((class_7469) signature).method_46277(((ServerGamePacketListenerImplAccess) this.player.field_13987).accessor$messageSignatureCache())));
  }

  @Override
  public void sendActionBar(final @NotNull Component message) {
    this.player.method_43502(this.controller.toNative(message), true);
  }

  @Override
  public void showBossBar(final @NotNull BossBar bar) {
    FabricServerAudiencesImpl.forEachInstance(controller -> {
      if (controller != this.controller) {
        controller.bossBars.unsubscribe(this.player, bar);
      }
    });
    this.controller.bossBars.subscribe(this.player, bar);
  }

  @Override
  public void hideBossBar(final @NotNull BossBar bar) {
    FabricServerAudiencesImpl.forEachInstance(controller -> controller.bossBars.unsubscribe(this.player, bar));
  }

  private long seed(final @NotNull Sound sound) {
    if (sound.seed().isPresent()) {
      return sound.seed().getAsLong();
    } else {
      return ((LevelAccess) this.player.method_37908()).accessor$threadSafeRandom().method_43055();
    }
  }

  private class_6880<class_3414> eventHolder(final @NotNull Sound sound) {
    final var soundEventRegistry = this.controller.registryAccess()
      .method_30530(class_7924.field_41225);
    final var soundKey = FabricAudiences.toNative(sound.name());

    final var eventOptional = soundEventRegistry.method_40264(class_5321.method_29179(class_7924.field_41225, soundKey));
    return eventOptional.isPresent() ? eventOptional.get() : class_6880.method_40223(class_3414.method_47908(soundKey));
  }

  @Override
  public void playSound(final @NotNull Sound sound) {
    this.playSound(sound, this.player.method_23317(), this.player.method_23318(), this.player.method_23321());
  }

  @Override
  public void playSound(final @NotNull Sound sound, final double x, final double y, final double z) {
    this.sendPacket(new class_2767(
      this.eventHolder(sound),
      GameEnums.SOUND_SOURCE.toMinecraft(sound.source()),
      x,
      y,
      z,
      sound.volume(),
      sound.pitch(),
      this.seed(sound)
    ));
  }

  @Override
  public void playSound(final @NotNull Sound sound, final Sound.@NotNull Emitter emitter) {
    final class_1297 targetEntity;
    if (emitter == Sound.Emitter.self()) {
      targetEntity = this.player;
    } else if (emitter instanceof class_1297) {
      targetEntity = (class_1297) emitter;
    } else {
      throw new IllegalArgumentException("Provided emitter '" + emitter + "' was not Sound.Emitter.self() or an Entity");
    }

    if (!this.player.method_37908().equals(targetEntity.method_37908())) {
      // don't send unless entities are in the same dimension
      return;
    }

    this.sendPacket(new class_2765(
      this.eventHolder(sound),
      GameEnums.SOUND_SOURCE.toMinecraft(sound.source()),
      targetEntity,
      sound.volume(),
      sound.pitch(),
      this.seed(sound)
    ));
  }

  @Override
  public void stopSound(final @NotNull SoundStop stop) {
    final @Nullable Key sound = stop.sound();
    final Sound.@Nullable Source src = stop.source();
    final @Nullable class_3419 cat = src == null ? null : GameEnums.SOUND_SOURCE.toMinecraft(src);
    this.sendPacket(new class_2770(sound == null ? null : FabricAudiences.toNative(sound), cat));
  }

  @Override
  public void openBook(final @NotNull Book book) {
    final class_1799 bookStack = new class_1799(class_1802.field_8360, 1);
    final class_2487 bookTag = bookStack.method_7948();
    bookTag.method_10582(class_1843.field_30935, validateField(this.adventure$plain(book.title()), class_1843.field_30930, class_1843.field_30935));
    bookTag.method_10582(class_1843.field_30937, this.adventure$plain(book.author()));
    final class_2499 pages = new class_2499();
    if (book.pages().size() > class_1843.field_30933) {
      throw new IllegalArgumentException("Book provided had " + book.pages().size() + " pages, but is only allowed a maximum of " + class_1843.field_30933);
    }
    for (final Component page : book.pages()) {
      pages.add(class_2519.method_23256(validateField(this.adventure$serialize(page), class_1843.field_30932, "page")));
    }
    bookTag.method_10566(class_1843.field_30938, pages);
    bookTag.method_10556(class_1843.field_30941, true); // todo: any parseable texts?

    final class_1799 previous = this.player.method_31548().method_7391();
    this.sendPacket(new class_2653(-2, this.player.field_7512.method_37421(), this.player.method_31548().field_7545, bookStack));
    this.player.method_7315(bookStack, class_1268.field_5808);
    this.sendPacket(new class_2653(-2, this.player.field_7512.method_37421(), this.player.method_31548().field_7545, previous));
  }

  private static String validateField(final String content, final int length, final String name) {
    if (content == null) {
      return content;
    }

    final int actual = content.length();
    if (actual > length) {
      throw new IllegalArgumentException("Field '" + name + "' has a maximum length of " + length + " but was passed '" + content + "', which was " + actual + " characters long.");
    }
    return content;
  }

  private String adventure$plain(final @NotNull Component component) {
    return PlainTextComponentSerializer.plainText().serialize(this.controller.renderer().render(component, this));
  }

  private String adventure$serialize(final @NotNull Component component) {
    return GsonComponentSerializer.gson().serialize(this.controller.renderer().render(component, this));
  }

  @Override
  public void showTitle(final @NotNull Title title) {
    if (title.subtitle() != Component.empty()) {
      this.sendPacket(new class_5903(this.controller.toNative(title.subtitle())));
    }

    final Title.@Nullable Times times = title.times();
    if (times != null) {
      final int fadeIn = ticks(times.fadeIn());
      final int fadeOut = ticks(times.fadeOut());
      final int dwell = ticks(times.stay());
      if (fadeIn != -1 || fadeOut != -1 || dwell != -1) {
        this.sendPacket(new class_5905(fadeIn, dwell, fadeOut));
      }
    }

    if (title.title() != Component.empty()) {
      this.sendPacket(new class_5904(this.controller.toNative(title.title())));
    }
  }

  @Override
  public <T> void sendTitlePart(final @NotNull TitlePart<T> part, final @NotNull T value) {
    Objects.requireNonNull(value, "value");
    if (part == TitlePart.TITLE) {
      this.sendPacket(new class_5904(this.controller.toNative((Component) value)));
    } else if (part == TitlePart.SUBTITLE) {
      this.sendPacket(new class_5903(this.controller.toNative((Component) value)));
    } else if (part == TitlePart.TIMES) {
      final Title.Times times = (Title.Times) value;
      final int fadeIn = ticks(times.fadeIn());
      final int fadeOut = ticks(times.fadeOut());
      final int dwell = ticks(times.stay());
      if (fadeIn != -1 || fadeOut != -1 || dwell != -1) {
        this.sendPacket(new class_5905(fadeIn, dwell, fadeOut));
      }
    } else {
      throw new IllegalArgumentException("Unknown TitlePart '" + part + "'");
    }
  }

  static int ticks(final @NotNull Duration duration) {
    return duration.getSeconds() == -1 ? -1 : (int) (duration.toMillis() / 50);
  }

  @Override
  public void clearTitle() {
    this.sendPacket(new class_5888(false));
  }

  @Override
  public void resetTitle() {
    this.sendPacket(new class_5888(true));
  }

  @Override
  public void sendPlayerListHeader(final @NotNull Component header) {
    requireNonNull(header, "header");
    ((ServerPlayerBridge) this.player).bridge$updateTabList(this.controller.toNative(header), null);
  }

  @Override
  public void sendPlayerListFooter(final @NotNull Component footer) {
    requireNonNull(footer, "footer");
    ((ServerPlayerBridge) this.player).bridge$updateTabList(null, this.controller.toNative(footer));

  }

  @Override
  public void sendPlayerListHeaderAndFooter(final @NotNull Component header, final @NotNull Component footer) {
    ((ServerPlayerBridge) this.player).bridge$updateTabList(
      this.controller.toNative(requireNonNull(header, "header")),
      this.controller.toNative(requireNonNull(footer, "footer")));
  }

  @Override
  public void sendResourcePacks(final @NotNull ResourcePackRequest request) {
    final List<class_2596<class_8705>> packets = new ArrayList<>(request.packs().size());
    if (request.replace()) {
      packets.add(new class_9053(Optional.empty()));
    }

    final net.minecraft.@Nullable class_2561 prompt = this.toNativeNullable(request.prompt());
    for (final Iterator<ResourcePackInfo> it = request.packs().iterator(); it.hasNext();) {
      final ResourcePackInfo pack = it.next();
      packets.add(new class_2720(
        pack.id(),
        pack.uri().toASCIIString(),
        pack.hash(),
        request.required(),
        it.hasNext() ? null : prompt
      ));

      if (request.callback() != ResourcePackCallback.noOp()) {
        ((ServerCommonPacketListenerImplBridge) this.player.field_13987).adventure$bridge$registerPackCallback(pack.id(), this.controller, request.callback());
      }
    }

    this.sendBundle(packets);
  }

  @Override
  public void removeResourcePacks(final @NotNull UUID id, final @NotNull UUID@NotNull... others) {
    this.sendBundle(
      MonkeyBars.nonEmptyArrayToList(pack -> new class_9053(Optional.of(pack)), id, others)
    );
  }

  @Override
  public void clearResourcePacks() {
    this.sendPacket(new class_9053(Optional.empty()));
  }

  private net.minecraft.@Nullable class_2561 toNativeNullable(final @Nullable Component comp) {
    return comp == null ? null : this.controller.toNative(comp);
  }

  @Override
  public @NotNull Pointers pointers() {
    return ((PointerProviderBridge) this.player).adventure$pointers();
  }

  @Override
  public @NotNull FabricAudiencesInternal controller() {
    return this.controller;
  }
}
