/*
 * This file is part of adventure-platform-fabric, licensed under the MIT License.
 *
 * Copyright (c) 2023 KyoriPowered
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package net.kyori.adventure.platform.fabric;

import java.util.UUID;
import net.kyori.adventure.identity.Identity;
import net.kyori.adventure.platform.fabric.impl.ControlledAudience;
import net.kyori.adventure.platform.fabric.impl.server.FabricServerAudiencesImpl;
import net.kyori.adventure.resource.ResourcePackCallback;
import net.kyori.adventure.resource.ResourcePackStatus;
import net.kyori.adventure.text.Component;
import net.kyori.adventure.text.logger.slf4j.ComponentLogger;
import net.minecraft.class_3222;
import net.minecraft.class_8609;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import static java.util.Objects.requireNonNull;

/**
 * Resource pack callbacks for the Fabric environment.
 *
 * @since 5.11.0
 */
public final class FabricResourcePackCallback {
  private static final ComponentLogger LOGGER = ComponentLogger.logger();
  private static final Component DEFAULT_KICK_MESSAGE = Component.translatable("multiplayer.requiredTexturePrompt.disconnect"); // vanilla translation

  private FabricResourcePackCallback() {
  }

  /**
   * Kick a resource pack receiver if they do not accept a resource pack.
   *
   * @return the kicking callback
   * @since 5.11.0
   */
  public static @NotNull ResourcePackCallback kickIfNotApplied() {
    return kickIfNotApplied(DEFAULT_KICK_MESSAGE);
  }

  /**
   * Kick a resource pack receiver if they do not accept a resource pack.
   *
   * @param kickMessage the message to kick the player with
   * @return the kicking callback
   * @since 5.11.0
   */
  public static @NotNull ResourcePackCallback kickIfNotApplied(final @NotNull Component kickMessage) {
    requireNonNull(kickMessage, "kickMessage");

    return (uuid, status, audience) -> {
      if (!status.intermediate() && status != ResourcePackStatus.SUCCESSFULLY_LOADED) { // we've reached a terminal, non-successful state
        // now we attempt to extract a connection that can be kicked -- assuming that it's players that will be sent a resource pack
        if (!(audience instanceof ControlledAudience controlled)) {
          LOGGER.debug("Audience {} was not a ControlledAudience, we cannot kick them", audience);
          return;
        }
        final class_8609 kicker;
        if (audience instanceof class_3222 player) { // when you send a resource pack via the Audience implemented on ServerPlayer
          kicker = player.field_13987;
        } else {
          final @Nullable UUID id = audience.get(Identity.UUID).orElse(null);
          if (id == null) return; // not a player?
          if (!(controlled.controller() instanceof FabricServerAudiencesImpl server)) return;

          final class_3222 ply = server.server().method_3760().method_14602(id);
          if (ply == null) return;

          kicker = ply.field_13987;
        }

        LOGGER.debug("Audience {} did not successfully apply a resource pack with ID {}, kicking with message: {}", kicker.method_52404(), uuid, kickMessage);
        kicker.method_52396(controlled.controller().toNative(kickMessage));
      }
    };
  }
}
