/*
 * This file is part of adventure-platform-fabric, licensed under the MIT License.
 *
 * Copyright (c) 2020-2024 KyoriPowered
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package net.kyori.adventure.platform.fabric.impl.client;

import java.net.MalformedURLException;
import java.time.Duration;
import java.util.Objects;
import java.util.UUID;
import net.kyori.adventure.audience.MessageType;
import net.kyori.adventure.bossbar.BossBar;
import net.kyori.adventure.chat.ChatType;
import net.kyori.adventure.chat.SignedMessage;
import net.kyori.adventure.identity.Identity;
import net.kyori.adventure.inventory.Book;
import net.kyori.adventure.key.Key;
import net.kyori.adventure.platform.fabric.FabricAudiences;
import net.kyori.adventure.platform.fabric.impl.AdventureCommon;
import net.kyori.adventure.platform.fabric.impl.ControlledAudience;
import net.kyori.adventure.platform.fabric.impl.FabricAudiencesInternal;
import net.kyori.adventure.platform.fabric.impl.GameEnums;
import net.kyori.adventure.platform.fabric.impl.PointerProviderBridge;
import net.kyori.adventure.platform.fabric.impl.accessor.minecraft.world.level.LevelAccess;
import net.kyori.adventure.platform.fabric.impl.client.mixin.minecraft.resources.sounds.AbstractSoundInstanceAccess;
import net.kyori.adventure.pointer.Pointered;
import net.kyori.adventure.pointer.Pointers;
import net.kyori.adventure.resource.ResourcePackInfo;
import net.kyori.adventure.resource.ResourcePackRequest;
import net.kyori.adventure.resource.ResourcePackStatus;
import net.kyori.adventure.sound.Sound;
import net.kyori.adventure.sound.SoundStop;
import net.kyori.adventure.text.Component;
import net.kyori.adventure.title.Title;
import net.kyori.adventure.title.TitlePart;
import net.minecraft.class_1106;
import net.minecraft.class_1109;
import net.minecraft.class_1113;
import net.minecraft.class_1297;
import net.minecraft.class_1659;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_3417;
import net.minecraft.class_3419;
import net.minecraft.class_3872;
import net.minecraft.class_5819;
import net.minecraft.class_746;
import net.minecraft.class_7469;
import net.minecraft.class_7591;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class ClientAudience implements ControlledAudience {
  private final class_310 client;
  private final FabricClientAudiencesImpl controller;

  public ClientAudience(final class_310 client, final FabricClientAudiencesImpl renderer) {
    this.client = client;
    this.controller = renderer;
  }

  @Override
  public @NotNull FabricAudiencesInternal controller() {
    return this.controller;
  }

  @Override
  public void sendMessage(final @NotNull Component message) {
    this.client.field_1705.method_1743().method_1812(this.controller.toNative(message));
  }

  private net.minecraft.class_2556.class_7602 toMc(final ChatType.Bound bound) {
    return AdventureCommon.chatTypeToNative(bound, this.controller);
  }

  @Override
  public void sendMessage(final @NotNull Component message, final ChatType.@NotNull Bound boundChatType) {
    final net.minecraft.class_2556.class_7602 bound = this.toMc(boundChatType);
    this.client.field_1705.method_1743().method_44811(bound.method_44837(this.controller.toNative(message)), null, class_7591.method_44709());
  }

  @Override
  public void sendMessage(final @NotNull SignedMessage signedMessage, final ChatType.@NotNull Bound boundChatType) {
    final net.minecraft.class_2556.class_7602 bound = this.toMc(boundChatType);
    final Component message = Objects.requireNonNullElse(signedMessage.unsignedContent(), Component.text(signedMessage.message()));

    this.client.field_1705.method_1743().method_44811(bound.method_44837(this.controller.toNative(message)), (class_7469) signedMessage.signature(), this.tag(signedMessage));
  }

  private class_7591 tag(final SignedMessage message) {
    if (message == null) {
      return null;
    } else if (message.isSystem()) {
      return class_7591.method_44751();
    } else if (message.unsignedContent() != null && !message.unsignedContent().equals(Component.text(message.message()))) {
      return class_7591.method_44710(message.message());
    } else if (message.signature() == null) {
      return class_7591.method_44709();
    }

    return null;
  }

  @Override
  public void deleteMessage(final SignedMessage.@NotNull Signature signature) {
    this.client.field_1705.method_1743().method_44812((class_7469) signature);
  }

  @Override
  @Deprecated
  public void sendMessage(final Identity source, final @NotNull Component message, final @NotNull MessageType type) {
    if (this.client.method_29042(source.uuid())) return;

    final class_1659 visibility = this.client.field_1690.method_42539().method_41753();
    if (type == MessageType.CHAT) {
      // Add to chat queue (following delay and such)
      if (visibility == class_1659.field_7538) {
        this.client.field_1705.method_1743().method_44811(this.controller.toNative(message), null, null);
      }
    } else {
      // Add immediately as a system message
      if (visibility == class_1659.field_7538 || visibility == class_1659.field_7539) {
        this.client.field_1705.method_1743().method_1812(this.controller.toNative(message));
      }
    }
  }

  @Override
  public void sendActionBar(final @NotNull Component message) {
    this.client.field_1705.method_1758(this.controller.toNative(message), false);
  }

  @Override
  public void showTitle(final @NotNull Title title) {
    final net.minecraft.@Nullable class_2561 titleText = title.title() == Component.empty() ? null : this.controller.toNative(title.title());
    final net.minecraft.@Nullable class_2561 subtitleText = title.subtitle() == Component.empty() ? null : this.controller.toNative(title.subtitle());
    final Title.@Nullable Times times = title.times();
    this.client.field_1705.method_34004(titleText);
    this.client.field_1705.method_34002(subtitleText);
    this.client.field_1705.method_34001(
      this.adventure$ticks(times == null ? null : times.fadeIn()),
      this.adventure$ticks(times == null ? null : times.stay()),
      this.adventure$ticks(times == null ? null : times.fadeOut())
    );
  }

  @Override
  public <T> void sendTitlePart(final @NotNull TitlePart<T> part, final @NotNull T value) {
    Objects.requireNonNull(value, "value");
    if (part == TitlePart.TITLE) {
      this.client.field_1705.method_34004(this.controller.toNative((Component) value));
    } else if (part == TitlePart.SUBTITLE) {
      this.client.field_1705.method_34002(this.controller.toNative((Component) value));
    } else if (part == TitlePart.TIMES) {
      final Title.Times times = (Title.Times) value;
      this.client.field_1705.method_34001(
        this.adventure$ticks(times.fadeIn()),
        this.adventure$ticks(times.stay()),
        this.adventure$ticks(times.fadeOut())
      );
    } else {
      throw new IllegalArgumentException("Unknown TitlePart '" + part + "'");
    }
  }

  private int adventure$ticks(final @Nullable Duration duration) {
    return duration == null || duration.getSeconds() == -1 ? -1 : (int) (duration.toMillis() / 50);
  }

  @Override
  public void clearTitle() {
    this.client.field_1705.method_34004(null);
    this.client.field_1705.method_34002(null);
  }

  @Override
  public void resetTitle() {
    this.client.field_1705.method_1742();
    this.clearTitle();
  }

  @Override
  public void showBossBar(final @NotNull BossBar bar) {
    BossHealthOverlayBridge.listener(this.client.field_1705.method_1740(), this.controller).add(bar);
  }

  @Override
  public void hideBossBar(final @NotNull BossBar bar) {
    BossHealthOverlayBridge.listener(this.client.field_1705.method_1740(), this.controller).remove(bar);
  }

  private long seed(final @NotNull Sound sound) {
    if (sound.seed().isPresent()) {
      return sound.seed().getAsLong();
    } else {
      final @Nullable class_746 player = this.client.field_1724;
      if (player != null) {
        return ((LevelAccess) player).accessor$threadSafeRandom().method_43055();
      } else {
        return 0l;
      }
    }
  }

  @Override
  public void playSound(final @NotNull Sound sound) {
    final @Nullable class_746 player = this.client.field_1724;
    if (player != null) {
      this.playSound(sound, player.method_23317(), player.method_23318(), player.method_23321());
    } else {
      // not in-game
      this.client.method_1483().method_4873(new class_1109(FabricAudiences.toNative(sound.name()), GameEnums.SOUND_SOURCE.toMinecraft(sound.source()),
        sound.volume(), sound.pitch(), class_5819.method_43049(this.seed(sound)), false, 0, class_1113.class_1114.field_5478, 0, 0, 0, true));
    }
  }

  @Override
  public void playSound(final @NotNull Sound sound, final Sound.@NotNull Emitter emitter) {
    final class_1297 targetEntity;
    if (emitter == Sound.Emitter.self()) {
      targetEntity = this.client.field_1724;
    } else if (emitter instanceof final class_1297 entity) {
      targetEntity = entity;
    } else {
      throw new IllegalArgumentException("Provided emitter '" + emitter + "' was not Sound.Emitter.self() or an Entity");
    }

    // Initialize with a placeholder event
    final class_1106 mcSound = new class_1106(
      class_3417.field_15197,
      GameEnums.SOUND_SOURCE.toMinecraft(sound.source()),
      sound.volume(),
      sound.pitch(),
      targetEntity,
      this.seed(sound)
    );
    // Then apply the ResourceLocation of our real sound event
    ((AbstractSoundInstanceAccess) mcSound).setLocation(FabricAudiences.toNative(sound.name()));

    this.client.method_1483().method_4873(mcSound);
  }

  @Override
  public void playSound(final @NotNull Sound sound, final double x, final double y, final double z) {
    this.client.method_1483().method_4873(new class_1109(
      FabricAudiences.toNative(sound.name()),
      GameEnums.SOUND_SOURCE.toMinecraft(sound.source()),
      sound.volume(),
      sound.pitch(),
      class_5819.method_43049(this.seed(sound)),
      false,
      0,
      class_1113.class_1114.field_5476,
      x,
      y,
      z,
      false
    ));
  }

  @Override
  public void stopSound(final @NotNull SoundStop stop) {
    final @Nullable Key sound = stop.sound();
    final @Nullable class_2960 soundIdent = sound == null ? null : FabricAudiences.toNative(sound);
    final Sound.@Nullable Source source = stop.source();
    final @Nullable class_3419 category = source == null ? null : GameEnums.SOUND_SOURCE.toMinecraft(source);
    this.client.method_1483().method_4875(soundIdent, category);
  }

  @Override
  public void openBook(final @NotNull Book book) {
    this.client.method_1507(new class_3872(new class_3872.class_3931(book.pages().stream().map(this.controller::toNative).toList())));
  }

  @Override
  public void sendPlayerListHeader(final @NotNull Component header) {
    this.client.field_1705.method_1750().method_1925(header == Component.empty() ? null : this.controller.toNative(header));
  }

  @Override
  public void sendPlayerListFooter(final @NotNull Component footer) {
    this.client.field_1705.method_1750().method_1925(footer == Component.empty() ? null : this.controller.toNative(footer));
  }

  @Override
  public void sendPlayerListHeaderAndFooter(final @NotNull Component header, final @NotNull Component footer) {
    this.sendPlayerListHeader(header);
    this.sendPlayerListFooter(footer);
  }

  @Override
  public void sendResourcePacks(final @NotNull ResourcePackRequest request) {
    if (request.replace()) {
      this.client.method_1516().method_55537();
    }

    for (final ResourcePackInfo info : request.packs()) {
      ListeningPackFeedbackWrapper.registerCallback(info.id(), request.callback(), this);

      try {
        this.client.method_1516().method_55523(info.id(), info.uri().toURL(), info.hash());
      } catch (final MalformedURLException ex) {
        request.callback().packEventReceived(info.id(), ResourcePackStatus.INVALID_URL, this);
      }
      // TODO: required, prompting?
    }
  }

  @Override
  public void removeResourcePacks(final @NotNull UUID id, final @NotNull UUID @NotNull ... others) {
    this.client.method_1516().method_55520(id);
    for (final UUID other : others) {
      this.client.method_1516().method_55520(other);
    }
  }

  @Override
  public void clearResourcePacks() {
    this.client.method_1516().method_55537();
  }

  @Override
  public @NotNull Pointers pointers() {
    final @Nullable class_746 clientPlayer = this.client.field_1724;
    if (clientPlayer != null) {
      return ((PointerProviderBridge) clientPlayer).adventure$pointers();
    } else {
      return ((Pointered) this.client).pointers();
    }
  }
}
