/*
 * This file is part of adventure-platform-fabric, licensed under the MIT License.
 *
 * Copyright (c) 2024 KyoriPowered
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package net.kyori.adventure.platform.fabric.impl.service;

import com.google.auto.service.AutoService;
import com.google.gson.JsonElement;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.serialization.JsonOps;
import java.util.List;
import net.kyori.adventure.key.Key;
import net.kyori.adventure.platform.fabric.FabricAudiences;
import net.kyori.adventure.platform.fabric.impl.AdventureCommon;
import net.kyori.adventure.platform.fabric.impl.nbt.FabricDataComponentValue;
import net.kyori.adventure.text.event.DataComponentValue;
import net.kyori.adventure.text.event.DataComponentValueConverterRegistry;
import net.kyori.adventure.text.serializer.gson.GsonDataComponentValue;
import net.minecraft.class_2509;
import net.minecraft.class_2520;
import net.minecraft.class_7923;
import net.minecraft.class_9331;
import org.jetbrains.annotations.NotNull;

@AutoService(DataComponentValueConverterRegistry.Provider.class)
public final class DataComponentValueConverterProvider implements DataComponentValueConverterRegistry.Provider {
  private static final Key ID = AdventureCommon.res("platform/fabric");

  @Override
  public @NotNull Key id() {
    return ID;
  }

  @Override
  @SuppressWarnings({"rawtypes", "unchecked"})
  public @NotNull Iterable<DataComponentValueConverterRegistry.Conversion<?, ?>> conversions() {
    return List.of(
      DataComponentValueConverterRegistry.Conversion.convert(
        FabricDataComponentValue.Present.class,
        GsonDataComponentValue.class,
        (k, codec) -> GsonDataComponentValue.gsonDataComponentValue((JsonElement) codec.codec().encodeStart(JsonOps.INSTANCE, codec.value()).getOrThrow())
      ),
      DataComponentValueConverterRegistry.Conversion.convert(
        GsonDataComponentValue.class,
        FabricDataComponentValue.class,
        (k, gson) -> {
          final class_9331<?> type = resolveComponentType(k);
          return new FabricDataComponentValue.Present(type.method_57876().parse(JsonOps.INSTANCE, gson.element()).getOrThrow(RuntimeException::new), type.method_57876());
        }
      ),
      DataComponentValueConverterRegistry.Conversion.convert(
        DataComponentValue.TagSerializable.class,
        FabricDataComponentValue.class,
        (k, tagSerializable) -> {
          final class_9331<?> type = resolveComponentType(k);
          final class_2520 decodedSnbt;
          try {
            decodedSnbt = tagSerializable.asBinaryTag().get(FabricDataComponentValue.SNBT_CODEC);
          } catch (final CommandSyntaxException ex) {
            throw new IllegalArgumentException("Unable to parse SNBT value", ex);
          }

          return new FabricDataComponentValue.Present(type.method_57876().parse(class_2509.field_11560, decodedSnbt).getOrThrow(RuntimeException::new), type.method_57876());
        }
      ),
      DataComponentValueConverterRegistry.Conversion.convert(
        DataComponentValue.Removed.class,
        FabricDataComponentValue.class,
        (k, $) -> FabricDataComponentValue.Removed.INSTANCE
      )
    );
  }

  private static class_9331<?> resolveComponentType(final Key key) {
    final class_9331<?> type = class_7923.field_49658.method_10223(FabricAudiences.toNative(key));
    if (type == null) {
      throw new IllegalArgumentException("Unknown data component type " + key.asString());
    }

    return type;
  }
}
