package net.leanix.dropkit.api;

import net.leanix.dropkit.oauth.ClientCredentialAccessTokenFactory;
import net.leanix.dropkit.oauth.ClientCredentialsApiAuthFilter;
import net.leanix.dropkit.oauth.OAuth2ClientConfig;

import com.fasterxml.jackson.jaxrs.json.JacksonJsonProvider;
import org.glassfish.jersey.client.ClientConfig;
import org.glassfish.jersey.client.ClientProperties;
import org.glassfish.jersey.filter.LoggingFilter;

import java.net.URI;

import javax.ws.rs.client.ClientBuilder;

/**
 * Client factory.
 *
 * Returns a jersey api client with configured with client credential
 * authentication.
 */
public class ClientFactory {

    /**
     * Factory method to create an oauth2 api client used by an instance.
     * No HTTP traffic logging.
     *
     * @param config
     * @return
     */
    public static Client create(OAuth2ClientConfig config) {
        return create(config, false);
    }

    /**
     * Factory method to create an oauth2 api client used by an instance,
     * optionally installing a logging filter that emits all HTTP traffic to
     * java util logging.
     *
     * @param config
     * @param enableHttpLogging switch HTTP traffic logging off or on
     * @return
     */
    public static Client create(OAuth2ClientConfig config, boolean enableHttpLogging) {
        URI tokenURI = URI.create(config.getTokenUrl());

        ClientConfig cc = new ClientConfig();
        cc.property(ClientProperties.FOLLOW_REDIRECTS, Boolean.TRUE);
        if (enableHttpLogging) {
            cc.register(LoggingFilter.class);
        }
        cc.register(getAuthFilter(ClientCredentialAccessTokenFactory.create(config)));
        cc.register(JacksonJsonProvider.class);

        javax.ws.rs.client.Client c = ClientBuilder.newClient(cc);

        return new Client(config.getBaseUrl(), c);
    }

    /**
     * Factory method to create an api client used by an instance.
     *
     * @param baseUrl like https://mtm.leanix.net/api/v1 , without resource
     * path.
     * @param tokenUrl
     * @param verificationUrl
     * @param clientId
     * @param clientSecret
     * @return
     */
    public static Client create(
            final String baseUrl,
            final String tokenUrl,
            final String verificationUrl,
            final String clientId,
            final String clientSecret
    ) {
        return create(baseUrl, tokenUrl, verificationUrl, clientId, clientSecret, false);
    }

    /**
     * Factory method to create an api client used by an instance,
     * optionally installing a logging filter that emits all HTTP traffic to
     * java util logging.
     *
     * @param baseUrl like https://mtm.leanix.net/api/v1 , without resource
     * path.
     * @param tokenUrl
     * @param introspectionUrl
     * @param clientId
     * @param clientSecret
     * @param enableHttpLogging switch HTTP traffic logging off or on
     * @return
     */
    public static Client create(
            final String baseUrl,
            final String tokenUrl,
            final String introspectionUrl,
            final String clientId,
            final String clientSecret,
            final boolean enableHttpLogging
    ) {
        OAuth2ClientConfig config = new OAuth2ClientConfig();
        config.setBaseUrl(baseUrl);
        config.setTokenUrl(tokenUrl);
        config.setIntrospectionUrl(introspectionUrl);
        config.setClientId(clientId);
        config.setClientSecret(clientSecret);

        return create(config, enableHttpLogging);
    }

    /**
     * Factory method to create an api client that uses no authentication.
     *
     * @param baseUrl like https://mtm.leanix.net/api/v1 , without resource
     * path.
     * @return
     */
    public static Client create(String baseUrl) {
        javax.ws.rs.client.Client c = createJerseyClientWithJacksonSerializer();
        return new Client(baseUrl, c);
    }

    /**
     * Creates a jersey client that can read application/json bodies.
     *
     * @return
     */
    public static javax.ws.rs.client.Client createJerseyClientWithJacksonSerializer() {
        ClientConfig cc = new ClientConfig();
        cc.property(ClientProperties.FOLLOW_REDIRECTS, true);
        cc.register(JacksonJsonProvider.class);
        return ClientBuilder.newClient(cc);
    }

    private static ClientCredentialsApiAuthFilter getAuthFilter(ClientCredentialAccessTokenFactory factory) {
        return new ClientCredentialsApiAuthFilter(factory);
    }
}
