package net.leanix.dropkit.oauth;

import net.leanix.dropkit.api.ClientFactory;
import net.leanix.dropkit.oauth.responses.AccessTokenResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.ws.rs.client.Client;

import java.nio.charset.Charset;
import java.util.Base64;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;

/**
 * Provides an access token for two legged oauth.
 *
 *
 */
public class ClientCredentialAccessTokenFactory {
    private static final Charset UTF8 = Charset.forName("UTF-8");

    private final OAuth2ClientConfig config;
    private final String authorization;
    private final Client apiClient;

    private final Logger logger = LoggerFactory.getLogger(ClientCredentialAccessTokenFactory.class);
    private AccessTokenResponse accessTokenResponse;

    /**
     * Factory method.
     *
     * @param config
     * @return
     */
    public static ClientCredentialAccessTokenFactory create(OAuth2ClientConfig config) {
        Client apiClient = ClientFactory.createJerseyClientWithJacksonSerializer();
        return new ClientCredentialAccessTokenFactory(config, apiClient);
    }

    public ClientCredentialAccessTokenFactory(
            OAuth2ClientConfig config,
            Client apiClient
    ) {
        StringBuilder sb = new StringBuilder(512);
        sb.append(config.getClientId()).append(':').append(config.getClientSecret());
        String userAndPw = sb.toString();

        sb.setLength(0);
        sb.append("Basic ").append(
                Base64.getEncoder().encodeToString(userAndPw.getBytes(UTF8))
        );
        authorization = sb.toString();
        this.config = config;
        this.apiClient = apiClient;
    }

    public String getAccessToken() throws FlowException {
        if (accessTokenResponse == null) {
            fetchToken();
        } else if (accessTokenResponse.isExpired()) {
            fetchToken();
        }

        return accessTokenResponse.getAccessToken();
    }

    private void fetchToken() throws FlowException {
        try {
            accessTokenResponse = apiClient
                    .target(config.getTokenUrl().concat("?grant_type=client_credentials"))
                    .request(MediaType.APPLICATION_JSON)
                    .header(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON)
                    .header(HttpHeaders.AUTHORIZATION, authorization)
                    .post(null, AccessTokenResponse.class);
        } catch (RuntimeException ex) {
            throwFlowException("Failed to retrieve a new oauth token from " + config.getTokenUrl() + " using " + authorization, ex);
        }
    }

    private void throwFlowException(String message, Exception ex) throws FlowException {
        logger.error(message, ex);
        throw new FlowException(message);
    }
}
