package LinkFuture.Compressor;

import LinkFuture.Compressor.Model.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.io.IOException;
import java.io.StringWriter;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Created by Cyokin on 3/6/14.
 * CrunchFileController
 */
public class CrunchFileController {
    protected static final Logger logger = LoggerFactory.getLogger(CrunchFileController.class);
    private static final ConcurrentHashMap<String,String> StaticOutputCache = new ConcurrentHashMap<>();
    public synchronized static String buildStaticFileReference(String outputFileName, String crunchFilePath, boolean isDebug, StaticFileType type, boolean isUnblocked, boolean isReference) throws IOException {
        StringBuilder sw = new StringBuilder();
        sw.append(outputFileName);
        sw.append(crunchFilePath);
        sw.append(isDebug);
        sw.append(type.toString());
        sw.append(isUnblocked);
        sw.append(isReference);
        String key = sw.toString().toLowerCase();
        if(!StaticOutputCache.containsKey(key))
        {
            logger.trace("Build {}", outputFileName);
            CrunchFileMetaInfo crunchFileList =  Utility.fromXml(Utility.getStringFromFile(crunchFilePath), CrunchFileMetaInfo.class);
            if(crunchFileList==null || crunchFileList.CrunchFileListGroup==null)
            {
                throw new IllegalArgumentException("Specific file not exist or not standard format, " + crunchFilePath);
            }
            StringBuilder staticReference = new StringBuilder();
            for(CrunchFileListInfo list : crunchFileList.CrunchFileListGroup)
            {
                if(list.Name.equalsIgnoreCase(outputFileName))
                {
                    if(list.RemoteFileList!=null)
                    {
                        for(CrunchFileInfo myFile : list.RemoteFileList)
                        {
                            staticReference.append(getStaticFileOutput(type, myFile.FilePath, isUnblocked, isReference));
                        }
                    }

                    if(isDebug)
                    {
                        if(list.CrunchFileList!=null){
                            for(CrunchFileInfo myFile : list.CrunchFileList)
                            {
                                staticReference.append(getStaticFileOutput(type, myFile.FilePath, isUnblocked, isReference));
                            }
                        }
                    }
                    else
                    {
                        staticReference.append(getStaticFileOutput(type, list.OutputFilePath, isUnblocked, isReference));
                    }
                    break;
                }
            }
            String output = staticReference.toString();
            if (staticReference.length() > 0 && isUnblocked && type == StaticFileType.JavaScript && isReference)
            {
                output = "[" + output.substring(0,output.length() -1) + "]";
            }
            StaticOutputCache.put(key,output);
            return output;
        }
        return StaticOutputCache.get(key);
    }
    private static String getStaticFileOutput(StaticFileType type, String outputFileName, boolean isUnblocked, boolean isReference)
    {
        switch (type)
        {
            case JavaScript:
                String jsFilePath = Utility.getJSUri(outputFileName);
                String jsFileReferenceUrl = outputFileName.startsWith("//")?outputFileName:jsFilePath;
                if (isUnblocked && isReference)
                {
                    return String.format("\"%s\",", jsFileReferenceUrl);
                }
                if (!isReference)
                {
                    try {
                        return String.format("<script type=\"text/javascript\">%s</script>", Utility.getStringFromUrl(Utility.getJSConfigPath(outputFileName)));
                    } catch (Exception e) {
                        return String.format("<script type=\"text/javascript\">/*download js file failed(%s) %s */</script>",outputFileName, Utility.getDetail(e));
                    }
                }
                return String.format("<script type=\"text/javascript\" src=\"%s\"></script>", jsFileReferenceUrl);
            case Css:
                String cssFilePath = Utility.getCssUri(outputFileName);
                String cssFileReferenceUrl = outputFileName.startsWith("//")?outputFileName:cssFilePath;
                if (!isReference)
                {
                    try {
                        return String.format("<style type=\"text/css\">%s</style>", Utility.getStringFromUrl(Utility.getCssConfigPath(outputFileName)));
                    } catch (IOException e) {
                        return String.format("<style type=\"text/css\">/*download css file failed(%s)*/</style>",cssFilePath);
                    }
                }
                return String.format("<link rel=\"Stylesheet\" type=\"text/css\" href=\"%s\" />", cssFileReferenceUrl);
            default:
                throw new IllegalArgumentException("type");
        }
    }

    //region Crunch File Manager
    @SuppressWarnings("UnusedDeclaration")
    public static String buildJsFileReference(String outputFileName) throws IOException {
        return  buildJsFileReference(outputFileName, Config.isDebug(), false, true);
    }
    public static String buildInlineJsFileReference(String outputFileName) throws IOException {
        return  buildJsFileReference(outputFileName, Config.isDebug(), false, false);
    }
    public static String buildUnlockJsFileReference(String loadJSFileName,String outputFileName,String callbackFunction) throws IOException {
        String scriptFileArray = buildJsFileReference(outputFileName, Config.isDebug(), true, true);
        StringWriter sb = new StringWriter();
        sb.append(buildInlineJsFileReference(loadJSFileName));
        if(scriptFileArray.length()>0)
        {
            sb.append("<script language=\"javascript\"> $loadJS(");
            sb.append(scriptFileArray);
            sb.append(String.format(",function(){%s});",callbackFunction));
            sb.append("</script>");
        }
        return sb.toString();
    }
    public static String buildCssFileReference(String outputFileName) throws IOException {
        return  buildCssFileReference(outputFileName, Config.isDebug(), false, true);
    }

    public static String buildJsFileReference(String outputFileName,boolean isDebug,boolean isUnblocked,boolean isReference) throws IOException {
        return CrunchFileController.buildStaticFileReference(outputFileName, Config.getJsCrunchFilePath(), isDebug, StaticFileType.JavaScript, isUnblocked, isReference);
    }
    public static String buildCssFileReference(String outputFileName,boolean isDebug,boolean isUnblocked,boolean isReference) throws IOException {
        return CrunchFileController.buildStaticFileReference(outputFileName, Config.getCssCrunchFilePath(), isDebug, StaticFileType.Css, isUnblocked, isReference);
    }
    //endregion
}
