package LinkFuture.EnvInjection;

import LinkFuture.EnvInjection.Model.EnvironmentInfo;
import LinkFuture.EnvInjection.Model.InjectionInfo;
import LinkFuture.EnvInjection.Model.InjectionSettingInfo;
import LinkFuture.EnvInjection.Model.ItemInfo;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Properties;

/**
 * Created by Cyokin on 10/24/2014.
 */
public class EnvInjectionController {

    public static void main(String args[]) throws Exception {
        WriteLog("Start injection");
        if(args.length!=4)
        {
            usage();
            System.exit(1);
        }
        HashMap<String,String> params = findParam(args);
        String filePath = params.get("xml");
        String env = params.get("env");
        String defaultEnv = params.get("def");
        if (filePath == null) {
            usage();
            System.exit(1);
        }
        Path xmlPath = Paths.get(filePath).toAbsolutePath();
        WriteLog("Reading configuration file '%s'", xmlPath);
        if(!Files.exists(xmlPath))
        {
            WriteError("Specific file '%s' can't be found", xmlPath);
            System.exit(1);
        }
        String xmlFolder = xmlPath.getParent().toString();
        InjectionSettingInfo meta = Utility.fromXml(Utility.getStringFromFile(filePath), InjectionSettingInfo.class);
        Injection(env,meta,xmlFolder);
        WriteLog("End injection successfully!");
    }
    private static HashMap<String,String> findParam(String args[]){
        HashMap<String,String> output = new HashMap();
        String key = null;
        for(int i=0;i<args.length; i++)
        {
            if(args[i]!=null && args[i].startsWith("--"))
            {
                key = args[i].substring(2);
            }
            else if (key!=null)
            {
                output.put(key,args[i]);
            }
        }
        return output;
    }
    private static void usage(){
        WriteError("Usage: java -jar linkfuture.envInjection.jar --env <environment in file path> --xml <filePath>");
    }
    public static void Injection(String env,InjectionSettingInfo meta,String xmlFolder) throws Exception {
        if(meta.EnvironmentList==null || meta.EnvironmentList.size()==0)
        {
            WriteError("You don't have any environments setting");
            System.exit(1);
        }
        if(env==null || env.length()==0)
        {
            for(EnvironmentInfo setting:meta.EnvironmentList.values())
            {
                if(setting.IsDefault)
                {
                    env = setting.Key;
                }
            }
        }
        if(!meta.EnvironmentList.containsKey(env))
        {
            WriteError("Specific environment '%s' can't be found in xml file", env);
            System.exit(1);
        }
        EnvironmentInfo environment = meta.EnvironmentList.get(env);
        WriteLog("Your current environment is %s", env);
        for (InjectionInfo myInject:environment.InjectionList)
        {
            Path injectFile = Paths.get(xmlFolder,myInject.Path).toAbsolutePath();
            if(!Files.exists(injectFile))
            {
                WriteError("Specific file '%s' can't be found, injection failed", injectFile);
                System.exit(1);
            }
            WriteLog("Injecting file %s",injectFile);
            switch (myInject.Type)
            {
                case Property:
                    InjectToProperty(myInject,environment,meta,injectFile);
                    break;
                case Xml:
                    InjectToXml(myInject,environment,meta,injectFile);
                    break;
                case Replace:
                    Replace(myInject, environment, meta, injectFile);
                    break;
            }
        }
    }
    public static void InjectToXml(InjectionInfo myInject,EnvironmentInfo environment,InjectionSettingInfo meta,Path injectFile) throws IOException, SAXException, ParserConfigurationException, XPathExpressionException, TransformerException {
        String filePath = injectFile.toString();
        DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
        DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
        Document doc = documentBuilder.parse(injectFile.toString());

        for (ItemInfo item: myInject.InjectionKeys)
        {
             String myValue = FindValue(item.Value,environment,meta);
            if(myValue==null)
            {
                WriteLog("Specific key '%s' can't be found, injection skipped", item.Value);
            }
            else
            {
                XPath xpath = XPathFactory.newInstance().newXPath();
                NodeList nodeList = ((NodeList) xpath.compile(item.Key).evaluate(doc, XPathConstants.NODESET));
                if(nodeList.getLength()==0)
                {
                    WriteLog("Specific key '%s' can't be found in file '%s', injection skipped", item.Key, filePath);
                }
                for (int i=0;i< nodeList.getLength();i++)
                {
                    Node node = nodeList.item(i);
                    if(node.getNodeType()==1)
                    {
                        node.setTextContent(myValue);
                    }
                    else
                    {
                        node.setNodeValue(myValue);
                    }
                }
            }
        }

        TransformerFactory transformerFactory = TransformerFactory.newInstance("com.sun.org.apache.xalan.internal.xsltc.trax.TransformerFactoryImpl",null);
        Transformer transformer = transformerFactory.newTransformer();
        transformer.transform(new javax.xml.transform.dom.DOMSource(doc), new StreamResult(filePath));
    }
    public static void InjectToProperty(InjectionInfo myInject,EnvironmentInfo environment,InjectionSettingInfo meta,Path injectFile) throws IOException {
        FileInputStream in = new FileInputStream(injectFile.toString());
        Properties props = new Properties();
        props.load(in);
        in.close();

        FileOutputStream out = new FileOutputStream(injectFile.toString());
        for (ItemInfo item: myInject.InjectionKeys)
        {
            String myValue = FindValue(item.Value,environment,meta);
            if(myValue==null)
            {
                WriteLog("Specific key '%s' can't be found, injection skipped", item.Value);
            }
            else
            {
                props.setProperty(item.Key,myValue);
            }
        }
        props.store(out, null);
        out.close();
    }
    public static void Replace(InjectionInfo myInject,EnvironmentInfo environment,InjectionSettingInfo meta,Path injectFile) throws FileNotFoundException {
        String myFile = Utility.getStringFromFile(injectFile.toString());
        for (ItemInfo item: myInject.InjectionKeys)
        {
            String myValue = FindValue(item.Value,environment,meta);
            if(myValue==null)
            {
                WriteLog("Specific key '%s' can't be found, injection skipped", item.Value);
            }
            else
            {
                myFile = myFile.replace("${" + item.Key + "}",myValue);
            }
        }
        Utility.saveStringToFile(injectFile.toString(),myFile,true);
    }
    public static String FindValue(String myValue,EnvironmentInfo environment,InjectionSettingInfo meta) {
        if(myValue.startsWith("${") & myValue.endsWith("}"))
        {
            String key = myValue.substring(2,myValue.length()-1);
            if(environment.PropertyList!=null && environment.PropertyList.containsKey(key))
            {
                return environment.PropertyList.get(key).Value;
            }
            if(meta.PropertyList!=null && meta.PropertyList.containsKey(key))
            {
                return meta.PropertyList.get(key).Value;
            }
            return null;
        }
        else
        {
            return myValue;
        }
    }

    private static void WriteLog(String msg,Object... args){
        if(args.length>0)
        {
            System.out.println("[EnvInjection] " + String.format(msg, args));
        }
        else
        {
            System.out.println("[EnvInjection] " + msg);
        }
    }
    private static void WriteError(String msg,Object... args){
        if(args.length>0)
        {
            System.err.println("[EnvInjection] " + String.format(msg, args));
        }
        else
        {
            System.err.println("[EnvInjection] " + msg);
        }
    }
}
