/*
 * Decompiled with CFR 0.152.
 */
package net.maizegenetics.taxa;

import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.ImmutableSortedMap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Ordering;
import com.google.common.collect.SetMultimap;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import net.maizegenetics.taxa.TaxaList;
import net.maizegenetics.taxa.TaxaListBuilder;
import net.maizegenetics.taxa.TaxaListTableReport;
import net.maizegenetics.taxa.Taxon;
import net.maizegenetics.util.TableReportUtils;
import net.maizegenetics.util.Utils;
import org.apache.log4j.Logger;

public class TaxaListIOUtils {
    private static final Logger myLogger = Logger.getLogger(TaxaListIOUtils.class);
    private static final String DELIMITER = "\t";

    private TaxaListIOUtils() {
    }

    public static Multimap<String, Taxon> getMapOfTaxonByAnnotation(TaxaList taxaList, String annotation) {
        ImmutableMultimap.Builder annoMap = new ImmutableMultimap.Builder().orderKeysBy((Comparator)Ordering.natural());
        for (Taxon taxon : taxaList) {
            for (String value : taxon.getAnnotation().getTextAnnotation(annotation)) {
                annoMap.put((Object)value, (Object)taxon);
            }
        }
        return annoMap.build();
    }

    public static Optional<Map<String, Taxon>> getUniqueMapOfTaxonByAnnotation(TaxaList taxaList, String annotation) {
        TreeMap<String, Taxon> annoMap = new TreeMap<String, Taxon>();
        for (Taxon taxon : taxaList) {
            for (String value : taxon.getAnnotation().getTextAnnotation(annotation)) {
                if (annoMap.containsKey(value)) {
                    return Optional.empty();
                }
                annoMap.put(value, taxon);
            }
        }
        return Optional.of(ImmutableSortedMap.copyOf(annoMap));
    }

    public static TaxaList subsetTaxaListByAnnotation(TaxaList baseTaxaList, String annotation, String annoValue) {
        TaxaListBuilder tlb = new TaxaListBuilder();
        block0: for (Taxon taxon : baseTaxaList) {
            for (String value : taxon.getAnnotation().getTextAnnotation(annotation)) {
                if (!value.equals(annoValue)) continue;
                tlb.add(taxon);
                continue block0;
            }
        }
        return tlb.build();
    }

    public static TaxaList retainSpecificAnnotations(TaxaList baseTaxaList, String[] annotationsToKeep) {
        ImmutableSet keepers = new ImmutableSet.Builder().addAll(Arrays.asList(annotationsToKeep)).build();
        TaxaListBuilder tlb = new TaxaListBuilder();
        for (Taxon taxon : baseTaxaList) {
            Taxon.Builder tb = new Taxon.Builder(taxon.getName());
            for (Map.Entry<String, String> entry : taxon.getAnnotation().getAllAnnotationEntries()) {
                if (!keepers.contains(entry.getKey())) continue;
                tb.addAnno(entry.getKey(), entry.getValue());
            }
            tlb.add(tb.build());
        }
        return tlb.build();
    }

    public static TaxaList removeSpecificAnnotations(TaxaList baseTaxaList, String[] annotationsToRemove) {
        ImmutableSet keepers = new ImmutableSet.Builder().addAll(Arrays.asList(annotationsToRemove)).build();
        TaxaListBuilder tlb = new TaxaListBuilder();
        for (Taxon taxon : baseTaxaList) {
            Taxon.Builder tb = new Taxon.Builder(taxon.getName());
            for (Map.Entry<String, String> entry : taxon.getAnnotation().getAllAnnotationEntries()) {
                if (keepers.contains(entry.getKey())) continue;
                tb.addAnno(entry.getKey(), entry.getValue());
            }
            tlb.add(tb.build());
        }
        return tlb.build();
    }

    public static Set<String> allAnnotationKeys(TaxaList baseTaxaList) {
        ImmutableSet.Builder keepers = new ImmutableSet.Builder();
        for (Taxon taxon : baseTaxaList) {
            for (Map.Entry<String, String> entry : taxon.getAnnotation().getAllAnnotationEntries()) {
                keepers.add((Object)entry.getKey());
            }
        }
        return keepers.build();
    }

    public static void exportAnnotatedTaxaListTable(TaxaList taxa, String filename) {
        try (BufferedWriter writer = Utils.getBufferedWriter(filename);){
            writer.append("<TaxaList>\n");
            TableReportUtils.saveDelimitedTableReport(new TaxaListTableReport(taxa), DELIMITER, writer, true);
        }
        catch (Exception e) {
            myLogger.debug((Object)e.getMessage(), (Throwable)e);
            throw new IllegalStateException("TaxaListIOUtils: exportAnnotatedTaxaListTable: problem saving file: " + filename);
        }
    }

    public static TaxaList importAnnotatedTaxaList(String filename) {
        TaxaListBuilder builder = new TaxaListBuilder();
        try (BufferedReader reader = Utils.getBufferedReader(filename);){
            String line;
            String header = reader.readLine().trim();
            if (!header.equalsIgnoreCase("<TaxaList>")) {
                throw new IllegalArgumentException("TaxaListIOUtils: importAnnotatedTaxaList: This file doesn't start with <TaxaList>: " + filename);
            }
            header = reader.readLine().trim();
            String[] columns = header.split(DELIMITER);
            for (int i = 0; i < columns.length; ++i) {
                columns[i] = columns[i].trim();
            }
            if (!columns[0].equalsIgnoreCase("Taxa")) {
                throw new IllegalArgumentException("TaxaListIOUtils: importAnnotatedTaxaList: First column should be Taxa: " + filename);
            }
            int numColumns = columns.length;
            int lineNum = 2;
            while ((line = reader.readLine()) != null) {
                line = line.trim();
                ++lineNum;
                String[] annotations = line.split(DELIMITER);
                if (numColumns != annotations.length) {
                    throw new IllegalStateException("TaxaListIOUtils: importAnnotatedTaxaList: number of annotations doesn't match number of columns line: " + lineNum + " taxon: " + annotations[0].trim());
                }
                Taxon.Builder currentTaxon = new Taxon.Builder(annotations[0].trim());
                for (int i = 1; i < numColumns; ++i) {
                    String value = annotations[i].trim();
                    if (value.isEmpty()) continue;
                    currentTaxon.addAnno(columns[i], value);
                }
                builder.add(currentTaxon.build());
            }
        }
        catch (Exception e) {
            myLogger.debug((Object)e.getMessage(), (Throwable)e);
            throw new IllegalStateException("TaxaListIOUtils: importAnnotatedTaxaList: Problem reading file: " + filename + "\n" + e.getMessage());
        }
        return builder.build();
    }

    public static TaxaList readTaxaAnnotationFile(String fileName, String taxaNameField, Map<String, String> filters, boolean mergeSameNames) {
        ArrayList<Taxon> taxaAL = TaxaListIOUtils.readTaxaAnnotationFileAL(fileName, taxaNameField, filters);
        if (taxaAL == null) {
            return null;
        }
        TaxaListBuilder tlb = new TaxaListBuilder();
        taxaAL.stream().forEach(taxa -> {
            if (mergeSameNames) {
                tlb.addOrMerge((Taxon)taxa);
            } else {
                tlb.add((Taxon)taxa);
            }
        });
        return tlb.sortTaxaAlphabetically().build();
    }

    public static ArrayList<Taxon> readTaxaAnnotationFileAL(String fileName, String taxaNameField, Map<String, String> filters) {
        try {
            BufferedReader fileIn = Utils.getBufferedReader(fileName, 1000000);
            fileIn.mark(65536);
            String line = fileIn.readLine();
            ArrayList<Taxon> taxaAL = new ArrayList<Taxon>();
            int indexOfName = 0;
            ArrayList<String> headers = new ArrayList<String>();
            ArrayList<Boolean> isQuant = new ArrayList<Boolean>();
            if (line.contains(taxaNameField)) {
                int i = 0;
                for (String header : line.split("\\t")) {
                    if (header.equals(taxaNameField)) {
                        indexOfName = i;
                    }
                    isQuant.add(header.startsWith("#") || header.startsWith("<#"));
                    headers.add(header.replace(">", "").replace("<", "").replace("#", ""));
                    ++i;
                }
            } else {
                fileIn.reset();
            }
            while ((line = fileIn.readLine()) != null) {
                String[] s = line.split("\\t");
                Taxon.Builder anID = new Taxon.Builder(s[indexOfName]);
                for (int i = 0; i < s.length; ++i) {
                    String[] cs;
                    if (i == indexOfName) continue;
                    for (String ta : cs = s[i].split(";")) {
                        if (ta == null || ta.isEmpty()) continue;
                        if (((Boolean)isQuant.get(i)).booleanValue()) {
                            if (ta.equals("NA")) {
                                anID.addAnno((String)headers.get(i), Double.NaN);
                                continue;
                            }
                            anID.addAnno((String)headers.get(i), Double.parseDouble(ta));
                            continue;
                        }
                        anID.addAnno((String)headers.get(i), ta);
                    }
                }
                Taxon t = anID.build();
                if (!TaxaListIOUtils.doesTaxonHaveAllAnnotations(t, filters)) continue;
                taxaAL.add(t);
            }
            Collections.sort(taxaAL, new Comparator<Taxon>(){

                @Override
                public int compare(Taxon taxa1, Taxon taxa2) {
                    return taxa1.getName().compareTo(taxa2.getName());
                }
            });
            return taxaAL;
        }
        catch (Exception e) {
            System.err.println("Error in Reading Annotated Taxon File:" + fileName);
            e.printStackTrace();
            return null;
        }
    }

    public static TaxaList readTaxaAnnotationFile(String fileName, String taxaNameField) {
        return TaxaListIOUtils.readTaxaAnnotationFile(fileName, taxaNameField, new HashMap<String, String>(), false);
    }

    public static boolean doesTaxonHaveAllAnnotations(Taxon taxon, Map<String, String> filters) {
        SetMultimap<String, String> taxonAnno = taxon.getAnnotation().getAnnotationAsMap();
        boolean keep = true;
        for (Map.Entry<String, String> entry : filters.entrySet()) {
            keep = false;
            for (String s1 : taxonAnno.get((Object)entry.getKey())) {
                if (!s1.equals(entry.getValue())) continue;
                keep = true;
            }
            if (keep) continue;
            break;
        }
        return keep;
    }

    public static SetMultimap<String, String> parseVCFHeadersIntoMap(String s) {
        if (s == null) {
            return null;
        }
        if (!s.startsWith("<") || !s.endsWith(">")) {
            return null;
        }
        String value = s.substring(1, s.length() - 1);
        ImmutableSetMultimap.Builder im = new ImmutableSetMultimap.Builder().orderKeysBy((Comparator)Ordering.natural()).orderValuesBy((Comparator)Ordering.natural());
        for (String s1 : Splitter.on((String)",").trimResults().split((CharSequence)value)) {
            String[] ssEntry = s1.split("=", 2);
            im.put((Object)ssEntry[0], (Object)ssEntry[1]);
        }
        return im.build();
    }

    public static List<String> readTissueAnnotationFile(String fileName, String tissueNameField) {
        try {
            BufferedReader fileIn = Utils.getBufferedReader(fileName, 1000000);
            fileIn.mark(65536);
            String line = fileIn.readLine();
            ArrayList<String> tissues = new ArrayList<String>();
            int indexOfTissue = -1;
            if (line.contains(tissueNameField)) {
                int idx = 0;
                for (String header : line.split("\\t")) {
                    if (header.equals(tissueNameField)) {
                        indexOfTissue = idx;
                        break;
                    }
                    ++idx;
                }
                if (indexOfTissue == -1) {
                    return null;
                }
            } else {
                fileIn.reset();
            }
            if (indexOfTissue == -1) {
                return null;
            }
            while ((line = fileIn.readLine()) != null) {
                String[] items = line.split("\\t");
                for (int idx = 0; idx < items.length; ++idx) {
                    if (idx != indexOfTissue || tissues.contains(items[idx])) continue;
                    tissues.add(items[idx]);
                }
            }
            Collections.sort(tissues);
            return tissues;
        }
        catch (Exception e) {
            System.err.println("Error in Reading Annotated Tissue File:" + fileName);
            e.printStackTrace();
            return null;
        }
    }
}

