/*
 * Decompiled with CFR 0.152.
 */
package net.markenwerk.utils.mail.dkim;

import com.sun.mail.util.CRLFOutputStream;
import com.sun.mail.util.QPEncoderStream;
import jakarta.mail.Header;
import jakarta.mail.MessagingException;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.interfaces.RSAPrivateKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Base64;
import java.util.Collection;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Pattern;
import net.i2p.crypto.eddsa.EdDSAPrivateKey;
import net.markenwerk.utils.data.fetcher.BufferedDataFetcher;
import net.markenwerk.utils.data.fetcher.DataFetchException;
import net.markenwerk.utils.mail.dkim.Canonicalization;
import net.markenwerk.utils.mail.dkim.DkimException;
import net.markenwerk.utils.mail.dkim.DkimMessage;
import net.markenwerk.utils.mail.dkim.DkimSigningException;
import net.markenwerk.utils.mail.dkim.DomainKeyUtil;
import net.markenwerk.utils.mail.dkim.KeyPairType;
import net.markenwerk.utils.mail.dkim.SigningAlgorithm;

public class DkimSigner {
    private static final int MAX_HEADER_LENGTH = 67;
    private static final String DKIM_SIGNATUR_HEADER = "DKIM-Signature";
    private static final Pattern SIGNING_DOMAIN_PATTERN = Pattern.compile("(.+)\\.(.+)");
    private static final Set<String> MANDATORY_HEADERS_TO_SIGN = new TreeSet<String>(String.CASE_INSENSITIVE_ORDER);
    private static final Set<String> DEFAULT_HEADERS_TO_SIGN = new HashSet<String>();
    private final Set<String> headersToSign = new TreeSet<String>(String.CASE_INSENSITIVE_ORDER);
    private final String signingDomain;
    private final String selector;
    private final KeyPairType keyPairType;
    private final PrivateKey privateKey;
    private SigningAlgorithm signingAlgorithm;
    private MessageDigest messageDigest;
    private Signature signature;
    private Canonicalization headerCanonicalization;
    private Canonicalization bodyCanonicalization;
    private String identity;
    private boolean lengthParam;
    private boolean copyHeaderFields;
    private boolean checkDomainKey;

    public DkimSigner(String signingDomain, String selector, File derFile) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException, DkimException {
        this(signingDomain, selector, new FileInputStream(derFile));
    }

    public DkimSigner(String signingDomain, String selector, InputStream derStream) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        this(signingDomain, selector, DkimSigner.readPrivateKey(derStream));
    }

    private static RSAPrivateKey readPrivateKey(InputStream derStream) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException {
        byte[] privKeyBytes = new BufferedDataFetcher().fetch(derStream, true);
        KeyFactory rsaKeyFactory = KeyFactory.getInstance("RSA");
        PKCS8EncodedKeySpec privateKeySpec = new PKCS8EncodedKeySpec(privKeyBytes);
        return (RSAPrivateKey)rsaKeyFactory.generatePrivate(privateKeySpec);
    }

    public DkimSigner(String signingDomain, String selector, RSAPrivateKey privateKey) throws DkimException {
        this.checkSigningDomain(signingDomain);
        this.headersToSign.addAll(DEFAULT_HEADERS_TO_SIGN);
        this.signingDomain = signingDomain;
        this.selector = selector.trim();
        this.keyPairType = KeyPairType.RSA;
        this.privateKey = privateKey;
        this.setSigningAlgorithm(this.keyPairType.getDefaultSigningAlgorithm());
        this.setHeaderCanonicalization(Canonicalization.RELAXED);
        this.setBodyCanonicalization(Canonicalization.SIMPLE);
        this.setCheckDomainKey(true);
    }

    public DkimSigner(String signingDomain, String selector, EdDSAPrivateKey privateKey) throws DkimException {
        this.checkSigningDomain(signingDomain);
        this.headersToSign.addAll(DEFAULT_HEADERS_TO_SIGN);
        this.signingDomain = signingDomain;
        this.selector = selector.trim();
        this.keyPairType = KeyPairType.ED25519;
        this.privateKey = privateKey;
        this.keyPairType.initialize();
        this.setSigningAlgorithm(this.keyPairType.getDefaultSigningAlgorithm());
        this.setHeaderCanonicalization(Canonicalization.RELAXED);
        this.setBodyCanonicalization(Canonicalization.SIMPLE);
        this.setCheckDomainKey(true);
    }

    private void checkSigningDomain(String signingDomain) {
        if (null == signingDomain || !SIGNING_DOMAIN_PATTERN.matcher(signingDomain).matches()) {
            throw new DkimException(signingDomain + " is an invalid signing domain");
        }
    }

    public void addHeaderToSign(String header) {
        if (null != header && 0 != header.length()) {
            this.headersToSign.add(header);
        }
    }

    public void removeHeaderToSign(String header) {
        if (null != header && 0 != header.length() && !DkimSigner.isMandatoryHeader(header)) {
            this.headersToSign.remove(header);
        }
    }

    private static boolean isMandatoryHeader(String header) {
        return MANDATORY_HEADERS_TO_SIGN.contains(header);
    }

    public SigningAlgorithm getSigningAlgorithm() {
        return this.signingAlgorithm;
    }

    public void setSigningAlgorithm(SigningAlgorithm signingAlgorithm) throws DkimException {
        if (!this.keyPairType.supportsSigningAlgorithm(signingAlgorithm)) {
            throw new DkimException("Unsupported signing algorithm: " + (Object)((Object)signingAlgorithm));
        }
        try {
            this.messageDigest = MessageDigest.getInstance(signingAlgorithm.getHashNotation());
        }
        catch (NoSuchAlgorithmException e) {
            throw new DkimException("Unknown hashing algorithm: " + signingAlgorithm.getHashNotation(), e);
        }
        try {
            this.signature = Signature.getInstance(signingAlgorithm.getJavaNotation());
            this.signature.initSign(this.privateKey);
        }
        catch (NoSuchAlgorithmException e) {
            throw new DkimException("Unknown signing algorithm " + signingAlgorithm.getJavaNotation(), e);
        }
        catch (InvalidKeyException e) {
            throw new DkimException("Invalid private key", e);
        }
        this.signingAlgorithm = signingAlgorithm;
    }

    public Canonicalization getHeaderCanonicalization() {
        return this.headerCanonicalization;
    }

    public void setHeaderCanonicalization(Canonicalization canonicalization) {
        this.headerCanonicalization = canonicalization;
    }

    public Canonicalization getBodyCanonicalization() {
        return this.bodyCanonicalization;
    }

    public void setBodyCanonicalization(Canonicalization canonicalization) {
        this.bodyCanonicalization = canonicalization;
    }

    public String getIdentity() {
        return this.identity;
    }

    public void setIdentity(String identity) throws DkimException {
        if (null != identity) {
            this.checkIdentity(identity);
        }
        this.identity = identity;
    }

    private void checkIdentity(String identity) {
        if (!identity.endsWith("@" + this.signingDomain) && !identity.endsWith("." + this.signingDomain)) {
            throw new DkimException("The domain part of " + identity + " isn't " + this.signingDomain + " or a subdomain thereof");
        }
    }

    public boolean getLengthParam() {
        return this.lengthParam;
    }

    public void setLengthParam(boolean lengthParam) {
        this.lengthParam = lengthParam;
    }

    @Deprecated
    public boolean isZParam() {
        return this.isCopyHeaderFields();
    }

    @Deprecated
    public void setZParam(boolean zParam) {
        this.setCopyHeaderFields(zParam);
    }

    public boolean isCopyHeaderFields() {
        return this.copyHeaderFields;
    }

    public void setCopyHeaderFields(boolean copyHeaderFields) {
        this.copyHeaderFields = copyHeaderFields;
    }

    public boolean isCheckDomainKey() {
        return this.checkDomainKey;
    }

    public void setCheckDomainKey(boolean checkDomainKey) {
        this.checkDomainKey = checkDomainKey;
    }

    protected String sign(DkimMessage message) throws MessagingException {
        if (this.checkDomainKey) {
            this.checkDomainKey();
        }
        LinkedHashMap<String, String> signatureData = new LinkedHashMap<String, String>();
        signatureData.put("v", "1");
        signatureData.put("a", this.signingAlgorithm.getDkimNotation());
        signatureData.put("q", "dns/txt");
        signatureData.put("c", this.getHeaderCanonicalization().getType() + "/" + this.getBodyCanonicalization().getType());
        signatureData.put("t", Long.toString(this.getSentDate(message).getTime() / 1000L));
        signatureData.put("s", this.selector);
        signatureData.put("d", this.signingDomain);
        if (null != this.identity) {
            signatureData.put("i", DkimSigner.quotedPrintable(this.identity));
        }
        StringBuilder headerNames = new StringBuilder();
        StringBuilder headerValues = new StringBuilder();
        StringBuilder headerFieldCopy = new StringBuilder();
        Set<String> mandatoryHeaders = this.compileMandatoryHeaders();
        for (Header header : this.compileHeadersToSign(message)) {
            String headerName = header.getName();
            String headerValue = header.getValue();
            headerNames.append(headerName).append(":");
            headerValues.append(this.headerCanonicalization.canonicalizeHeader(headerName, headerValue));
            headerValues.append("\r\n");
            mandatoryHeaders.remove(headerName);
            if (!this.copyHeaderFields) continue;
            headerFieldCopy.append(headerName);
            headerFieldCopy.append(":");
            headerFieldCopy.append(DkimSigner.quotedPrintable(headerValue.trim()).replace("|", "=7C"));
            headerFieldCopy.append("|");
        }
        if (!mandatoryHeaders.isEmpty()) {
            throw new DkimSigningException("Could not find mandatory headers: " + DkimSigner.join(mandatoryHeaders, ", "));
        }
        signatureData.put("h", headerNames.substring(0, headerNames.length() - 1));
        if (this.copyHeaderFields) {
            signatureData.put("z", headerFieldCopy.substring(0, headerFieldCopy.length() - 1));
        }
        String canonicalBody = this.canonicalizeBody(message);
        if (this.lengthParam) {
            signatureData.put("l", Integer.toString(canonicalBody.length()));
        }
        signatureData.put("bh", DkimSigner.base64Encode(this.messageDigest.digest(canonicalBody.getBytes())));
        String serializedSignature = this.serializeSignature(signatureData);
        headerValues.append(this.headerCanonicalization.canonicalizeHeader(DKIM_SIGNATUR_HEADER, serializedSignature));
        byte[] signature = this.createSignature(headerValues.toString().getBytes());
        return "DKIM-Signature: " + serializedSignature + DkimSigner.fold(DkimSigner.base64Encode(signature), 3);
    }

    private void checkDomainKey() throws DkimSigningException {
        try {
            DomainKeyUtil.getDomainKey(this.signingDomain, this.selector).check(this.identity, this.privateKey);
        }
        catch (DkimException e) {
            throw new DkimSigningException("Failed to obtain the domain key for " + this.signingDomain + "." + this.selector, e);
        }
    }

    private Date getSentDate(DkimMessage message) throws MessagingException {
        Date sentDate = message.getSentDate();
        if (null == sentDate) {
            sentDate = new Date();
        }
        return sentDate;
    }

    private Set<String> compileMandatoryHeaders() {
        TreeSet<String> mandatoryHeaders = new TreeSet<String>(String.CASE_INSENSITIVE_ORDER);
        mandatoryHeaders.addAll(MANDATORY_HEADERS_TO_SIGN);
        return mandatoryHeaders;
    }

    private List<Header> compileHeadersToSign(DkimMessage message) throws DkimSigningException {
        LinkedList<Header> reverseOrderHeaderLines = new LinkedList<Header>();
        for (Header header : this.getMessageHeaders(message)) {
            if (!this.headersToSign.contains(header.getName())) continue;
            reverseOrderHeaderLines.add(0, header);
        }
        return reverseOrderHeaderLines;
    }

    private Iterable<Header> getMessageHeaders(DkimMessage message) throws DkimSigningException {
        try {
            return this.headerIterable(message.getAllHeaders());
        }
        catch (MessagingException e) {
            throw new DkimSigningException("Could not retrieve the header fields for signing", (Exception)((Object)e));
        }
    }

    private Iterable<Header> headerIterable(final Enumeration<Header> headers) throws MessagingException {
        return new Iterable<Header>(){

            @Override
            public Iterator<Header> iterator() {
                return DkimSigner.this.headerIterator(headers);
            }
        };
    }

    private Iterator<Header> headerIterator(final Enumeration<Header> headers) {
        return new Iterator<Header>(){

            @Override
            public boolean hasNext() {
                return headers.hasMoreElements();
            }

            @Override
            public Header next() {
                return (Header)headers.nextElement();
            }
        };
    }

    private String canonicalizeBody(DkimMessage message) throws DkimSigningException {
        try {
            byte[] bodyBytes = message.getEncodedBody().getBytes();
            ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            new BufferedDataFetcher().copy((InputStream)new ByteArrayInputStream(bodyBytes), (OutputStream)new CRLFOutputStream((OutputStream)buffer));
            return this.bodyCanonicalization.canonicalizeBody(buffer.toString());
        }
        catch (DataFetchException e) {
            throw new DkimSigningException("Failed to canonicalize the line terminators of the message body", (Exception)((Object)e));
        }
    }

    private String serializeSignature(Map<String, String> signatureData) {
        int position = 0;
        StringBuilder builder = new StringBuilder();
        for (Map.Entry<String, String> entry : signatureData.entrySet()) {
            StringBuilder entryBuilder = new StringBuilder();
            entryBuilder.append(entry.getKey()).append("=").append(entry.getValue()).append(";");
            if (position + entryBuilder.length() + 1 > 67) {
                position = entryBuilder.length();
                builder.append("\r\n\t").append((CharSequence)entryBuilder);
                continue;
            }
            builder.append(" ").append((CharSequence)entryBuilder);
            position += 1 + entryBuilder.length();
        }
        builder.append("\r\n\tb=");
        return builder.toString().trim();
    }

    private byte[] createSignature(byte[] bytes) throws DkimSigningException {
        try {
            this.signature.update(bytes);
            return this.signature.sign();
        }
        catch (SignatureException e) {
            throw new DkimSigningException("Faild to create signature", e);
        }
    }

    private static String fold(String string, int offset) {
        int i = 0;
        StringBuilder builder = new StringBuilder();
        while (true) {
            if (offset > 0 && string.substring(i).length() > 67 - offset) {
                builder.append(string.substring(i, i + 67 - offset));
                i += 67 - offset;
                offset = 0;
                continue;
            }
            if (string.substring(i).length() <= 67) break;
            builder.append("\r\n\t").append(string.substring(i, i + 67));
            i += 67;
        }
        builder.append("\r\n\t").append(string.substring(i));
        return builder.toString();
    }

    private static String join(Collection<String> values, String separator) {
        StringBuilder builder = new StringBuilder();
        for (String value : values) {
            builder.append(value);
            builder.append(separator);
        }
        return builder.substring(0, builder.length() - separator.length());
    }

    private static String quotedPrintable(String s) {
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            QPEncoderStream encodeStream = new QPEncoderStream((OutputStream)out);
            encodeStream.write(s.getBytes());
            encodeStream.close();
            String encoded = out.toString();
            encoded = encoded.replaceAll(";", "=3B");
            encoded = encoded.replaceAll(" ", "=20");
            return encoded;
        }
        catch (IOException e) {
            return null;
        }
    }

    private static String base64Encode(byte[] bytes) {
        String encoded = Base64.getEncoder().encodeToString(bytes);
        encoded = encoded.replace("\n", "");
        encoded = encoded.replace("\r", "");
        return encoded;
    }

    static {
        MANDATORY_HEADERS_TO_SIGN.add("From");
        DEFAULT_HEADERS_TO_SIGN.addAll(MANDATORY_HEADERS_TO_SIGN);
        DEFAULT_HEADERS_TO_SIGN.add("To");
        DEFAULT_HEADERS_TO_SIGN.add("Subject");
        DEFAULT_HEADERS_TO_SIGN.add("Content-Description");
        DEFAULT_HEADERS_TO_SIGN.add("Content-ID");
        DEFAULT_HEADERS_TO_SIGN.add("Content-Type");
        DEFAULT_HEADERS_TO_SIGN.add("Content-Transfer-Encoding");
        DEFAULT_HEADERS_TO_SIGN.add("Cc");
        DEFAULT_HEADERS_TO_SIGN.add("Date");
        DEFAULT_HEADERS_TO_SIGN.add("In-Reply-To");
        DEFAULT_HEADERS_TO_SIGN.add("List-Subscribe");
        DEFAULT_HEADERS_TO_SIGN.add("List-Post");
        DEFAULT_HEADERS_TO_SIGN.add("List-Owner");
        DEFAULT_HEADERS_TO_SIGN.add("List-Id");
        DEFAULT_HEADERS_TO_SIGN.add("List-Archive");
        DEFAULT_HEADERS_TO_SIGN.add("List-Help");
        DEFAULT_HEADERS_TO_SIGN.add("List-Unsubscribe");
        DEFAULT_HEADERS_TO_SIGN.add("MIME-Version");
        DEFAULT_HEADERS_TO_SIGN.add("Message-ID");
        DEFAULT_HEADERS_TO_SIGN.add("Resent-Sender");
        DEFAULT_HEADERS_TO_SIGN.add("Resent-Cc");
        DEFAULT_HEADERS_TO_SIGN.add("Resent-Date");
        DEFAULT_HEADERS_TO_SIGN.add("Resent-To");
        DEFAULT_HEADERS_TO_SIGN.add("Reply-To");
        DEFAULT_HEADERS_TO_SIGN.add("References");
        DEFAULT_HEADERS_TO_SIGN.add("Resent-Message-ID");
        DEFAULT_HEADERS_TO_SIGN.add("Resent-From");
        DEFAULT_HEADERS_TO_SIGN.add("Sender");
    }
}

