/**
The MIT License (MIT) * Copyright (c) 2019 铭飞科技

 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:

 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.

 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package net.mingsoft.clean.biz.impl;

import cn.hutool.core.util.ObjectUtil;
import net.mingsoft.base.biz.impl.BaseBizImpl;
import net.mingsoft.base.dao.IBaseDao;
import net.mingsoft.basic.util.BasicUtil;
import net.mingsoft.clean.biz.ITableBiz;
import net.mingsoft.clean.dao.ITableDao;
import net.mingsoft.clean.entity.TableEntity;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.web.servlet.context.AnnotationConfigServletWebServerApplicationContext;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static java.util.regex.Pattern.compile;

/**
 * 文件清理管理持久化层
 * @author 铭飞开源团队
 * 创建日期：2019-11-21 17:08:10<br/>
 * 历史修订：<br/>
 */
 @Service("cleanTableBizImpl")
public class TableBizImpl extends BaseBizImpl implements ITableBiz {

    private static final Logger log = LoggerFactory.getLogger(TableBizImpl.class);

    @Autowired
    private AnnotationConfigServletWebServerApplicationContext applicationContext;
	@Autowired
	private ITableDao tableDao;
    /**
     * 上传路径
     */
    @Value("${ms.upload.path}")
    private String uploadFloderPath;
    /**
     * 上传路径映射
     */
    @Value("${ms.upload.mapping}")
    private String uploadMapping;
    /**
     * 上传路径映射
     */
    @Value("${ms.clean.pattern:/(upload/.*?(png|gif|bmp|pdf|mp4|avi|jpg|gif))}")
    private String cleanFilePattern;

    /**
     * 寻找文件正则
     */
    private static Pattern filePattern;

    @PostConstruct
    public void init()
    {
       // filePattern= compile("/(upload/.*?(png|gif|bmp|pdf|mp4|avi|jpg|gif))");
        filePattern= compile(cleanFilePattern);
    }

    @Override
	protected IBaseDao getDao() {
		// TODO Auto-generated method stub
		return tableDao;
	}

    @Override
    public void clearFile() {
        List<TableEntity> list = queryAll();
        List<File> fileList=new ArrayList<File>();
        File relFile = new File(uploadFloderPath);

        if (relFile.isAbsolute()) {
            findFiles(relFile,fileList);
        }else {
            findFiles(new File(getRealPath(uploadFloderPath)),fileList);
            findFiles(new File(getRealPath("ueditor")),fileList);
        }
        long startTime=System.currentTimeMillis();
        List<String> exitFiles = new ArrayList<>();
        //获取文件上传路径
        String uploadFile=relFile.isAbsolute()?relFile.toString():getRealPath("");
        for (TableEntity tableEntity : list) {
            //自定义表
            if(StringUtils.isNotBlank(tableEntity.getCleanField())){
                String[] fields = tableEntity.getCleanField().split(",");
                try {
                    exitFiles.addAll(tableTraList(tableEntity.getCleanTable(),fields ,uploadFile));
                }catch (Exception e){
                    log.error("清理错误:",e);
                }
            }
        }
        int deleFile = deleFile(fileList, exitFiles);
        long endTime=System.currentTimeMillis();
        log.info("当前程序耗时 {}ms,找到{}个文件,需验证{}个文件,删除{}个文件",
                (endTime-startTime),
                list.size(),
                exitFiles.size(),
                deleFile);
    }
    private int deleFile(List<File> source,List<String> exitFile){
        int i= 0;
        for (File file : source) {
            //判断文件是否存在数据库里
            if(!exitFile.contains(file.getPath())){
                i++;
                file.delete();
            }
        }
        return i;
    }
    /**
     * 通过表名找到表中所有文件
     * @param table 表名
     * @param fields 字段数组 字段类型 json为element上传组件的json，需要有path字段,string为富文本内容，path单纯路径
     * @param rootPath
     * @return
     */
    public  List<String> tableTraList(String table,String[] fields,String rootPath) {
        List<String> paths=new ArrayList<>();
        List<Map<String,String>> list = (List<Map<String,String>>) excuteSql("SELECT * FROM " + table);
        for (Map<String,String> map : list) {
            for (String key : fields) {
                if(ObjectUtil.isNotNull(map.get(key))){
                    Matcher matcher = filePattern.matcher(map.get(key));
                    while(matcher.find()) {
                        String fileName=(matcher.group(1));
                        File file = new File(rootPath, formatFileName(fileName));
                        if(file.exists()&&StringUtils.isNotBlank(fileName)){
                            paths.add(file.getPath());
                        }
                    }
                }
            }
        }
        return paths;
    }

    public String formatFileName(String url){
        if(new File(uploadFloderPath).isAbsolute()){
            try {
                return url.substring(uploadMapping.replace("/**", "").length());
            }catch (Exception e){
            }
        }
        return  url;
    }

    /**
     * 递归找目录下的文件
     * @param diy
     * @param list
     */
    private void findFiles(File diy, List<File> list){
        File[] files = diy.listFiles();
        if(ObjectUtil.isNotNull(diy)&&ObjectUtil.isNotNull(list)&&diy.exists()){
            for (File file : files) {
                if(file.isFile()){
                    list.add(file);
                }else {
                    //递归文件下去
                    findFiles(file,list);
                }
            }
        }
    }

    /**
     * 获取站点根目录
     * @param filePath
     * @return
     */
    public String getRealPath(String filePath) {
        String classPath = BasicUtil.getClassPath(filePath);
        if (classPath.startsWith("file")) {
            return System.getProperty("user.dir") + File.separator + filePath;
        } else {
            String path = applicationContext.getServletContext().getRealPath("/");
            if (!StringUtils.isEmpty(filePath)) {
                String last = path.charAt(path.length() - 1) + "";
                String frist = filePath.charAt(0) + "";
                if (last.equals(File.separator)) {
                    if (!frist.equals("\\") && !frist.equals("/")) {
                        path = path + filePath;
                    } else {
                        path = path + filePath.substring(1);
                    }
                } else if (!frist.equals("\\") && !frist.equals("/")) {
                    path = path + File.separator + filePath;
                } else {
                    path = path + filePath;
                }
            }
            return path;
        }
    }
}